Extension { #name : 'AbstractException' }

{ #category : 'Private' }
AbstractException class >> legacyNumberToClass: aSmallInt [
  "returns a Class"
  |arr|
  arr := LegacyErrNumMap atOrNil: aSmallInt .
  "arr is { aClass . aReasonString . legacyTemplateStringOrNil }
      or { firstClass . aReasonString . legacyTemplateStringOrNil.
                { list of split classes...} }
      per code in errordict.gs "

  arr ifNotNil:[
    ^ (arr atOrNil:1 ) ifNil:[ Error ].
  ].
  ^ Error

]

{ #category : 'Private' }
AbstractException class >> legacyNumberToClasses: aSmallInt [
  "returns nil , a Class or an Array of Classes"
  |arr|
  arr := LegacyErrNumMap atOrNil: aSmallInt .
  "arr is { aClass . aReasonString . legacyTemplateStringOrNil }
      or { firstClass . aReasonString . legacyTemplateStringOrNil.
                { list of split classes...} }
      per code in errordict.gs "

  arr ifNotNil:[
    (arr atOrNil: 4) ifNotNil:[ :arrayOfClasses | ^ arrayOfClasses] .
    ^ arr atOrNil:1  "a single class"
  ].
  ^ nil

]

{ #category : 'Private' }
AbstractException class >> legacyNumberToText: aSmallInt [
  "Returns a string describing the exception that would be created
   for the specified error number.  For use when a GciError does
   not include error text. Returns nil if the argument is not a valid error number"
  | str |
  (LegacyErrNumMap atOrNil: aSmallInt) ifNotNil:[ :arr | 
     arr _isArray ifTrue:[
       (arr atOrNil: 1) ifNotNil:[:cls | cls isBehavior ifTrue:[ str := 'a ', cls name asString ]]. 
       str ifNil:[ str := String new ].
       (arr atOrNil: 2) ifNotNil:[:x| str add: $  ; add: x asString "error symbol" ].
       (arr atOrNil: 3) ifNotNil:[:textArr |
         textArr do:[:elem |
           elem ifNotNil:[:x | x isString ifTrue:[ str add: $ ; add: x ]].
         ].
       ].
    ] ifFalse:[
      aSmallInt <= 1999 ifTrue:[
        str := 'a CompileError ', aSmallInt asString, ', ', arr asString .
      ]
    ]
  ].
  ^ str

]

{ #category : 'Private' }
AbstractException >> _description: subclassDetails [
  "for compatibility with previous releases."
  ^ self buildMessageText: subclassDetails
]

{ #category : 'Private' }
AbstractException >> _errNumPrefix [
  ^ '(error '

]

{ #category : 'Private' }
AbstractException >> _matchesExpectedLegacyNumber: aSmallInt [
  "used by topaz EXPECTERROR command, returns true if
   receiver is one of the classes produced by specified error number,
   false otherwise. "
^ [ | arr |
    (arr := LegacyErrNumMap atOrNil: aSmallInt) ifNil:[ ^ false ].
    (arr atOrNil: 4) ifNotNil:[ :inner |
      (arr atOrNil: 1) ifNotNil:[ :c | (self isKindOf: c) ifTrue:[ ^ true ]].
      inner do:[:aClass | (self isKindOf: aClass) ifTrue:[ ^ true ]]
    ].
    (arr atOrNil: 1) ifNotNil:[ :c |
       ^ (self isKindOf: c) and: [ self gsNumber == aSmallInt ]
    ].
    false
  ] onException: Error do:[:ex | ex return: false ]

]

{ #category : 'Accessing' }
AbstractException >> asString [
  ^ messageText ifNil:[ self buildMessageText  .
        messageText ifNil:[ super asString , ' errorA during buildMessageText' ] .
    ]
]

{ #category : 'Instance initialization' }
AbstractException >> buildMessageText [
  "Reimplement in subclasses , passing details to buildMessageText: "
  
  ^ self buildMessageText: nil 
]

{ #category : 'Instance initialization' }
AbstractException >> buildMessageText: subclassDetails [

  "Create the full GemStone message text if needed,
   save it in messageText instVar , and return it.
   Should be invoked by #description in a subclass. "

  messageText ifNil:[
    [ | str num |
      (str := String withAll:'a ') add: self class name ; add: ' occurred ' .
      str add: self _errNumPrefix , (num := gsNumber) asString ; add: $) .
      gsReason ifNotNil:[ :r| str add:', reason:' ; add: r asString ].
      subclassDetails ifNotNil:[
        "details instVar already included in subclassDetails if desired"
         str add:', '; add: subclassDetails .
      ] ifNil:[ | d |
        (d := gsDetails) ifNil:[
          d := (LegacyErrNumMap atOrNil: num) ifNotNil:[:m | m atOrNil: 3].
        ].
        d ifNotNil:[
          d _isArray ifTrue:[ | dStr | "details is a legacy message template"
            dStr := self _legacyDetails: d.
            dStr size > 0 ifTrue:[ str add:', ' ; add: dStr ].
          ] ifFalse:[
            str add:', ' ; add: d asString "ANSI details" .
          ]
        ].
      ].
      messageText := str .
    ] onException: Error do:[ :ex | 
      [
        messageText := self class name , ', errorC during buildMessageText' .
      ] onSynchronous: AbstractException do:[ :exB |
        exB return "ignore errors during assignment"
      ].
    ].
  ].
  ^ messageText ifNil:[ self class name , ', errorD during buildMessageText']
]

{ #category : 'Accessing' }
AbstractException >> messageText [
  "return ANSI messageText, if installed, else build the default text"
  ^ gsDetails ifNil:[ messageText 
      ifNil:[ self buildMessageText .
        messageText ifNil:[ super asString , ' errorB during buildMessageText' ] .
      ]
    ]
]
