Extension { #name : 'Float' }

{ #category : 'Private' }
Float class >> _mathPrim: anArg opcode: opcode [

"Mathematics primitive:
 opcode   function
    0     fromReadStreamInts: anArray
    3     fromStringLocaleC: aString"

<primitive: 179>
FloatingPointError _checkFpStatus .
opcode == 3 ifTrue:[ anArg _validateClass: String ].
opcode == 0 ifTrue:[ anArg _validateClass: Array  ].
^ self _primitiveFailed: #_mathPrim:opcode: args: { anArg . opcode }

]

{ #category : 'Constants' }
Float class >> denormalized [
  "The floating point representation does allow denormalized numbers"
  ^ true

]

{ #category : 'Constants' }
Float class >> e [
  "The mathematical constant  e  as representable in an 8 byte IEEE Float"
  ^  2.718281828459045

]

{ #category : 'Constants' }
Float class >> emax [
  "The largest base 2 exponent supported by Float, as defined by IEC/ISO 10967.
  Note that floating-point exponents as defined by IEC/ISO 10967 and the ANSI
  Smalltalk standards are one greater than the exponents defined by IEEE 754
  standard. This is because IEEE is modeled on a *significand*,
  1 <= significand < 2,
  and IEC/ISO/ 10967 and ANSI Smalltalk are modeled on a *fraction*,
  0 <= fraction < 1. The two models are equivalent in representable values,
  but the exponents differ by one"

  ^1024

]

{ #category : 'Constants' }
Float class >> emin [
  "The smallest base 2 exponent supported by Float, as defined by IEC/ISO 10967.
  Note that floating-point exponents as defined by IEC/ISO 10967 and the ANSI
  Smalltalk standards are one greater than the exponents defined by IEEE 754
  standard. This is because IEEE is modeled on a *significand*,
  1 <= significand < 2,
  and IEC/ISO/ 10967 and ANSI Smalltalk are modeled on a *fraction*,
  0 <= fraction < 1. The two models are equivalent in representable values,
  but the exponents differ by one"

  ^ -1021

]

{ #category : 'Constants' }
Float class >> epsilon [
  "Smallest Float such that    (1.0 + EPSLON) = 1.0) == false "
  ^ 2.2204460492503131E-16

]

{ #category : 'Constants' }
Float class >> fmax [
  "The largest Float smaller than PlusInfinity"
  ^ 1.7976931348623157E308

]

{ #category : 'Constants' }
Float class >> fmin [
  "The smallest Float greater than 0.0"
  ^ self fminDenormalized

]

{ #category : 'Constants' }
Float class >> fminDenormalized [
  "Smallest postive Float greater than 0.0"
  ^ 4.9406564584124654E-324

]

{ #category : 'Constants' }
Float class >> fminNormalized [
  "Smallest positive Float that is not a subnormal Float"
  ^ 2.2250738585072014E-308

]

{ #category : 'Instance Creation' }
Float class >> fromString: aString [

"Returns a SmallDouble or Float, constructed from aString.  The String
 must contain only Characters representing the object to be created, although
 leading and trailing blanks are permitted.

 If the string represents an exceptional float, it must contain one of the
 following strings, with leading and trailing blanks permitted:
    'PlusInfinity', 'MinusInfinity', 'PlusQuietNaN',
    'MinusQuietNaN', 'PlusSignalingNaN', or 'MinusSignalingNaN'.

 If the string does not conform to the above rules, an error may be generated,
 or a signaling NaN may be returned.

 If the string is larger than 16271 bytes, an error is generated."

<primitive: 120>
FloatingPointError _checkFpStatus .
aString _validateClass: String .
^ self _primitiveFailed: #fromString: args: { aString }

]

{ #category : 'Instance Creation' }
Float class >> fromStringLocaleC: aString [

"Same as fromString:  except that decimal point in aString must use  $.  "

^ self _mathPrim: aString opcode:3

]

{ #category : 'Storing and Loading' }
Float class >> loadFrom: passiveObj [

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into a new instance of the receiver.
 Returns the new instance."

| inst |

inst := (self fromStringLocaleC:  passiveObj upToSeparator) _asFloat .
passiveObj hasRead: inst.
^inst

]

{ #category : 'Instance Creation' }
Float class >> new [

"Returns a PlusSignalingNaN.  You can use this method to
 define a Float without specifying its value."

<primitive: 139>
^ self _primitiveFailed: #new

]

{ #category : 'Exceptions' }
Float class >> noInexactResultDo: aBlock [
 "Return value of executing aBlock, suppressing any
  FloatingPointError for inexactResult that might occur."
 | prev |
 ^ [ | fpe |
     prev := (fpe := FloatingPointError) _enableExceptions: 0 .
     fpe _enableExceptions: (prev bitAnd: 16rF"no inexactResult").
     aBlock value
   ] ensure: [ | fpe |
     fpe := FloatingPointError .
     prev ~~ 0 ifTrue:[ fpe _getExceptions: true ]. "clear any pending inexact result"
     fpe _enableExceptions: prev
   ]

]

{ #category : 'Constants' }
Float class >> pi [
  "The mathematical constant  pi  as representable in an 8 byte IEEE Float"
  ^ 3.141592653589793

]

{ #category : 'Constants' }
Float class >> precision [
  "The number of radix digits in the floating point representation.
   This is the number of bits in the mantissa of a Float plus one."
  ^ 53

]

{ #category : 'Constants' }
Float class >> radix [
  "The base of the floating point representation"
  ^ 2

]

{ #category : 'Private' }
Float >> _asFloat [

"Returns the receiver."

^ self

]

{ #category : 'Private' }
Float >> _asSmallFloat [

^ self asSmallFloat

]

{ #category : 'Converting' }
Float >> _coerce: aNumber [
	"Returns an instance of Float with the same value as 'aNumber'."

	"This method must be defined by all subclasses of Number."

	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ 
			"avoid sending asFloat to a CharacterCollection"
			aNumber _validateClass: Number ].
	^ aNumber asFloat
]

{ #category : 'Accessing' }
Float >> _exponentBits [
  "Return exponent bits of receiver as a SmallInteger"
  ^ self _mathPrim: 15

]

{ #category : 'Converting' }
Float >> _generality [

"Returns the integer that represents the ordering of the receiver in the
 generality hierarchy."

"Reimplemented from Number"

^ 90

]

{ #category : 'Accessing' }
Float >> _getKind [

"Returns an integer, 1...6, for the kind of the receiver, per Number>>kind."

<primitive: 110>

^ self _primitiveFailed: #_getKind

]

{ #category : 'Accessing' }
Float >> _isNaN [

"Returns whether the receiver is quiet NaN or signaling NaN."

^ self _getKind > 4

]

{ #category : 'Converting' }
Float >> _makeQuietNaN [

"This method returns a quiet NaN with the same sign as the input NaN.  If the
 input is not a NaN, this method returns the receiver."

   "Is the receiver a NaN?"
| kind |
kind := self _getKind .
(kind == 6 "quiet NaN" or:[ kind == 5 "signaling NaN"])
  ifFalse: [ ^self ]
  ifTrue: [ (self sign < 0)
                ifFalse: [ ^PlusQuietNaN ]
                ifTrue: [ ^MinusQuietNaN ] ]

]

{ #category : 'Accessing' }
Float >> _mantissaBits [
  "Return mantissa bits of receiver as a SmallInteger"
  ^ self _mathPrim: 14

]

{ #category : 'Private' }
Float >> _mathPrim: opcode [

"Trigonometric and other primitive functions:

 opcode   function
    0     exp
    1     cos
    2     sin
    3     tan
    4     arcTan
    5     arcSin
    6     arcCos
    7     ln
    8     hash
    9     log  base 10
   10     SmallDouble>>_asFloat
   11     _sign
   12     _asFraction
   13     asStringLocaleC
   14     _mantissaBits
   15     _exponentBits
   16     frexp
   17     arcCosh
   18     arcSinh
   19     arcTanh
   20     cosh
   21     erf
   22     erfc
   23     log2
   24     modf
   25     sinh
   26     tanh
   27     signBit
   28     exponent
   29     mantissa
   30     asStringLocaleCLegacy
   31     _smallFloatAsStringLocaleC
   "

<primitive: 136>
FloatingPointError _checkFpStatus .
^ self _primitiveFailed: #_mathPrim: args: { opcode }

]

{ #category : 'Private' }
Float >> _mathPrim: anArg opcode: opcode [

"Mathematics primitive:

 opcode   function
    0     fromReadStreamInts: anArray
    1     asStringUsingFormat: anArg
            argument must be a format descriptor Array
    2     raisedToInteger: anArg
    3     fromStringLocaleC: anArg
    4     raisedTo: anArg
    5     atan2(receiver, anArg) per math.h
    6     hypot(receiver, anArg) per math.h
    7     ldexp(receiver, anArg) per math.h
    8     asStringUsingFormatLegacy: anArg
"

<primitive: 179>
FloatingPointError _checkFpStatus .
opcode <= 3 ifTrue:[
  opcode == 2 ifTrue:[ anArg _validateClass: Integer ].
  (opcode == 1 or:[ opcode == 8 ]) ifTrue:[ | v |
    anArg _validateClass: Array .
    anArg size == 3 ifFalse:[ anArg _error: #rtErrArgOutOfRange args:{ 3 . 3 } ] .
    (anArg at:1 ) _validateClass: SmallInteger .
    (anArg at:2 ) _validateClass: SmallInteger .
    (anArg at:3 ) _validateClass: Boolean .
    ((v := anArg at:1) < -1000 or:[ v > 1000 ])
        ifTrue:[ v _error: #rtErrArgOutOfRange args:{ -1000 . 1000 } ] .
    ((v := anArg at:2) < 0 or:[ v  > 1000 ])
        ifTrue:[ v _error: #rtErrArgOutOfRange args:{ -1000 . 1000 } ] .
   ] .
  ^ self _primitiveFailed: #_mathPrim:opcode: args: { anArg . opcode }
].
opcode == 7 ifTrue:[
  anArg _validateClass: SmallInteger
] ifFalse:[
  anArg _validateClass: Float .
].
^ self _primitiveFailed: #_mathPrim:opcode: args: { anArg . opcode }

]

{ #category : 'Accessing' }
Float >> _partsAsString [
  "Return a string describing the sign, mantissa and exponent of the receiver.
   There are 1 sign bit, 52 mantissa bits, and 11 exponent bits. "
| str |
str := 'sign ' copy .
str add: (self _sign < 0 ifTrue:[ '1'] ifFalse:['0']);
   add: ', mantissa 16r' ; add: self _mantissaBits asHexString ;
   add: ', exp 16r' ; add: self _exponentBits asHexString .
^ str

]

{ #category : 'Truncation and Rounding' }
Float >> _rounded: withFpeBoolean [
"Returns the Integer nearest in value to the receiver.
 If withFpeBoolean == true may raises FloatingPointError for inexact results"

<primitive: 656>
| kind |
withFpeBoolean ifTrue:[ FloatingPointError _checkFpStatus ].
kind := self _getKind .
kind >= 5 ifTrue:[
  (kind == 6 "signaling NaN" ) ifTrue: [ Float _raiseInvalidOperationException ].
  "If a NaN, returns correct sign of self"
  ^self _makeQuietNaN
].
^ self _primitiveFailed: #_rounded:

]

{ #category : 'Private' }
Float >> _roundTo: aNumber coerce: aBoolean [
	| kind rnd |
	aBoolean class == Boolean
		ifFalse: [ aBoolean _validateClass: Boolean ].
	FloatingPointError _checkFpStatus.
	kind := self _getKind.
	kind >= 5
		ifTrue: [ 
			kind == 5
				ifTrue: [ Float _raiseInvalidOperationException ].	"signaling NaN"
			^ self _makeQuietNaN ].
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ aNumber _validateClass: Number ].
	aNumber = 0
		ifTrue: [ 
			"should have been handled in primitive"
			self
				_primitiveFailed: #'_roundTo:coerce:'
				args:
					{aNumber.
					aBoolean} ].	"Handle infinity receivers and exceptional args here"
	rnd := (self / aNumber) rounded.

	aBoolean
		ifTrue: [ 
			"roundAndCoerceTo:  semantics"
			aNumber _isInteger
				ifTrue: [ ^ rnd * aNumber ].
			aNumber _isScaledDecimal
				ifTrue: [ ^ rnd * aNumber ].
			(aNumber isKindOf: AbstractFraction)
				ifTrue: [ ^ rnd * aNumber ] ].
	^ rnd asFloat * aNumber
]

{ #category : 'Private' }
Float >> _sign [

"Returns 1 if the receiver is greater than zero, -1 if the receiver is
 less than zero, 1 if the receiver is 0.0 and -1 if the receiver is -0.0 ."

^ self _mathPrim: 11

]

{ #category : 'Private' }
Float >> _smallFloatAsStringLocaleC [

^ self _mathPrim: 31

]

{ #category : 'Truncation and Rounding' }
Float >> _truncated: withFpeBoolean [

"Returns the integer that is closest to the receiver, on the same side
 of the receiver as zero is located.

 If the receiver is an exceptional float (NaN or Infinity) , returns
 the receiver.
 If withFpeBoolean == true may raises FloatingPointError for inexact results.
"

<primitive: 109>
withFpeBoolean ifTrue:[ FloatingPointError _checkFpStatus ].
^ self _primitiveFailed: #truncated

]

{ #category : 'Truncation and Rounding' }
Float >> _truncateTo: aNumber coerce: aBoolean [
	<primitive: 660>
	| kind argFlt q |
	aBoolean class == Boolean
		ifFalse: [ aBoolean _validateClass: Boolean ].
	FloatingPointError _checkFpStatus.
	kind := self _getKind.
	kind >= 5
		ifTrue: [ 
			kind == 6
				ifTrue: [ Float _raiseInvalidOperationException ].	"signaling NaN"
			^ self _makeQuietNaN ].
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ aNumber _validateClass: Number ].
	aNumber = 0
		ifTrue: [ 
			"should have been handled in primitive"
			self
				_primitiveFailed: #'_truncateTo:coerce:'
				args:
					{aNumber.
					aBoolean} ].	"Handle infinity receivers and exceptional args here"
	q := self quo: (argFlt := aNumber asFloat).
	aBoolean
		ifTrue: [ 
			"truncateAndCoerceTo: semantics"
			aNumber _isInteger
				ifTrue: [ ^ q * aNumber ].
			aNumber _isScaledDecimal
				ifTrue: [ ^ q * aNumber ].
			(aNumber isKindOf: AbstractFraction)
				ifTrue: [ ^ q * aNumber ] ].
	^ q * argFlt
]

{ #category : 'Arithmetic' }
Float >> - aNumber [

"Returns the difference between the receiver and aNumber."

<primitive: 107>
FloatingPointError _checkFpStatus .
^ self _retry: #- coercing: aNumber

]

{ #category : 'Arithmetic' }
Float >> * aNumber [

"Multiply the receiver by aNumber and returns the result."

<primitive: 102>
FloatingPointError _checkFpStatus .
^ self _retry: #* coercing: aNumber

]

{ #category : 'Arithmetic' }
Float >> / aNumber [

"Divide the receiver by aNumber and returns the result."

<primitive: 108>
FloatingPointError _checkFpStatus .
^ self _retry: #/ coercing: aNumber

]

{ #category : 'Arithmetic' }
Float >> + aNumber [

"Returns the sum of the receiver and aNumber."

<primitive: 106>
FloatingPointError _checkFpStatus .
^ self _retry: #+ coercing: aNumber

]

{ #category : 'Comparing' }
Float >> < aNumber [

"Returns true if the receiver is less than aNumber; returns false otherwise."

	<primitive: 118>

	| sk ak |
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ ArgumentTypeError signal: 'Expected a Number' ].
	sk := self _getKind.
	sk > 4
		ifTrue: [ ^ false ].
	sk == 3
		ifTrue: [ ^ self _compareInfinityFor: #'<' with: aNumber ].
	ak := aNumber _getKind.
	ak > 4
		ifTrue: [ ^ false ].
	ak == 3
		ifTrue: [ ^ aNumber sign == 1 ].	
	"If we get here, self and arg are both finite."
	^ (AbstractFraction _coerce: self) < aNumber
]

{ #category : 'Comparing' }
Float >> <= aNumber [
	"Returns true if the receiver is less than aNumber; returns false otherwise."

	<primitive: 121>
	| sk ak |
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ ArgumentTypeError signal: 'Expected a Number' ].
	sk := self _getKind.
	sk > 4
		ifTrue: [ ^ false ].  "NaN"
	sk == 3
		ifTrue: [ ^ self _compareInfinityFor: #'<=' with: aNumber ].
	ak := aNumber _getKind.
	ak > 4
		ifTrue: [ ^ false ].
	ak == 3
		ifTrue: [ ^ aNumber sign == 1 ].	  "Inf"
	"If we get here, self and arg are both finite."
	^ (AbstractFraction _coerce: self) <= aNumber
]

{ #category : 'Comparing' }
Float >> = aNumber [
	"Returns true if the receiver is equal to aNumber; returns false otherwise."

	<primitive: 119>
	| sk ak |
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ false ].
	sk := self _getKind.
	sk > 4
		ifTrue: [ ^ false ].
	sk == 3
		ifTrue: [ ^ self _compareInfinityFor: #'=' with: aNumber ].
	ak := aNumber _getKind.
	ak > 4
		ifTrue: [ ^ false ].
	ak == 3
		ifTrue: [ ^ false ].	
	"If we get here, self and arg are both finite."
	^ (AbstractFraction _coerce: self) = aNumber
]

{ #category : 'Comparing' }
Float >> ~= aNumber [
	"Returns true if the receiver is not equal to aNumber; returns false
 otherwise."

	<primitive: 177>
										
	| sk ak |
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ true ].
	sk := self _getKind.
	sk > 4
		ifTrue: [ ^ true ].
	sk == 3
		ifTrue: [ ^ self _compareInfinityFor: #'~=' with: aNumber ].
	ak := aNumber _getKind.
	ak > 4
		ifTrue: [ ^ true ].
	ak == 3
		ifTrue: [ ^ true ].	
	"If we get here, self and arg are both finite."
	^ (AbstractFraction _coerce: self) ~= aNumber
]

{ #category : 'Arithmetic' }
Float >> arcCos [

"Returns the arc-cosine of the receiver in radians."

^ self _mathPrim: 6

]

{ #category : 'Arithmetic' }
Float >> arcCosh [

"Returns inverse hyperbolic cosine of the receiver using acosh() in math.h"

^ self _mathPrim: 17

]

{ #category : 'Arithmetic' }
Float >> arcSin [

"Returns the arc-sine of the receiver in radians."

^ self _mathPrim: 5

]

{ #category : 'Arithmetic' }
Float >> arcSinh [

"Returns inverse hyperbolic sine of the receiver using asinh() in math.h"

^ self _mathPrim: 18

]

{ #category : 'Arithmetic' }
Float >> arcTan [

"Returns the arc-tangent of the receiver in radians."

^ self _mathPrim: 4

]

{ #category : 'Arithmetic' }
Float >> arcTan2: anArg [
  "Returns arg tangent of receiver and anArg using atan2() in math.h .
   Similar to arcTan except sign of both receiver and anArg
   contribute to the quadrant of the result. "

  ^ self _mathPrim: anArg opcode: 5

]

{ #category : 'Arithmetic' }
Float >> arcTanh [

"Returns inverse hyperbolic tangent of the receiver using tanh() in math.h"

^ self _mathPrim: 19

]

{ #category : 'Converting' }
Float >> asCanonicalForm [
	"Answer self, or, if I am a Float with an equivalent
	SmallDouble, answer that SmallDouble."

	| res |
	self class ~~ SmallDouble
		ifTrue: [ 
			(self = 0.0 and: [ self _sign == -1 ])
				ifTrue: [ ^ -0.0 ].
			res := self + 0.0.
			res class == SmallDouble
				ifTrue: [ ^ res ] ].
	^ self
]

{ #category : 'Converting' }
Float >> asDecimalFloat [
	"Returns a DecimalFloat representing the receiver."

	| kind aString |
	kind := self _getKind.
	kind > 2
		ifTrue: [ 
			kind == 4
				ifTrue: [ 
					self _sign < 0
						ifTrue: [ ^ -0.0f0 ]
						ifFalse: [ ^ 0.0f0 ] ].
			kind == 3
				ifTrue: [ 
					self sign < 0
						ifTrue: [ ^ DecimalMinusInfinity ]
						ifFalse: [ ^ DecimalPlusInfinity ] ].
			kind == 5
				ifTrue: [ 
					self sign < 0
						ifTrue: [ ^ DecimalMinusQuietNaN ]
						ifFalse: [ ^ DecimalPlusQuietNaN ] ].
			self sign < 0
				ifTrue: [ ^ DecimalMinusSignalingNaN ]
				ifFalse: [ ^ DecimalPlusSignalingNaN ] ].

	"Print self rounded to 20 significant digits to correspond
	to the 20-digit mantissa of a DecimalFloat."
	aString := self asStringUsingFormat: #(0 19 true).
	^ DecimalFloat fromString: aString
]

{ #category : 'Converting' }
Float >> asFloat [

"Returns the receiver."

^ self

]

{ #category : 'Deprecated' }
Float >> asSmallFloat [

"The class SmallFloat is deprecated.
 Returns an instance of SmallFloat, which will be a NaN if the receiver
 exceeds the exponent range of a 4 byte IEEE float .  This conversion will
 lose precision."

<primitive: 132>
FloatingPointError _checkFpStatus .
^ self _primitiveFailed: #asSmallFloat
]

{ #category : 'Formatting' }
Float >> asString [

"Returns a String corresponding to the value of the receiver.  Where applicable,
 returns one of the following Strings: 'PlusInfinity', 'MinusInfinity',
 'PlusQuietNaN', 'MinusQuietNaN', 'PlusSignalingNaN', or 'MinusSignalingNaN'.

 The receiver is printed with the fewest significant digits needed to uniquely
 identify it. It is printed in exponential format unless the exponent would be
 between -4 and 15, in which case it is printed in decimal (nnn.nnnnn) format."

<primitive: 131>
^ self _primitiveFailed: #asString
]

{ #category : 'Formatting' }
Float >> asStringLegacy [

"Returns a String corresponding to the value of the receiver.  Where applicable,
 returns one of the following Strings: 'PlusInfinity', 'MinusInfinity',
 'PlusQuietNaN', 'MinusQuietNaN', 'PlusSignalingNaN', or 'MinusSignalingNaN'.

 The receiver is printed using the C printf format string '%.16E'  .
"

<primitive: 1017>
^ self _primitiveFailed: #asStringLegacy

]

{ #category : 'Formatting' }
Float >> asStringLocaleC [

"Result is same as for asString, except that decimal point is always
 using  $.   For use in passivation/activation "

^ self _mathPrim: 13

]

{ #category : 'Formatting' }
Float >> asStringLocaleCLegacy [

"Result is same as for asStringLegacy, except that decimal point is always
 using  $. "

^ self _mathPrim: 30

]

{ #category : 'Formatting' }
Float >> asStringUsingFormat: anArray [

"Returns a String corresponding to the receiver, using the format specified by
 anArray.  The Array contains three elements: two Integers and a Boolean.
 Generates an error if any element of the Array is missing or is of the wrong
 class.

 The first element of the Array (an Integer between -1000 and 1000)
 specifies a minimum number of Characters in the result String (that is,
 the 'width' of the string).  If this element is positive, the resulting
 String is padded with blanks to the right of the receiver.  If this element
 is negative, the blanks are added to the left of the receiver.  If the
 value of this element is not large enough to completely represent the Float,
 a longer String will be generated.

 The second element of the Array (a positive Integer less than or equal to
 1000) specifies the number of digits to display to the right of the decimal
 point.  If the value of this element exceeds the number of digits required to
 specify the Float, the result is right-padded with 0 to the required width.
 If the value of this element is insufficient to completely specify the Float,
 the value of the Float is rounded (see #rounded).

 The third element of the Array (a Boolean) indicates whether or not to display
 the magnitude using exponential notation.  (The value true indicates
 exponential notation and false indicates decimal notation.)

 For example, the number 12.3456 displayed with two different format Arrays
 would appear as follows:

 Format          Output
 #(10 5 true)    '1.23456e01'
 #(10 2 false)   '12.35     '        "

^ self _mathPrim: anArray opcode: 1

]

{ #category : 'Formatting' }
Float >> asStringUsingFormatLegacy: anArray [

"Same as asStringUsingFormat: except the exponent letter is E instead of e .'
 Examples:

 Format          Output
 #(10 5 true)    '1.23456E01'
 #(10 2 false)   '12.35     '        "

^ self _mathPrim: anArray opcode: 8

]

{ #category : 'Truncation and Rounding' }
Float >> ceiling [

"Returns the integer that is closest to the receiver, on the same side
 of the receiver as positive infinity.
 If receiver is a signaling NaN, raise a floating point exception"

<primitive: 658>
| kind |
FloatingPointError _checkFpStatus .
kind := self _getKind .
kind >= 5 ifTrue:[
  (kind == 6 "signaling NaN") ifTrue: [ Float _raiseInvalidOperationException ].
  "If a NaN, returns correct sign of self"
  ^self _makeQuietNaN
].
^ self _primitiveFailed: #ceiling

]

{ #category : 'Copying' }
Float >> copy [
  "If the receiver is a NaN or infinity, and FloatingPointError
   is configured to signal invalid operation errors, signals a FloatingPointError."
  | kind bits flag |
  kind := self _getKind .
  ( kind == 3"infinity" or:[ kind >= 5 "Nan"]) ifTrue:[
    bits := FloatingPointError _enableExceptions: -1 .
    (flag := bits bitAnd:  8) ~~ 0 ifTrue:[ "invalid operation enabled"
       ^ FloatingPointError new actual: flag ; signal
    ].
  ].
  ^ super copy

]

{ #category : 'Arithmetic' }
Float >> cos [

"Returns the cosine of the receiver which is treated as an
 angle expressed in radians."

^ self _mathPrim: 1

]

{ #category : 'Arithmetic' }
Float >> cosh [

"Returns the hyperbolic cosine of the receiver using cosh() from math.h"

^ self _mathPrim: 20

]

{ #category : 'Accessing' }
Float >> denominator [

"Returns the denominator of a Fraction representing the receiver."

   "If an infinite or quiet NaN, returns self"
| kind |
kind := self _getKind .
(kind >= 3) ifTrue:[ "exceptional"
  (kind == 3 "infinity" or:[ kind == 5 "quiet NaN"]) ifTrue: [ ^self ].

   "If a signaling NaN, raise a floating point exception & returns self"
  (kind == 6 "signaling NaN") ifTrue:[
      Float _raiseInvalidOperationException.
      ^self _makeQuietNaN
  ].
].
^ (self asFraction) denominator

]

{ #category : 'Arithmetic' }
Float >> erf [

"Returns the error function of receiver using erf() from math.h"

^ self _mathPrim: 21

]

{ #category : 'Arithmetic' }
Float >> erfc [

"Returns complementary error function of receiver using erfc() from math.h"

^ self _mathPrim: 22

]

{ #category : 'Arithmetic' }
Float >> exp [

"Returns e raised to the power of the receiver."

^ self _mathPrim: 0

]

{ #category : 'Accessing' }
Float >> exponent [
  "Returns the unbiased exponent of the receiver"

  ^ self _mathPrim: 28

]

{ #category : 'Truncation and Rounding' }
Float >> floor [

"Returns the integer that is closest to the receiver, on the same side
 of the receiver as negative infinity."

<primitive: 657>
| kind |
FloatingPointError _checkFpStatus .
kind := self _getKind .
kind >= 5 ifTrue:[
  (kind == 6 "signaling NaN" ) ifTrue: [ Float _raiseInvalidOperationException ].
  "If a NaN, returns correct sign of self"
  ^self _makeQuietNaN
].
^ self _primitiveFailed: #floor

]

{ #category : 'Arithmetic' }
Float >> frexp [

"Returns an Array { fractionalPart . powerOf2Exponent }
 for the receiver , per frexp() in math.h "

^ self _mathPrim: 16

]

{ #category : 'Comparing' }
Float >> hash [

"Returns a numerical hash code for the receiver."

"Two Numbers for which = is true, must have the same hash value."

^ self _mathPrim: 8

]

{ #category : 'Arithmetic' }
Float >> hypot: anArg [

"Returns      ((self*self) + (anArg*anArg)) sqrt
 using hypot() from math.h"

  ^ self _mathPrim: anArg opcode: 6

]

{ #category : 'Truncation and Rounding' }
Float >> integerPart [

"Returns an integer representing the receiver truncated toward zero."

^ self truncated asFloat

]

{ #category : 'Arithmetic' }
Float >> ldexp: anArg [

"Returns   self * (2 raisedToInteger: anArg)
                using ldexp() from math.h"

  ^ self _mathPrim: anArg opcode: 7

]

{ #category : 'Arithmetic' }
Float >> ln [

"Returns the natural logarithm of the receiver."

^ self _mathPrim: 7

]

{ #category : 'Arithmetic' }
Float >> log10 [

"Returns the base 10 logarithm of the receiver."

^ self _mathPrim: 9

]

{ #category : 'Arithmetic' }
Float >> log2 [

"Returns base 2 logarithm of receiver using  log2() from math.h"

^ self _mathPrim: 23

]

{ #category : 'Accessing' }
Float >> mantissa [
  "Returns the mantissa of the receiver, including the
   implied most-significant 1 bit if receiver is neither zero nor subnormal"

  ^ self _mathPrim: 29

]

{ #category : 'Arithmetic' }
Float >> modf [

"Returns an Array of 2 Floats,  { fractional_part . integral_part }
 using  modf() from math.h "

^ self _mathPrim: 24

]

{ #category : 'Arithmetic' }
Float >> modulo: aNumber [
	"Divide the receiver by aNumber and returns the remainder,
 i.e. the result of the C expression    self % aNumber .
 Returns PlusQuietNaN if aNumber == 0.0   . "

	<primitive: 800>
	FloatingPointError _checkFpStatus.
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ aNumber _validateClass: Number ].
	^ self modulo: aNumber asFloat
]

{ #category : 'Repository Conversion' }
Float >> needsFixingAfterConversion [
  ^true

]

{ #category : 'Accessing' }
Float >> numerator [

"Returns the numerator of a Fraction representing the receiver."

   "If an infinite or quiet NaN, returns self"
| kind |
kind := self _getKind .

kind >= 3 ifTrue:[ "exceptional"
  (kind == 3 "infinity" or:[ kind == 5 "quiet NaN"]) ifTrue: [ ^self ].

   "If a signaling NaN, raise a floating point exception & returns self"
  (kind == 6 "signaling NaN") ifTrue: [
      Float _raiseInvalidOperationException.
      ^self _makeQuietNaN
   ].
].
^ (self asFraction) numerator

]

{ #category : 'Copying' }
Float >> postCopy [
  ^ self immediateInvariant

]

{ #category : 'Arithmetic' }
Float >> raisedTo: aNumber [

"Returns the receiver raised to the power of the argument."

aNumber _isNumber ifFalse:[ aNumber _validateClass: Number ].
^ self _mathPrim: aNumber asFloat opcode: 4

]

{ #category : 'Arithmetic' }
Float >> raisedToInteger: aNumber [

"Returns the receiver raised to the power of the argument."

^ self _mathPrim: aNumber opcode: 2

]

{ #category : 'Truncation and Rounding' }
Float >> roundAndCoerceTo: aNumber [
"Returns the multiple of aNumber that is nearest in value to the receiver.

 If aNumber is a kind of Integer, ScaledDecimal, or AbstractFraction,
 the result will be an instance of the class of aNumber .

 If a signaling NaN, raise a floating point exception"
  ^ self _roundTo: aNumber coerce: true
]

{ #category : 'Truncation and Rounding' }
Float >> rounded [

"Returns the Integer nearest in value to the receiver.
 If a signaling NaN, raise a floating point exception"

^ self _rounded: true

]

{ #category : 'Truncation and Rounding' }
Float >> roundedNoFpe [

"Returns the Integer nearest in value to the receiver,
 ignoring any #inexactResult that might be enabled in FloatingPointError"

^ self _rounded: false

]

{ #category : 'Truncation and Rounding' }
Float >> roundTo: aNumber [
"Returns the multiple of aNumber that is nearest in value to the receiver.
 If a signaling NaN, raise a floating point exception"

  ^ self _roundTo: aNumber coerce: false
]

{ #category : 'Accessing' }
Float >> sign [

"Returns 1 if the receiver is greater than zero, -1 if the receiver is
 less than zero, and zero if the receiver is zero."

<primitive: 130>
^ self _primitiveFailed: #sign

]

{ #category : 'Accessing' }
Float >> signBit [

  "Returns  1 for a negative receiver and 0 for a positive receiver"

 ^ self _mathPrim: 27

]

{ #category : 'Arithmetic' }
Float >> sin [

"Returns the sine of the receiver which is treated as an
 angle expressed in radians."

^ self _mathPrim: 2

]

{ #category : 'Arithmetic' }
Float >> sinh [

"Returns the hyperbolic sine of the receiver using sinh() from math.h"

^ self _mathPrim: 25

]

{ #category : 'Arithmetic' }
Float >> sqrt [

"Returns the square root of the receiver."
"Reimplemented from Number"

<primitive: 104>
FloatingPointError _checkFpStatus .
^ self _primitiveFailed: #sqrt

]

{ #category : 'Arithmetic' }
Float >> tan [

"Returns the tangent of the receiver which is treated as an
 angle expressed in radians."

^ self _mathPrim: 3

]

{ #category : 'Arithmetic' }
Float >> tanh [

"Returns the hyperbolic tangent of the receiver using tanh() from math.h"

^ self _mathPrim: 26

]

{ #category : 'Truncation and Rounding' }
Float >> truncateAndCoerceTo: aNumber [

"Returns the multiple of aNumber that is closest to the receiver, on the
 same side of the receiver as zero is located.  In particular, returns
 the receiver if the receiver is a multiple of aNumber.

 If aNumber is a kind of Integer, ScaledDecimal, or AbstractFraction,
 the result will be an instance of the class of aNumber .

 If a signaling NaN, raise a floating point exception"

^ self _truncateTo: aNumber coerce: true
]

{ #category : 'Truncation and Rounding' }
Float >> truncated [
"Returns the integer that is closest to the receiver, on the same side
 of the receiver as zero is located.

 If the receiver is an exceptional float (NaN or Infinity) , returns
 the receiver."

^ self _truncated: true

]

{ #category : 'Truncation and Rounding' }
Float >> truncatedNoFpe [
"Returns the integer that is closest to the receiver, on the same side
 of the receiver as zero is located.
 ignoring any #inexactResult that might be enabled in FloatingPointError"

^ self _truncated: false

]

{ #category : 'Truncation and Rounding' }
Float >> truncateTo: aNumber [
"Returns the multiple of aNumber that is closest to the receiver, on the
 same side of the receiver as zero is located.  In particular, returns
 the receiver if the receiver is a multiple of aNumber.
 If a signaling NaN, raise a floating point exception"

^ self _truncateTo: aNumber coerce: false
]
