Extension { #name : 'DateAndTimeANSI' }

{ #category : 'private' }
DateAndTimeANSI class >> _zoneOffsetAtLocal: aDateAndTime [
  ^ (TimeZone current ifNil:[ TimeZoneInfo default]) offsetAtLocal: aDateAndTime

]

{ #category : 'private' }
DateAndTimeANSI class >> _zoneOffsetAtPosix: posixSeconds [
  ^ (TimeZone current ifNil:[ TimeZoneInfo default]) _offsetAtPosix: posixSeconds
]

{ #category : 'private' }
DateAndTimeANSI class >> _zoneOffsetAtUTC: aDateAndTime [
  ^ (TimeZone current ifNil:[ TimeZoneInfo default]) offsetAtUTC: aDateAndTime
]

{ #category : 'other' }
DateAndTimeANSI class >> clockPrecision [
	"Answer a <Duration> such that after that period of time passes,
	#now is guaranteed to give a different result.
	Ideally implementations should answer the least such duration."

	^Duration seconds: 1.

]

{ #category : 'Instance Creation' }
DateAndTimeANSI class >> fromString: aString [
	"YYYY-MM-DDTHH:MM:SS+HH:MM"

  	| offsetDivider offset secondsEnd | 
	offsetDivider := aString findString: ':' startingAt: 19.
	offset := offsetDivider > 0 
		ifTrue: [secondsEnd := offsetDivider - 4.
		    Duration
			days:    0
			hours:   (aString copyFrom: offsetDivider - 2 to: offsetDivider - 1 ) asNumber *
								  ((aString at: offsetDivider - 3) == $+ ifTrue: [1] ifFalse: [-1])
			minutes: (aString copyFrom: offsetDivider + 1 to: aString size) asNumber
			seconds: 0]
		ifFalse: [secondsEnd := aString size.
		    nil].

	^self
		year:   (aString copyFrom:  1 to:  4) asNumber
		month:  (aString copyFrom:  6 to:  7) asNumber
		day:    (aString copyFrom:  9 to: 10) asNumber
		hour:   (aString copyFrom: 12 to: 13) asNumber
		minute: (aString copyFrom: 15 to: 16) asNumber
		second: (aString copyFrom: 18 to: secondsEnd) asNumber
		offset: offset
]

{ #category : 'Migration' }
DateAndTimeANSI class >> migrateNew [

	^self basicNew.

]

{ #category : 'Instance Creation' }
DateAndTimeANSI class >> new [

"Disallowed"

self shouldNotImplement: #new

]

{ #category : 'Instance Creation' }
DateAndTimeANSI class >> now [
"  Answer a <DateAndTime> representing the current date and time in the local time specified by the
   implementation."

  | res |
  (res := super new) initializeAsNow.
  ^ res.

]

{ #category : 'private' }
DateAndTimeANSI class >> partsFrom: secondsFromBase [
	"#(year dayOfYear monthIndex dayOfMonth hour minute second)"
	| x y array year isLeapYear numDaysArr |
	array := Array new: 7.
      Float noInexactResultDo:[
	x := secondsFromBase .	"secondsFromBase is possibly a Float"
	array at: 7 put: x \\ 60 .
	x := x // 60 .	"minutes from base, an Integer"
      ].
	array at: 6 put: x \\ 60 .
	x := x // 60.	"hours from base"
	array at: 5 put: x \\ 24.
	x := x // 24.	"days from base"
	year := 2001.
	y := x // 146097.
	year := y * 400 + year.
	x := x - (y * 146097).		"days since beginning of 400-year cycle"

	y := x // 36524 min: 3.
	year := y * 100 + year.
	x := x - (y * 36524).		"days since beginning of 100-year cycle"

	y := x // 1461 min: 96.
	year := y * 4 + year.
	x := x - (y * 1461).			"days since beginning of 4-year cycle"

	y := x // 365 min: 3.
	year := y + year.
	x := x - (y * 365) + 1.		"days since beginning of year"
	array at: 1 put: year;
	      at: 2 put: x .
	x <= 31 ifTrue: [
		 array at: 3 put: 1;
			at: 4 put: x .
	         ^ array
	].
	x <= 59 ifTrue: [
		 array at: 3 put: 2;
			at: 4 put: x - 31 .
		^ array .
	].
	isLeapYear := year \\ 4 == 0 _and: [year \\ 100 ~~ 0 _or: [year \\ 400 == 0]].
	isLeapYear ifTrue: [
		x == 60 ifTrue: [
			array at: 3 put: 2;
				at: 4 put: 29 .
			^ array .
		].
		x := x - 1.
	].
	array at: 3 put: 3.
	x := x - 59.
	numDaysArr := #(31 30 31 30 31 31 30 31 30 31)  .
	1 to: numDaysArr size do:[:j | | each |
		each := numDaysArr at: j .
		x <= each ifTrue: [
			array at: 4 put: x .
			^ array .
		].
		array at: 3 put: (array at: 3) + 1.
		x := x - each.
	].
	self error: 'invalid date'.

]

{ #category : 'Instance Creation' }
DateAndTimeANSI class >> posixSeconds: aNumber offset: aDuration [
	"Unix time, or POSIX time, is a system for describing instances in time,
	defined as the number of seconds that have elapsed since 00:00:00 Coordinated
	Universal Time (UTC), 1 January 1970, not counting leap seconds."

	| res |
	(res := super new) _posixSeconds: aNumber offset: aDuration.
	 ^ res.

]

{ #category : 'Instance Creation' }
DateAndTimeANSI class >> secondsLocal: aNumber offset: aDuration [
  | res |
  (res := super new)
     _secondsLocal: aNumber offset: aDuration.
  ^ res

]

{ #category : 'other' }
DateAndTimeANSI class >> secondsSince2001 [

	self subclassResponsibility: #secondsSince2001

]

{ #category : 'Instance Creation' }
DateAndTimeANSI class >> secondsUTC: aNumber offset: aDuration [
  ^ self secondsUTC: aNumber offsetSeconds: (aDuration ifNotNil:[ aDuration asSeconds])

]

{ #category : 'Instance Creation' }
DateAndTimeANSI class >> secondsUTC: aNumber offsetSeconds: anOffset [
  | res |
  (res := super new) _secondsUTC: aNumber offsetSeconds: anOffset.
  ^ res

]

{ #category : 'Instance Creation' }
DateAndTimeANSI class >> year: year day: dayOfYear hour: hour minute: minute second: second [
"  Answer the least <DateAndTime> which is the second <second> of the minute <minute> of the hour
   <hour> of the day <dayOfYear> of the year <year> in the local time specified by the implementation. 
   The second must be a <number> greater than or equal to 0 and strictly less than 60.  floats are 
      allowed and implement subsecond times.
   The minute must be an <integer> between 0 and 59 inclusive. 
   The hour must be an <integer> between 0 and 23 inclusive. 
   The day must be an <integer> between 1 and 366 inclusive. 
   An implementation may not impose any limits on the year other than those imposed on <integer> constants.
   It is possible that the time specified does not exist in the local time specified by the implementation.
   If there is a time change such that the local time is set forward and the time specified is in the
   interregnum, then that time does not exist in the local time. If there is a time change that sets the 
   local time back there are times which are ambiguous. 
   Years are in the astronomical Gregorian calendar, in which the year 1 A.D. is year 1. The year 1 B.C. is 
   year 0, and the year 2 B.C. is year -1 in the astronomical Gregorian calendar and so on. 
     Parameters
        year <integer> unspecified
        dayOfYear <integer> unspecified
        hour <integer> unspecified
        minute <integer> unspecified
        second <number> unspecified
     Errors
        month is not between 1 and 12 inclusive.
        dayOfYear greater than the number of days in the year year of the astronomical Gregorian
        calendar.
        hour is not between 0 and 23 inclusive.
        minute is not between 0 and 59 inclusive.
        second is not greater than or equal to 0 and strictly less than 60.
        the time specified does not exist."

	^self
		year: year
		day: dayOfYear
		hour: hour
		minute: minute
		second: second
		offset: nil.

]

{ #category : 'Instance Creation' }
DateAndTimeANSI class >> year: year day: dayOfYear hour: hour minute: minute second: second offset: offset [
"  Answer the least <DateAndTime> which is the second <second> of the minute <minute> of the hour
   <hour> of the day <dayOfYear> of the year <year>, offset from UTC by <offset>.
   The second must be a <number> greater than or equal to 0 and strictly less than 60.  floats are 
      allowed and implement subsecond times.
   The minute must be an <integer> between 0 and 59 inclusive. 
   The hour must be an <integer> between 0 and 23 inclusive. 
   The day must be an <integer> between 1 and 366 inclusive. 
   An implementation may not impose any limits on the year other than those imposed on <integer> constants.
   It is possible that the time specified does not exist in the local time specified by the implementation.
   If there is a time change such that the local time is set forward and the time specified is in the
   interregnum, then that time does not exist in the local time. If there is a time change that sets the 
   local time back there are times which are ambiguous. 
   The result is the least <DateAndTime> that conforms to the given parameters.
   Years are in the astronomical Gregorian calendar, in which the year 1 A.D. is year 1. The year 1 B.C. is 
   year 0, and the year 2 B.C. is year -1 in the astronomical Gregorian calendar and so on. 
     Parameters
        year <integer> unspecified
        dayOfYear <integer> unspecified
        hour <integer> unspecified
        minute <integer> unspecified
        second <number> unspecified
        offset <Duration> unspecified
     Errors
        month is not between 1 and 12 inclusive.
        dayOfYear greater than the number of days in the year year of the astronomical Gregorian
        calendar.
        hour is not between 0 and 23 inclusive.
        minute is not between 0 and 59 inclusive.
        second is not greater than or equal to 0 and strictly less than 60."

	| isLeapYear daysInYear x days secs |
	isLeapYear := year \\ 4 == 0 and: [year \\ 100 ~~ 0 or: [year \\ 400 == 0]].
	daysInYear := isLeapYear ifTrue: [366] ifFalse: [365].
	(1 <= dayOfYear and: [dayOfYear <= daysInYear]) 
		ifFalse: [(OutOfRange new name: 'dayOfYear' min: 1 max: daysInYear actual: dayOfYear) signal].
	(0 <= second and: [second < 60]) ifFalse: [(OutOfRange new name: 'second' min: 0 max: 59 actual: second) signal].
	(0 <= minute and: [minute < 60]) ifFalse: [(OutOfRange new name: 'minute' min: 0 max: 59 actual: minute) signal].
	(0 <= hour and: [hour < 24]) ifFalse: [(OutOfRange new name: 'hour' min: 0 max: 23 actual: hour) signal].  "fix 51523"
	x := year - 1901.
	days := x * 365
		+ (x // 4)
		- (x // 100)
		+ (x + 300 "1901 - 1601" // 400)
		+ dayOfYear
		- 36526.		"set base to 2001-01-01"
	secs := days * 24 + hour * 60 + minute * 60 + second.
	 ^self
	 	secondsLocal: secs
	 	offset: offset.

]

{ #category : 'Instance Creation' }
DateAndTimeANSI class >> year: year month: month day: dayOfMonth hour: hour minute: minute second: second [
"  Answer the least <DateAndTime> which is the second <second> of the minute <minute> of the hour
   <hour> of the day <dayOfMonth> of the month <month> of the year <year> in the local time 
   specified by the implementation. 
   The second must be a <number> greater than or equal to 0 and strictly less than 60.  floats are 
      allowed and implement subsecond times.
   The minute must be an <integer> between 0 and 59 inclusive. 
   The hour must be an <integer> between 0 and 23 inclusive. 
   The day must be an <integer> between 1 and 366 inclusive. 
    An implementation may not impose any limits on the year other than those imposed on <integer> constants.
   If there is a time change such that the local time is set forward and the time specified is in the
   interregnum, then that time does not exist in the local time. If there is a time change that sets the 
   local time back there are times which are ambiguous. 
   Years are in the astronomical Gregorian calendar, in which the year 1 A.D. is year 1. The year 1 B.C. is 
   year 0, and the year 2 B.C. is year -1 in the astronomical Gregorian calendar and so on. 
     Parameters
        year <integer> 
        month <integer>
        dayOfMonth <integer> 
        hour <integer> 
        minute <integer> 
        second <number> 
     Errors
        month is not between 1 and 12 inclusive.
        dayOfYear greater than the number of days in the year year of the astronomical Gregorian
        calendar.
        hour is not between 0 and 23 inclusive.
        minute is not between 0 and 59 inclusive.
        second is not greater than or equal to 0 and strictly less than 60.
        the time specified does not exist."

	^self
		year: year
		month: month
		day: dayOfMonth
		hour: hour
		minute: minute
		second: second
		offset: nil.

]

{ #category : 'Instance Creation' }
DateAndTimeANSI class >> year: year month: month day: dayOfMonth hour: hour minute: minute second: second offset: offset [
"  Answer the least <DateAndTime> which is the second <second> of the minute <minute> of the hour
   <hour> of the day <dayOfMonth> of the month <month> of the year <year> offset from UTC by <offset>. 
   The minute must be an <integer> between 0 and 59 inclusive. 
   The hour must be an <integer> between 0 and 23 inclusive. 
   The day must be an <integer> between 1 and 366 inclusive. 
   An implementation may not impose any limits on the year other than those imposed on <integer> constants.
   It is possible that the time specified does not exist in the local time specified by the implementation.
   If there is a time change such that the local time is set forward and the time specified is in the
   interregnum, then that time does not exist in the local time. If there is a time change that sets the 
   local time back there are times which are ambiguous. 
   The result is the least <DateAndTime> that conforms to the given parameters.
     Parameters
        year <integer> 
        month <integer> 
        dayOfMonth <integer> 
        hour <integer> 
        minute <integer> 
        second <number> 
        offset <Duration> 
     Errors
        month is not between 1 and 12 inclusive.
        dayOfMonth greater than the number of days in the month month of year year of the
        astronomical Gregorian calendar.
        hour is not between 0 and 23 inclusive.
        minute is not between 0 and 59 inclusive.
        second is not greater than or equal to 0 and strictly less than 60."

	| isLeapYear dayOfYear daysPerMonth daysInMonth |
	(1 <= month and: [month <= 12]) ifFalse: [(OutOfRange new name: 'month' min: 1 max: 12 actual: month) signal].
	isLeapYear := year \\ 4 == 0 and: [year \\ 100 ~~ 0 or: [year \\ 400 == 0]].
	daysPerMonth := isLeapYear
		ifTrue: [#(31 29 31 30 31 30 31 31 30 31 30 31)]
		ifFalse: [#(31 28 31 30 31 30 31 31 30 31 30 31)].
	daysInMonth := daysPerMonth at: month.
	(1 <= dayOfMonth and: [dayOfMonth <= daysInMonth]) 
		ifFalse: [ (OutOfRange new name: 'dayOfMonth' min: 1 max: daysInMonth actual: dayOfMonth) signal].
	dayOfYear := (#(0 31 59 90 120 151 181 212 243 273 304 334) at: month) + dayOfMonth.
	(month > 2 and: [isLeapYear]) ifTrue: [dayOfYear := dayOfYear + 1].
	^self
		year: year
		day: dayOfYear
		hour: hour
		minute: minute
		second: second
		offset: offset.

]

{ #category : 'private' }
DateAndTimeANSI >> _addSeconds: argSeconds [
  ^ self asSeconds + argSeconds

]

{ #category : 'Accessing' }
DateAndTimeANSI >> _offsetSeconds [
  ^ offset

]

{ #category : 'private' }
DateAndTimeANSI >> _posixSeconds: aNumber offset: aDuration [
  self isInvariant ifTrue:[ ArgumentTypeError signal:'Attempt to modify invariant object'].
  self _seconds: aNumber - 978307200 .
  aDuration ifNil: [
    offset := self class _zoneOffsetAtUTC: self.
  ] ifNotNil: [
    offset  := aDuration asSeconds.
  ].
  self immediateInvariant .
]

{ #category : 'private' }
DateAndTimeANSI >> _seconds: aNumber [
  "All instances of kinds of DateAndTimeANSI are created with seconds being 
   a ScaledDecimal with scale 6 "

  self isInvariant ifTrue:[ ArgumentTypeError signal:'Attempt to modify invariant object'].
  aNumber _isInteger ifTrue:[
    seconds := aNumber
  ] ifFalse:[
    aNumber _isScaledDecimal ifTrue:[
      aNumber scale == 6  ifTrue:[
        seconds := aNumber
      ] ifFalse:[
        seconds := aNumber withScale: 6 
      ]
    ] ifFalse:[
      seconds := ScaledDecimal for: aNumber scale: 6 
    ]
  ]
]

{ #category : 'private' }
DateAndTimeANSI >> _secondsLocal: aNumber offset: anObject [
  self isInvariant ifTrue:[ ArgumentTypeError signal:'Attempt to modify invariant object'].
	self _seconds: aNumber .
	anObject ifNil:[
		offset := self class _zoneOffsetAtLocal: self.
	] ifNotNil: [
		offset  := anObject asSeconds.
	].
	self _seconds: aNumber - offset.
  self immediateInvariant .
]

{ #category : 'private' }
DateAndTimeANSI >> _secondsUTC: aNumber offsetSeconds: anOffset [
  self isInvariant ifTrue:[ ArgumentTypeError signal:'Attempt to modify invariant object'].
	self _seconds: aNumber.
	anOffset ifNil: [
		offset := self class _zoneOffsetAtUTC: self.
	] ifNotNil: [
		offset  := anOffset
	].
  self immediateInvariant .
]

{ #category : 'Arithmetic' }
DateAndTimeANSI >> - operand [
	"If operand is a <DateAndTime>, answer a <Duration> whose value is the period of time between
	the operand and the receiver. If operand is a <DateAndTime> prior to the receiver then the result
	is a <Duration> less than <Duration factory> #zero.
	If operand is a <Duration>, answer a new <DateAndTime> which represents the UTC time that is
	operand before the receiver and whose local time is the same as the receiver's. If operand is a
	duration less than <Duration factory> #zero then the result is a <DateAndTime> that is the
	absolute value of operand after the receiver."

	^(operand isKindOf: Duration) ifTrue: [
		self class
			secondsUTC: (self _addSeconds: operand asSeconds negated)
			offsetSeconds: self _offsetSeconds .
	] ifFalse: [
		Duration seconds: self asSeconds - operand asSeconds.
	].

]

{ #category : 'Arithmetic' }
DateAndTimeANSI >> + operand [
	"Answer a <DateAndTime> that represents the UTC time that is operand after the receiver and
	whose local time is the same as the receiver's. If operand is less than <Duration factory> #zero,
	the result is the <DateAndTime> that is that is the absolute value of operand before the receiver."

	^self class
		secondsUTC: (self _addSeconds: operand asSeconds )
		offsetSeconds: self _offsetSeconds .

]

{ #category : 'Comparing' }
DateAndTimeANSI >> < aDateAndTime [
"  Answer true if the UTC time represented by operand follows the UTC time represented by the
   receiver. Answer false otherwise.
   If the offsets of the receiver and operand are the same then their order is determined by their
   lexical order in the sequence #year, #month, #day, #hour24, #minute, #second. If their
   offsets differ then result is the same as if receiver asUTC < operand asUTC were evaluated."

	^self asSeconds < aDateAndTime asSeconds.

]

{ #category : 'Comparing' }
DateAndTimeANSI >> = aMagnitude [
"  Answer true if the comparand conforms to <DateAndTime> and if it represents the
   same UTC time as the receiver. Answer false otherwise. The local times of the receiver and
   operand are ignored."

	^(aMagnitude isKindOf: DateAndTimeANSI) _and: [self asSeconds = aMagnitude asSeconds].

]

{ #category : 'Converting' }
DateAndTimeANSI >> asDays [

	| x |
	x := self asSeconds .	"seconds from base"
	x := x // 60.	"minutes from base"
	x := x // 60.	"hours from base"
	x := x // 24.	"days from base"
	^x.

]

{ #category : 'private' }
DateAndTimeANSI >> asFloatParts [
	"#(year dayOfYear monthIndex dayOfMonth hour minute second)"
	^ self class partsFrom: (ScaledDecimal for: self asSeconds + self _offsetSeconds scale: 6) 	
           "seconds from base"
]

{ #category : 'private' }
DateAndTimeANSI >> asIntegerParts [
	"#(year dayOfYear monthIndex dayOfMonth hour minute second)"

	^ self class partsFrom: (self asSeconds + self _offsetSeconds ) roundedNoFpe "seconds from base"

]

{ #category : 'Converting' }
DateAndTimeANSI >> asLocal [
"  Answer a <DateAndTime> that represents the same UTC time as the receiver but in the local time
  specified by the implementation."

	^self class
		secondsUTC: self asSeconds
		offsetSeconds: nil.

]

{ #category : 'private' }
DateAndTimeANSI >> asParts [
	"#(year dayOfYear monthIndex dayOfMonth hour minute second)"

	^self asFloatParts.

]

{ #category : 'Converting' }
DateAndTimeANSI >> asPosixSeconds [
	"The number of seconds that have elapsed since 00:00:00 Coordinated
	Universal Time (UTC), 1 January 1970, not counting leap seconds. "

	^ self asSeconds + 978307200.

]

{ #category : 'Converting' }
DateAndTimeANSI >> asSeconds [
	"The number of seconds that have elapsed since 00:00:00 Coordinated
	Universal Time (UTC), 1 January 2001, not counting leap seconds."

	^ seconds
]

{ #category : 'Formatting' }
DateAndTimeANSI >> asString [
^ self printStringWithRoundedSeconds

]

{ #category : 'Converting' }
DateAndTimeANSI >> asUTC [
"  Answer a <DateAndTime> that represents the same absolute time as the receiver but in the local
   time UTC.
   The UTC local time must use the Gregorian calendar. <DateAndTimes> representing UTC times 
   prior to the	adoption of the Gregorian calendar must use the retrospective astronomical 
   Gregorian calendar. It is an invariant that
	<DateAndTime> asUTC offset = Duration zero."

	^self class
		secondsLocal: self asSeconds
		offset: Duration zero.


]

{ #category : 'Converting' }
DateAndTimeANSI >> beRounded [
  seconds := seconds roundedNoFpe
]

{ #category : 'initialize-release' }
DateAndTimeANSI >> currentTimeZone [

	^ TimeZone default.

]

{ #category : 'Accessing' }
DateAndTimeANSI >> dayOfMonth [
"  Answer an <integer> between 1 and 31 inclusive representing the number of the day in the month,
   in the local time of the receiver."

	^self asParts at: 4.

]

{ #category : 'Accessing' }
DateAndTimeANSI >> dayOfWeek [
"  Answer an <integer> between 1 and 7 inclusive representing the number of the day in the week, in
   the local time of the receiver. Sunday is 1, Monday is 2, and so on."

	^ ((self asSeconds + self _offsetSeconds) // 60) // 60 // 24 + 1 \\ 7 + 1  "1 January 2001 was a Monday"

]

{ #category : 'Accessing' }
DateAndTimeANSI >> dayOfWeekAbbreviation [
"  Answer an <readableString> which is the abbreviation of the name, in the local time of the
   receiver, of the day of the week which includes the receiver."

	^self dayOfWeekName copyFrom: 1 to: 3.

]

{ #category : 'Accessing' }
DateAndTimeANSI >> dayOfWeekName [
"  Answer an <readableString> which is the name, in the local time of the receiver, of the day of the
   week which includes the receiver."

	^#('Sunday' 'Monday' 'Tuesday' 'Wednesday' 'Thursday' 'Friday' 'Saturday') at: self dayOfWeek.

]

{ #category : 'Accessing' }
DateAndTimeANSI >> dayOfYear [
"  Answer an <integer> between 1 and 366 inclusive representing the number of the day in the year,
   in the local time of the receiver, which includes the receiver."

	^self asParts at: 2.

]

{ #category : 'Comparing' }
DateAndTimeANSI >> hash [

	^ seconds hash.

]

{ #category : 'Accessing' }
DateAndTimeANSI >> hour [
"  Answer an <integer> between 0 and 23 inclusive representing the number of the hour in the day, in
   the local time of the receiver, which includes the receiver. It is implementation defined whether a
   given local time uses the 12-hour clock or the 24-hour clock, except that the UTC local time must
   use the 24-hour clock."

	^ ((self asSeconds + self _offsetSeconds) // 60) // 60 \\ 24.

]

{ #category : 'Accessing' }
DateAndTimeANSI >> hour12 [
"  Answer an <integer> between 1 and 12 inclusive representing the hour in the day in the 12-hour
   clock of the local time of the receiver."

	^self hour - 1 \\ 12 + 1.

]

{ #category : 'Accessing' }
DateAndTimeANSI >> hour24 [
"  Answer an <integer> between 0 and 23 inclusive representing the hour in the day in the 24-hour
   clock of the local time of the receiver."

	^self hour.

]

{ #category : 'private' }
DateAndTimeANSI >> immediateInvariant [
 "Instances of SmallDateAndTime and DateAndTime are created as invariant,
  instances of other classes are variant, for compatibility with previous releases."
 self class == DateAndTime ifTrue:[ super immediateInvariant]
]

{ #category : 'initialize-release' }
DateAndTimeANSI >> initializeAsNow [
  self isInvariant ifTrue:[ ^ ArgumentTypeError signal:'Attempt to modify invariant object'].
	self _seconds: self class secondsSince2001.
	offset := (self currentTimeZone transitionAtUTC: self) offsetFromUTC.
  self immediateInvariant .
]

{ #category : 'Testing' }
DateAndTimeANSI >> isLeapYear [
"  Answer true if the year, which includes the receiver, in the local time of the receiver is a leap year,
   false otherwise.
   Two <DateAndTime> objects that are equal can give different results for #isLeapYear. Equality
   depends on their UTC time whereas #isLeapYear depends on their local time."

	| year |
	year := self year.
	^year \\ 4 == 0 _and: [year \\ 100 ~~ 0 _or: [year \\ 400 == 0]].

]

{ #category : 'Accessing' }
DateAndTimeANSI >> meridianAbbreviation [
"  Answer a <readableString> that is the abbreviation, in the local time of the receiver, of the name of
   the half of the day, which includes the receiver."

	^self hour < 12
		ifTrue: ['AM']
		ifFalse: ['PM'].

]

{ #category : 'Accessing' }
DateAndTimeANSI >> minute [
"  Answer an <integer> between 0 and 59 inclusive representing the minute of hour in the local time
   of the receiver."

	^ ((self asSeconds + self _offsetSeconds) // 60) \\ 60.

]

{ #category : 'Accessing' }
DateAndTimeANSI >> month [
"  Answer an <integer> between 1 and 12 inclusive representing the number of the month in the year,
   in the local time of the receiver, which includes the receiver."

	^self asParts at: 3.

]

{ #category : 'Accessing' }
DateAndTimeANSI >> monthAbbreviation [
"  Answer a <readableString> that is the abbreviation of the name of the month, in the local time of
   the receiver, which includes the receiver."

	^self monthName copyFrom: 1 to: 3.

]

{ #category : 'Accessing' }
DateAndTimeANSI >> monthName [
"  Answer a <readableString> that is the name of the month, in the local time of the receiver, which
   includes the receiver."

	^#(
		'January' 'February' 'March'
		'April' 'May' 'June'
		'July' 'August' 'September'
		'October' 'November' 'December'
	) at: self month.

]

{ #category : 'Accessing' }
DateAndTimeANSI >> offset [
"  Answer a <Duration> representing the difference between the local time of the receiver and UTC at
  the time of the receiver."

	^Duration seconds: self _offsetSeconds .

]

{ #category : 'Updating' }
DateAndTimeANSI >> offset: aDuration [
"  Answer a <DateAndTime> equivalent to the receiver but with its local time being offset from UTC
   by offset. The impact of this on any other local time property is unspecified.
   Implementations may define a limit to the range of offset, but it must be at least
     -12:00:00 to 12:00:00 inclusive.
   It is an invariant that if x is a <Duration> in range then
     (<DateAndTime> offset: x) offset = x
   Parameters
     offset <Duration> unspecified"

  ^ self class secondsUTC: self asSeconds  offsetSeconds: aDuration asSeconds .
]

{ #category : 'private' }
DateAndTimeANSI >> partsFrom: secondsFromBase [
  ^self class partsFrom: secondsFromBase
]

{ #category : 'Updating' }
DateAndTimeANSI >> posixSeconds: anInteger [
	"Returns a new instance with specified time since UTC 1 January 1970 .

  Unix time, or POSIX time, is a system for describing instances in time,
	defined as the number of seconds that have elapsed since 00:00:00 Coordinated
	Universal Time (UTC), 1 January 1970, not counting leap seconds." 

  ^ self class secondsUTC: anInteger - 978307200 offsetSeconds: self _offsetSeconds
]

{ #category : 'private' }
DateAndTimeANSI >> print: anArray on: aStream [

	self printDateFrom: anArray on: aStream.
	aStream nextPut: $T.
	self printTimeFrom: anArray on: aStream.
	self offset printOnHoursMinutes: aStream.

]

{ #category : 'private' }
DateAndTimeANSI >> printAnsiOn: aStream [

	self
		print: self asFloatParts
		on: aStream.

]

{ #category : 'private' }
DateAndTimeANSI >> printDateFrom: anArray on: aStream [
	| x |
	x := anArray at: 1.
	x < 0 ifTrue: [aStream nextPut: $-].
	(x := x abs) < 1000 ifTrue: [aStream nextPut: $0].
	x < 100 ifTrue: [aStream nextPut: $0].
	x < 10 ifTrue: [aStream nextPut: $0].
	x printOn: aStream.
	aStream nextPut: $-.
	(x := anArray at: 3) < 10 ifTrue: [aStream nextPut: $0].
	x printOn: aStream.
	aStream nextPut: $-.
	(x := anArray at: 4) < 10 ifTrue: [aStream nextPut: $0].
	x printOn: aStream.

]

{ #category : 'Formatting' }
DateAndTimeANSI >> printJsonOn: aStream [
  self printStringWithRoundedSeconds printJsonOn: aStream  "fix 50391"

]

{ #category : 'Formatting' }
DateAndTimeANSI >> printOn: aStream [

	self printAnsiOn: aStream.

]

{ #category : 'private' }
DateAndTimeANSI >> printRoundedOn: aStream [

	self
		print: self asIntegerParts
		on: aStream.

]

{ #category : 'Formatting' }
DateAndTimeANSI >> printString [
 "Return a String with the UTC time of the receiver offset from UTC by the offset of the receiver. 
   The result is formatted as
         -YYYY-MM-DDThh:mm:ss.s+ZZ:zz:z 
    where
        - is the <Character> $- if the year is less than 0, otherwise omitted
        YYYY is the year left zero filled to four places
        - is the <Character> $-
        MM is the month of the year left zero filled to two places
        - is the <Character> $-
        DD is the day of the month left zero filled to two places
        T is the <Character> $T
        hh is the hour in the 24-hour clock left zero filled to two places
        : is the <Character> $:
        mm is the minute left zero filled to two places
        : is the <Character> $:
        ss is the second left zero filled to two places
        . is the <Character> $. and is present only if the fraction of a second is non-zero
        s is the fraction of a second and is present only if non-zero
        + is the <Character> $+ if the offset is greater than or equal to Duration zero
        and the <Character> $- if it is less
        ZZ is the hours of the offset left zero filled to two places
        : is the <Character> $:,
        zz is the minutes of the offset left zero filled to two places,
        : is the <Character> $: and is present only if the seconds of the offset is non-zero,
        z is the seconds of the offset including any fractional part and is present only if nonzero."

	^super printString.

]

{ #category : 'private' }
DateAndTimeANSI >> printStringWithRoundedSeconds [

	| stream |
	stream := AppendStream on: String new .
	self printRoundedOn: stream.
	^stream contents.

]

{ #category : 'private' }
DateAndTimeANSI >> printTimeFrom: anArray on: aStream [
	| x |
	(x := anArray at: 5) < 10 ifTrue: [aStream nextPut: $0].
	x printOn: aStream.
	aStream nextPut: $:.
	(x := anArray at: 6) < 10 ifTrue: [aStream nextPut: $0].
	x printOn: aStream.
	aStream nextPut: $:.
	x := anArray at: 7.
	"See bugs 36718, 40624, 40643, and 40883, 48822"
  x _isInteger ifTrue:[
	  x < 10 ifTrue:[ aStream nextPut: $0 ].
    x printOn: aStream.
  ] ifFalse:[
	 (x _isFloat or:[ x _isScaledDecimal]) ifFalse:[ x := x asFloat "handle fraction"].
	  x asInteger < 10 ifTrue:[ aStream nextPut: $0 ].
	  aStream nextPutAll: x asStringLocaleC
  ]
]

{ #category : 'Converting' }
DateAndTimeANSI >> rounded [
  | rnd secs |
	rnd := (secs := self asSeconds) roundedNoFpe .
  rnd = secs ifTrue:[ ^ self ].
  ^ self class secondsUTC: rnd offsetSeconds: self _offsetSeconds .

]

{ #category : 'Accessing' }
DateAndTimeANSI >> second [
"  Answer a <number> greater than or equal to 0 and strictly less than 60 representing the second of
   the minute of the local time of the receiver."

	^ self asSeconds + self _offsetSeconds \\ 60.

]

{ #category : 'Accessing' }
DateAndTimeANSI >> timeZoneAbbreviation [
"  Answer a <readableString> that is the abbreviation of the name, in the local time of the receiver, of
   the time zone of the receiver.

   Not implemented (bug #43388); the offset is returned.  Most offsets cannot be easily disambiguated 
   between the thousands of distint timezones. "

	^self timeZoneName

]

{ #category : 'Accessing' }
DateAndTimeANSI >> timeZoneName [
"  Answer a <readableString> that is the name in the local time of the receiver of the time zone of the
   receiver.
 
   Not implemented (bug #43388); the offset is returned.  Most offsets cannot be easily disambiguated 
   between the thousands of distint timezones. "

	| stream |
	stream := AppendStream on: String new.
	self offset printOnHoursMinutes: stream.
	^stream contents.

]

{ #category : 'Accessing' }
DateAndTimeANSI >> year [
"  Answer an <integer> the number of the year which includes the receiver."

	^self asParts at: 1.

]
