Extension { #name : 'StringKeyValueDictionary' }

{ #category : 'Private' }
StringKeyValueDictionary >> _hashOfKey: aKey [

"computes the hash of the given key, equivalent to String>>hash"
<primitive: 993>
^ self _primitiveFailed: #_hashOfKey: args: { aKey }

]

{ #category : 'Accessing' }
StringKeyValueDictionary >> _lookup: aKey [

"Performs a search of the receiver for the key aKey, using code point comparison.
 Returns a SmallInteger,  least significant 30 bits are unsigned key offset
                          higher 30 bits of are unsigned bucket offset,
 or an Array of two SmallIntegers { keyOffset . bucketOffset }.
 The offsets are 1 based, a zero offset means not found, or no bucket.
 A result of nil means the key was not found."
<primitive: 83>
aKey _validateKindOfClass: String .
self _primitiveFailed: #_lookup: args: { aKey }.

]

{ #category : 'Removing' }
StringKeyValueDictionary >> _removeKey: aKey [

"Removes the key/value pair with key aKey from the receiver ,
 without accessing the value of the key/value pair.
 If no key/value pair is present with key aKey, generates an error.
 Does not remove empty collision buckets."
| x keyOfs buckOfs |
x := self _lookup: aKey .
x _isSmallInteger
  ifTrue:[ keyOfs := x bitAnd: 16r3FFFFFFF . buckOfs := x bitShift: -30]
  ifFalse:[ x ifNil:[ ^ self _errorKeyNotFound: aKey ].
            keyOfs := x at: 1 . buckOfs := x at: 2 ].
buckOfs ~~ 0 ifTrue:[ | collisionBkt |
  collisionBkt := self _at: keyOfs + 1 .
  collisionBkt _removePairAt: buckOfs  .
  collisionBkt numElements ~~ 0 ifTrue:[ numCollisions := numCollisions - 1 ].
] ifFalse:[
  self _at: keyOfs put: nil ; _at: keyOfs + 1 put: nil
].
numElements := numElements - 1.
^ self

]

{ #category : 'Private' }
StringKeyValueDictionary >> _reportString [
  | str |
  str := String new .
  self keysAndValuesDo:[:k :v |
   str add: k ; add: '  ' ; add: v asString ; lf .
  ].
  ^ str

]

{ #category : 'Updating' }
StringKeyValueDictionary >> _stringAt: aKey put: aValue [

"Stores the aKey/aValue pair in the hash dictionary, using code point comparison."

<primitive: 256>
self _primitiveFailed: #_stringAt:put: args: { aKey . aValue } .
self _uncontinuableError

]

{ #category : 'Accessing' }
StringKeyValueDictionary >> at: aKey ifAbsent: aBlock [

"Returns the value that corresponds to aKey.  If no such key/value pair
 exists, returns the result of evaluating the zero-argument block aBlock."
| x keyOfs buckOfs |
x := self _lookup: aKey .
x _isSmallInteger
  ifTrue:[ keyOfs := x bitAnd: 16r3FFFFFFF . buckOfs := x bitShift: -30]
  ifFalse:[ x ifNil:[ aBlock _isExecBlock ifFalse:[ aBlock _validateClass: ExecBlock ].
                     ^ aBlock value ].
            keyOfs := x at: 1 . buckOfs := x at: 2 ].
buckOfs ~~ 0 ifTrue:[ | collisionBkt |
  collisionBkt := self _at: keyOfs + 1 .
  ^ collisionBkt _at: buckOfs + 1 .
].
^ self _at: keyOfs + 1.

]

{ #category : 'Accessing' }
StringKeyValueDictionary >> at: aKey otherwise: defaultValue [

"Returns the value that corresponds to aKey.  If no such key/value pair
 exists, returns the given defaultValue ."

| x keyOfs buckOfs |
x := self _lookup: aKey .
x _isSmallInteger
  ifTrue:[ keyOfs := x bitAnd: 16r3FFFFFFF . buckOfs := x bitShift: -30]
  ifFalse:[ x ifNil:[ ^ defaultValue ].
            keyOfs := x at: 1 . buckOfs := x at: 2 ].
buckOfs ~~ 0 ifTrue:[ | collisionBkt |
  collisionBkt := self _at: keyOfs + 1 .
  ^ collisionBkt _at: buckOfs + 1 .
].
^ self _at: keyOfs + 1.

]

{ #category : 'Updating' }
StringKeyValueDictionary >> at: aKey put: aValue [

"Stores the aKey/aValue pair in the receiver.  Rebuilds the hash table if the
 addition caused the number of collisions to exceed the limit allowed.
 Returns aValue.  Uses code point comparison.

 If aKey is being added for the first time, an invariant copy of it is stored
 as the key.

 Gs64 v3.0 - behavior changed to match method comments.
 Previous versions made aKey invariant without making a copy of it, which
 could signal authorization errors or create unintended commit conflicts."

self _stringAt: aKey put: aValue .
(numCollisions > collisionLimit)
  ifTrue: [self rebuildTable: tableSize * 2].
^aValue

]

{ #category : 'Private' }
StringKeyValueDictionary >> collisionBucketClass [

"Returns the class of object to create when keys collide."

^ EqualityCollisionBucket

]

{ #category : 'Hashing' }
StringKeyValueDictionary >> compareKey: key1 with: key2 [

"No longer used by implementation of StringKeyValueDictionary"

self shouldNotImplement: #compareKey:with:

]

{ #category : 'Hashing' }
StringKeyValueDictionary >> hashFunction: aKey [

"No longer used by implementation of StringKeyValueDictionary"

self shouldNotImplement: #hashFunction:

]

{ #category : 'Accessing' }
StringKeyValueDictionary >> keyAt: aKey otherwise: defaultValue [

"Returns the key that corresponds to aKey.  If no such key/value pair
 exists, returns the given alternate value. "

| x keyOfs buckOfs |
x := self _lookup: aKey .
x _isSmallInteger
  ifTrue:[ keyOfs := x bitAnd: 16r3FFFFFFF . buckOfs := x bitShift: -30 ]
  ifFalse:[ x ifNil:[ ^ defaultValue ].
            keyOfs := x at: 1 . buckOfs := x at: 2 ].
buckOfs ~~ 0 ifTrue:[ | collisionBkt |
  collisionBkt := self _at: keyOfs + 1 .
  ^ collisionBkt _at: buckOfs
].
^ self _at: keyOfs .

]

{ #category : 'Accessing' }
StringKeyValueDictionary >> name [
"Returns the key of a key/value pair whose value is the receiver.
 If the receiver contains no such Association, returns nil."

| hashKey hashValue collisionBkt |
1 to: tableSize do: [ :tableIndex |
  hashKey := self keyAt: tableIndex.
  hashValue := self valueAt: tableIndex.
  (hashKey == nil)
    ifTrue: [ collisionBkt := hashValue.
              (collisionBkt == nil)
                 ifFalse: [ 1 to: collisionBkt numElements do: [ :i |
                            ((collisionBkt valueAt: i) == self)
                              ifTrue: [ ^collisionBkt keyAt: i ] ] ] ]
    ifFalse: [ (hashValue == self)
                  ifTrue: [ ^hashKey ] ]
  ].
^nil

]

{ #category : 'Formatting' }
StringKeyValueDictionary >> printNonRecursiveRepresentationOn: aStream recursionSet: anIdentitySet [

  self name ifNotNil:[:nam |
    aStream nextPutAll: self class name describeClassName .
    aStream nextPutAll: '( name: ';
            nextPutAll: nam printString ;
            nextPutAll: ' )' .
 ] ifNil:[
   super printNonRecursiveRepresentationOn: aStream recursionSet: anIdentitySet
 ]

]

{ #category : 'Removing' }
StringKeyValueDictionary >> removeKey: aKey ifAbsent: aBlock [

"Removes the key/value pair with key aKey from the receiver and returns
 the value.  If no key/value pair is present with key aKey, evaluates
 the zero-argument block aBlock and returns the result of that evaluation."

| x keyOfs buckOfs oldValue |
x := self _lookup: aKey .
x _isSmallInteger
  ifTrue:[ keyOfs := x bitAnd: 16r3FFFFFFF . buckOfs := x bitShift: -30]
  ifFalse:[ aKey ifNil:[ self _error: #rtErrNilKey].
            x ifNil:[ ^ aBlock value ].
            keyOfs := x at: 1 . buckOfs := x at: 2 ].

buckOfs ~~ 0 ifTrue:[ | collisionBkt |
  collisionBkt := self _at: keyOfs + 1 .
  oldValue := collisionBkt _at: buckOfs + 1 .
  collisionBkt _removePairAt: buckOfs  .
  collisionBkt numElements ~~ 0 ifTrue:[ numCollisions := numCollisions - 1 ].
] ifFalse:[ | vOfs |
  oldValue := self _at: (vOfs := keyOfs + 1) .
  self _at: keyOfs put: nil ; _at: vOfs put: nil
].
numElements := numElements - 1.
^ oldValue

]

{ #category : 'Removing' }
StringKeyValueDictionary >> removeKey: aKey otherwise: defaultValue [

  ^ self removeKey: aKey ifAbsent:[ defaultValue ]

]
