Extension { #name : 'SequenceableCollection' }

{ #category : 'Indexing Support' }
SequenceableCollection class >> _canCreateQueryOnInstances [
  "GsQuery may be created on most subclasses of Collection. Answer false if a GsQuery is not appropriate
   for the receiver."

  ^ true

]

{ #category : 'Instance Creation' }
SequenceableCollection class >> with: aValue [

"Returns an instance of the receiver containing the argument."
"Reimplemented to take advantage of the new: optimization "

| inst |
inst := self new: 1.
inst at: 1 put: aValue.
^inst

]

{ #category : 'Instance Creation' }
SequenceableCollection class >> with: aValue with: val2 [

"Returns an instance of the receiver containing the arguments."

"Reimplemented to take advantage of the new: optimization "

| inst |

inst := self new: 2.
inst at: 1 put: aValue;
     at: 2 put: val2.

^inst

]

{ #category : 'Instance Creation' }
SequenceableCollection class >> with: aValue with: val2 with: val3 [

"Returns an instance of the receiver containing the arguments."

"Reimplemented to take advantage of the new: optimization "

| inst |

inst := self new: 3.
inst at: 1 put: aValue;
     at:2 put: val2;
     at:3 put: val3.

^inst

]

{ #category : 'Instance Creation' }
SequenceableCollection class >> with: aValue with: val2 with: val3 with: val4 [

"Returns an instance of the receiver containing the arguments."

"Reimplemented to take advantage of the new: optimization "

| inst |

inst := self new: 4.
inst at: 1 put: aValue;
     at:2 put: val2;
     at:3 put: val3;
     at:4 put: val4.

^inst

]

{ #category : 'Private' }
SequenceableCollection >> _asCollectionForSorting [

""

^ self

]

{ #category : 'Private' }
SequenceableCollection >> _removeAll: aCollection ifAbsent: errorBlock [

"errorBlock must be non-nil "

  aCollection do: [:obj | | idx |
    idx := self indexOf: obj.
    idx == 0 ifTrue: [
      errorBlock value: obj .
    ].
    [ idx ~~ 0 ] whileTrue: [
      self removeFrom: idx to: idx.
      idx := self indexOf: obj startingAt: idx
    ].
  ].

]

{ #category : 'Concatenating' }
SequenceableCollection >> , aSequenceableCollection [

"Returns a new instance of the receiver's class that contains the elements of
 the receiver followed by the elements of aSequenceableCollection.

 Warning: Creating a new instance and copying the receiver take time.  If you
 can safely modify the receiver, it can be much faster to use the addAll:
 method.  See the documentation of the Concatenating category for more details."

 | res mySize argSize |
 mySize := self size .
 argSize := aSequenceableCollection size .
 res := self class new: mySize + argSize .
 mySize ~~ 0 ifTrue:[
   res replaceFrom: 1 to: mySize with: self startingAt: 1 .
 ].
 argSize ~~ 0 ifTrue:[
   res replaceFrom: 1 + mySize to: mySize + argSize with: aSequenceableCollection startingAt: 1.
 ].
 ^ res

]

{ #category : 'Comparing' }
SequenceableCollection >> = aSequenceableCollection [

"Returns true if all of the following conditions are true:

 1.  The receiver and aSequenceableCollection are of the same class.
 2.  The two collections are the same size.
 3.  The corresponding elements of the receiver and aSequenceableCollection
     are equal."

| selfSize |

(self == aSequenceableCollection)
  ifTrue: [ ^ true ].

(self class == aSequenceableCollection class)
  ifFalse: [ ^ false ].

selfSize := self size.
(selfSize == aSequenceableCollection size)
  ifFalse: [ ^ false ].

1 to: selfSize do: [ :i|
  ((self at: i) = (aSequenceableCollection at: i))
    ifFalse: [ ^ false ].
  ].

^ true.

]

{ #category : 'Enumerating' }
SequenceableCollection >> accompaniedBy: anObj do: aBlock [

"Evaluate the two argument block aBlock with anObj and each element of
  the receiver.  Return the receiver. "

1 to: self size do: [:j |
  aBlock value: anObj value: (self at: j)
]

]

{ #category : 'Adding' }
SequenceableCollection >> add: newObject [

"Makes newObject one of the receiver's elements and returns newObject. The
 new element is actually added as the last element of the receiver. This
 method is therefore equivalent to #addLast:."

^ self at: ((self size) + 1) put: newObject.

]

{ #category : 'Adding' }
SequenceableCollection >> add: newObject after: target [

"Adds newObject to the receiver immediately after the first element that is
 equal to target.  An object immediately follows another if its index is one
 greater than that of the other.  Returns newObject if the operation is
 successful.  Raises an error if the operation fails."

| index |

index := self indexOf: target.
(index == 0) ifTrue: [ self _error: #rtErrObjNotFound args: { target }].
^ self insertObject: newObject at: (index + 1).

]

{ #category : 'Adding' }
SequenceableCollection >> add: newObject before: target [

"Adds newObject to the receiver immediately before the first element that is
 equal to target.  An object immediately precedes another if its index is one
 less than that of the other.  Returns newObject if the operation is successful.
 Raises an error if the operation fails."

| index |

index := self indexOf: target.
(index == 0) ifTrue: [ self _error: #rtErrObjNotFound args: { target }].
^ self insertObject:  newObject at: index.

]

{ #category : 'Adding' }
SequenceableCollection >> addAll: aCollection afterIndex: index [

"Adds all the elements of aCollection to the receiver in the traversal order
 defined by the do: method for aCollection.  Inserts the new elements into the
 receiver immediately after the element in the receiver at position index.  If
 index is equal to zero, inserts the elements of aCollection at the beginning of
 the receiver.  Returns aCollection.

 The argument index must be a non-negative integer less than or equal to the
 receiver's size."

^ self insertAll: aCollection at: (index + 1).

]

{ #category : 'Adding' }
SequenceableCollection >> addAll: aCollection before: target [

"Adds all the elements of aCollection to the receiver immediately before the
 first element that is equal to target.  An object immediately precedes another
 if its index is one less than that of the other.  Returns aCollection if the
 operation is successful.  Raises an error if the operation fails."

| index |

index := self indexOf: target.
(index == 0) ifTrue: [ self _error: #rtErrObjNotFound args: { target }].
^ self insertAll: aCollection at: index.

]

{ #category : 'Adding' }
SequenceableCollection >> addAll: aCollection beforeIndex: index [

"Adds all the elements of aCollection to the receiver in the traversal order
 defined by the do: method for aCollection.  Inserts the new elements into the
 receiver immediately before the element in the receiver at position index.  If
 index is equal to the receiver's size plus one, inserts the elements of
 aCollection at the end of the receiver.  Returns aCollection.

 The argument index must be a positive integer less than or equal to the
 receiver's size plus one."

^ self insertAll: aCollection at: index .

]

{ #category : 'Adding' }
SequenceableCollection >> addLast: newObject [

"Adds newObject as the last element of the receiver and returns newObject."

^ self at: ((self size) + 1) put: newObject.

]

{ #category : 'Accessing' }
SequenceableCollection >> after: anObject [

"Returns the object immediately following the first object which is equal to
 anObject in the receiver."

| index |

index := self indexOf: anObject
             ifAbsent: [^ self _errorNotFound: anObject].

^ self at: (index + 1).

]

{ #category : 'Enumerating' }
SequenceableCollection >> allSatisfy: aBlock [

  1 to: self size do: [ :j |
     (aBlock value: (self at:j) ) ifFalse: [ ^ false].
  ].
  ^true.

]

{ #category : 'Enumerating' }
SequenceableCollection >> anySatisfy: aBlock [

  1 to: self size do: [ :j |
    (aBlock value: (self at:j) ) ifTrue: [ ^ true ].
  ].
  ^false.

]

{ #category : 'Comparing' }
SequenceableCollection >> at: anIndex equals: aSequenceableCollection [

"Returns true if aSequenceableCollection is contained in the receiver starting
 at index anIndex. Returns false otherwise."

| seqCollSize index |
seqCollSize := aSequenceableCollection size.

((anIndex + seqCollSize - 1) > (self size))
  ifTrue: [ ^ false ].

index := anIndex.
1 to: seqCollSize do: [ :j|
  ((self at: index) = (aSequenceableCollection at: j))
    ifFalse: [ ^ false ].
  index := index + 1.
  ].

^ true.

]

{ #category : 'Accessing' }
SequenceableCollection >> at: index ifAbsent: exceptionBlock [
	"Answer the element at the position index in the receiver. If there is no position corresponding to
	index in the receiver, then answer the result of evaluating exceptionBlock."

	(index between: 1 and: self size) ifFalse: [^exceptionBlock value].
	^self at: index.

]

{ #category : 'Updating' }
SequenceableCollection >> atAll: aCollection put: anObject [

"The argument aCollection is a collection of Integers that are used as indexes
 into the receiver.  At each element in the receiver for which its index is in
 aCollection, this method replaces the element with the argument anObject.
 Returns anObject."

aCollection accompaniedBy: self do: [ :me :index | me at: index put: anObject].
^ anObject.

]

{ #category : 'Updating' }
SequenceableCollection >> atAllPut: anObject [

"Assigns anObject to each of the receiver's elements and returns anObject."

1 to: self size do: [ :index | self at: index put: anObject].
^ anObject.

]

{ #category : 'Accessing' }
SequenceableCollection >> before: anObject [

"Returns the object immediately before the first object which is equal to
 anObject in the receiver."

| index |

index := self indexOf: anObject
             ifAbsent: [^ self _errorNotFound: anObject].

^ self at: (index - 1).

]

{ #category : 'Enumerating' }
SequenceableCollection >> collect: aBlock [

"Evaluates aBlock with each of the receiver's elements as the argument.
 Collects the resulting values into a collection of class specified by
 sending #speciesForCollect message to the receiver. Returns the new
 collection. The argument aBlock must be a one-argument block.

 The result preserves the ordering of the receiver.  That is, if element a
 comes before element b in the receiver, then element a is guaranteed to come
 before b in the result."

|result mySize|

mySize := self size.
result:= self speciesForCollect new: mySize .
1 to: mySize do: [:i| result at:i put:(aBlock value: (self at:i))] .
^result

]

{ #category : 'Copying' }
SequenceableCollection >> copyEmpty [

"Returns an empty copy of the receiver."

^self species new

]

{ #category : 'Deprecated' }
SequenceableCollection >> copyFrom: index1 count: aCount into: aColl startingAt: destIndex [

"Deprecated, non-ANSI.
 Copies elements of the receiver from (receiver at: index1)
  to (receiver at: (index2 + aCount - 1)), inclusive
 into aColl starting at destIndex, overwriting the previous contents.

 If (index2 + aCount - 1) is beyond the end of the receiver, then
 the copy will stop with the element  (receiver at: (receiver size)) .

 aColl must be a kind of SequenceableCollection .

 Returns the number of elements copied."

  | count |
  self deprecated: 'SequenceableCollection>>copyFrom:count:into:startingAt: deprecated v3.0' .
  count :=  aCount min: ( self size - index1 + 1) .
  aColl replaceFrom: destIndex to: destIndex + count - 1
          with: self startingAt: index1 .
  ^ count

]

{ #category : 'Copying' }
SequenceableCollection >> copyFrom: startIndex to: stopIndex [

"Returns a new SequenceableCollection containing the elements of the receiver
 between startIndex and stopIndex, inclusive.  The result is of the same class
 as the receiver.

 Both startIndex and stopIndex must be positive integers not larger than the
 size of the receiver, with startIndex <= stopIndex.
 If startIndex > stopIndex and both are positive, an empty collection is returned.
 "

| result resSize |
(startIndex > stopIndex) ifTrue: [
  stopIndex < 0 ifTrue:[ self _error: #rtErrBadCopyFromTo args: { stopIndex } ].
  ^ self class new
].
(startIndex < 1)
   ifTrue: [ ^ self _errorIndexOutOfRange: startIndex].

( (stopIndex > self size) or: [(stopIndex < 0)])
   ifTrue: [ ^ self _errorIndexOutOfRange: stopIndex].

result := self class new: (resSize := stopIndex - startIndex + 1).
result replaceFrom: 1 to: resSize with: self startingAt: startIndex .
^ result

]

{ #category : 'Deprecated' }
SequenceableCollection >> copyFrom: index1 to: index2 into: aSeqCollection startingAt: destIndex [

"Deprecated, non-ANSI.
 Copies the elements of the receiver between index1 and index2, inclusive,
 into aSeqCollection starting at destIndex, overwriting the previous
 contents.  If aSeqCollection is the same object as the receiver, the
 source and destination blocks may overlap. Returns the receiver."

 self deprecated: 'SequenceableCollection>>copyFrom:to:into:startingAt: deprecated v3.0.' .

 "bug42397--->"
 (index1 > index2)
  ifTrue: [  self _error: #rtErrBadCopyFromTo args: { index2 } ].

(index1 < 1)
  ifTrue: [ ^ self _errorIndexOutOfRange: index1].

(index2 > self size)
  ifTrue: [ ^ self _errorIndexOutOfRange: index2].

((destIndex < 1) or: [(destIndex > (aSeqCollection size + 1))])
   ifTrue: [ ^ aSeqCollection _errorIndexOutOfRange: destIndex].

 aSeqCollection replaceFrom: destIndex to: destIndex + (index2 - index1)
        with: self startingAt: index1 .
 ^ self

]

{ #category : 'Copying' }
SequenceableCollection >> copyReplaceAll: oldSubCollection with: newSubCollection [

"Returns a modified copy of the receiver in which all sequences of elements
 within the receiver that match the elements of oldSubCollection are replaced
 by the elements of newSubCollection."

| selfSize newSubCollSize oldSubCollSize result sourceIndex targetIndex
  subIndex |

	newSubCollection class isIndexable
		ifFalse: [^newSubCollection _errorNotIndexable].

selfSize := self size.
newSubCollSize := newSubCollection size.
oldSubCollSize := oldSubCollection size.

result := self species new.

sourceIndex := 1.
targetIndex := 1.

[ sourceIndex <= selfSize ] whileTrue: [
  subIndex := self indexOfSubCollection: oldSubCollection
              startingAt: sourceIndex
              ifAbsent: [
                result replaceFrom: targetIndex to: targetIndex + (selfSize - sourceIndex)
                       with: self startingAt: sourceIndex .
                ^ result.
              ].

  (subIndex > sourceIndex) ifTrue: [
      result replaceFrom: targetIndex to: targetIndex + ((subIndex - 1) - sourceIndex)
			with: self startingAt: sourceIndex .
      targetIndex := targetIndex + (subIndex - sourceIndex).
  ].

  (newSubCollSize ~~ 0) ifTrue: [
     result replaceFrom: targetIndex to: targetIndex + (newSubCollSize  - 1)
		with: newSubCollection startingAt: 1
  ].

  targetIndex := targetIndex + newSubCollSize.
  sourceIndex := subIndex + oldSubCollSize.
  ].

^ result.

]

{ #category : 'Copying' }
SequenceableCollection >> copyReplaceFrom: startIndex to: stopIndex with: aSequenceableCollection [
	"Returns a copy of the receiver in which all elements in the
	 receiver between indexes startIndex and stopIndex inclusive have
	 been replaced by those contained in aSequenceableCollection.

	 Rules, abbreviated from the ANSI specification:
		- start < 1 or start > receiver's size + 1 (error)
		- stop < start - 1 or stop > receiver's size (error)
		- stop = start - 1 and start <= receiver's size (insert between stop and start)
		- stop = start - 1 and start = receiver's size + 1 (append at the end of the new collection)
		- otherwise, replace from start through stop with the elements from replacementElements.
	"

	| selfSize seqCollSize result |
	aSequenceableCollection class isIndexable
		ifFalse: [^aSequenceableCollection _errorNotIndexable].

	selfSize := self size.
	seqCollSize := aSequenceableCollection size.
	result := self species new.
	(startIndex < 1
		or: [startIndex > (selfSize + 1)
		or: [stopIndex < (startIndex - 1)
		or: [stopIndex > selfSize]]])
			ifTrue: [^startIndex _error: #rtErrBadCopyFromTo args: {stopIndex}].

		"The stopIndex < startIndex case is an insertion/append operation."
	stopIndex = (startIndex - 1)
		ifTrue:
			[result
				replaceFrom: 1
					to: stopIndex
					with: self
					startingAt: 1;
				replaceFrom: stopIndex + 1
					to: stopIndex + seqCollSize
					with: aSequenceableCollection
					startingAt: 1;
				replaceFrom: stopIndex + seqCollSize + 1
					to: selfSize + seqCollSize
					with: self
					startingAt: stopIndex + 1]
		ifFalse:
			"The stopIndex >= startIndex case is the 'normal' operation."
			[result
				replaceFrom: 1
					to: startIndex - 1
					with: self
					startingAt: 1;
				replaceFrom: startIndex
					to: startIndex + seqCollSize - 1
					with: aSequenceableCollection
					startingAt: 1;
				replaceFrom: result size + 1
					to: seqCollSize + selfSize - (stopIndex - startIndex + 1)
					with: self
					startingAt: stopIndex + 1].

	^result

]

{ #category : 'Copying' }
SequenceableCollection >> copyReplaceFrom: startIndex to: stopIndex withObject: anObject [

"Returns a copy of the receiver in which all elements in the receiver between
 indexes startIndex and stopIndex inclusive have been replaced by anObject."

^self
   copyReplaceFrom: startIndex
   to: stopIndex
   with: {anObject}.

]

{ #category : 'Copying' }
SequenceableCollection >> copyReplacing: oldObject with: newObject [

"Returns a copy of the receiver in which all occurrences of objects equal to
 oldObject have been replaced by newObject."

  ^self copyReplacing: oldObject withObject: newObject

]

{ #category : 'Copying' }
SequenceableCollection >> copyReplacing: oldObject withObject: newObject [

"Returns a copy of the receiver in which all occurrences of objects equal to
 oldObject have been replaced by newObject."

| result |

result := self copy.
1 to: result size do: [ :i |
  (oldObject = (result at: i))
    ifTrue: [ result at: i put: newObject ]
  ].

^ result.

]

{ #category : 'Copying' }
SequenceableCollection >> copyWith: anObject [

"Returns a copy of the receiver with anObject appended at the end."

| newCollection |

newCollection := self copy.
newCollection addLast: anObject.

^ newCollection.

]

{ #category : 'Copying' }
SequenceableCollection >> copyWithout: anObject [

"Returns a copy of the receiver that does not contain the given object.
 Comparisons are by equality."

| copy element sz |

copy := self class new.

sz := 0.
1 to: self size do: [:i |
  element := self at: i.
  (element = anObject)
    ifFalse: [
      sz := sz + 1.
      copy at: sz put: element.
      ]
  ].

^copy

]

{ #category : 'Deprecated' }
SequenceableCollection >> deleteFrom: startIndex to: stopIndex [

self deprecated: 'SequenceableCollection>>deleteFrom:to: deprecated long before v3.0. Use removeFrom:to: instead.'.
^ self removeFrom: startIndex to: stopIndex.

]

{ #category : 'Deprecated' }
SequenceableCollection >> deleteObjectAt: anIndex [

self deprecated: 'SequenceableCollection>>deleteObjectAt: deprecated long before v3.0. Use removeAtIndex: instead.'.
^ self removeAtIndex: anIndex

]

{ #category : 'Enumerating' }
SequenceableCollection >> detect: aBlock ifNone: exceptionBlock [
  "Evaluates aBlock repeatedly, with elements of the receiver as the argument.
 Returns the first element for which aBlock evaluates to true.  If none of
 the receiver's elements evaluates to true, this evaluates the argument
 exceptionBlock and returns its value.  The argument aBlock must be a
 one-argument block, and exceptionBlock must be a zero-argument block."

  | result |
  aBlock _isExecBlock ifFalse:[
    aBlock class == SelectBlock
      ifTrue: [
      "send value: to the block to create the 4 element
        associative access Array; the argument nil is ignored to
        create the Array, but all SelectBlocks have one argument"
      result := (QueryExecuter on: self)
        _boundQueryDetect: (aBlock queryBlock value: nil).
      result == #'_incompletePathTraversal'
        ifTrue: [ ^ exceptionBlock value ].
      ^ result ]].
  1 to: self size do: [ :i | | v |
    (aBlock value: (v := self at: i))
      ifTrue: [ ^ v ] ].
  ^ exceptionBlock value

]

{ #category : 'Enumerating' }
SequenceableCollection >> doWithIndex: aBlock [

"Evaluates the two argument block aBlock using each element of the receiver as
 the first argument and the corresponding index as the second argument. Returns
 the receiver."

1 to: self size do: [ :i |
  aBlock value: (self at: i) value: i.
  ].

^ self.

]

{ #category : 'Deprecated' }
SequenceableCollection >> eq: aSequenceableCollection [

self deprecated: 'SequenceableCollection>>eq: deprecated long before v3.0.  Use = instead.'.
^ self = aSequenceableCollection.

]

{ #category : 'Searching' }
SequenceableCollection >> findFirst: aBlock [

"Returns the index of the first element in the receiver which causes the one
 argument block, aBlock, to evaluate true. Returns 0 if no element in the
 receiver causes the block to evaluate true."

1 to: self size do: [:i|
  (aBlock value: (self at: i)) ifTrue: [^i]].

^0

]

{ #category : 'Searching' }
SequenceableCollection >> findLast: aBlock [

"Returns the index of the last element in the receiver which causes the one
 argument block, aBlock, to evaluate true. Returns 0 if no element in the
 receiver causes the block to evaluate true."

self size _downTo: 1 do: [:i|
  (aBlock value: (self at: i)) ifTrue: [^i]].

^0

]

{ #category : 'Accessing' }
SequenceableCollection >> first [

"Returns the first element of the receiver.  Generates an error if the receiver
 is empty."

(self size == 0)
ifTrue: "attempt to access an empty Collection"
   [ ^ self _error: #objErrCollectionEmpty].
^ self at: 1

]

{ #category : 'Updating' }
SequenceableCollection >> first: anObject [

"Stores the given object in the first position in the receiver and returns
 anObject."

^ self at: 1 put: anObject.

]

{ #category : 'Enumerating' }
SequenceableCollection >> from: startIndex to: stopIndex do: aBlock [

"Evaluates the one argument block aBlock using each element of the receiver
 starting at startIndex and ending at stopIndex. Returns the receiver."

(startIndex < 1 or:[ stopIndex > self size])
   ifTrue: [ ^ self _errorIndexOutOfRange: startIndex ].

startIndex to: stopIndex do: [ :i|
  aBlock value: (self at: i).
  ].

^ self.

]

{ #category : 'Enumerating' }
SequenceableCollection >> from: startIndex to: stopIndex doWithIndex: aBlock [

"Evaluates the two argument block aBlock using each element of the receiver
 starting at startIndex and ending at stopIndex as the first argument and the
 corresponding index as the second argument. Returns the receiver."

(startIndex < 1 or:[ stopIndex > self size])
   ifTrue: [ ^ self _errorIndexOutOfRange: startIndex ].

startIndex to: stopIndex do: [ :i|
  aBlock value: (self at: i) value: i.
  ].

^ self.

]

{ #category : 'Enumerating' }
SequenceableCollection >> from: startIndex to: stopIndex keysAndValuesDo: aDyadicBlock [

"For each index in the range <startIndex> to <stopIndex>, the 2-argument block
 <aDyadicBlock> is evaluated with the index as the first argument and the element
 at that index as the second argument."

	^startIndex to: stopIndex do: [:index | aDyadicBlock value: index value: (self at: index)]

]

{ #category : 'Comparing' }
SequenceableCollection >> hash [

"Returns a numeric hash key for the receiver."

| mySize interval hashValue |

(mySize := self size) == 0
  ifTrue: [ ^15243 ].

"Choose an interval so that we sample at most 5 elements of the receiver"
interval := ((mySize - 1) // 4) max: 1.

hashValue := 4459.
1 to: mySize by: interval do: [ :i | | anElement |
  anElement := self at: i.
  (anElement isKindOf: SequenceableCollection)
    ifTrue: [
      hashValue := (hashValue bitShift: -1) bitXor: anElement size.
      ]
    ifFalse: [
      hashValue := (hashValue bitShift: -1) bitXor: anElement hash.
      ].
  ].

^ hashValue abs

]

{ #category : 'Comparing' }
SequenceableCollection >> hasIdenticalContents: aSequenceableCollection [

"Returns true if all of the following conditions are true:

 1.  The receiver and aSequenceableCollection are of the same class.
 2.  The two collections are the same size.
 3.  The corresponding elements of the receiver and aSequenceableCollection
     are identical."

| selfSize |

(self == aSequenceableCollection)
  ifTrue: [ ^ true ].

(self class == aSequenceableCollection class)
  ifFalse: [ ^ false ].

selfSize := self size.
(selfSize == aSequenceableCollection size)
  ifFalse: [ ^ false ].

1 to: selfSize do: [ :i|
  ((self at: i) == (aSequenceableCollection at: i))
    ifFalse: [ ^ false ].
  ].

^ true.

]

{ #category : 'Searching' }
SequenceableCollection >> identicalOccurrencesOf: anObject [

"Returns the number of the receiver's elements that are identical (==) to
 anObject."

| result |

result := 0.
1 to: self size do: [ :j |
  anObject == (self at: j) ifTrue: [ result := result + 1 ]
  ].
^result

]

{ #category : 'Searching' }
SequenceableCollection >> includes: anObject [

"Returns true if anObject is equal to one of the elements of the receiver.
 Returns false otherwise."

1 to: self size do:[:j |
  anObject = (self at:j ) ifTrue:[ ^ true ]
].
^false

]

{ #category : 'Searching' }
SequenceableCollection >> includesIdentical: anObject [

"Returns true if anObject is identical to one of the elements of the
 receiver.  Returns false otherwise."

^ (self indexOfIdentical: anObject) ~~ 0

]

{ #category : 'Searching' }
SequenceableCollection >> indexOf: anObject [

"Returns the index of the first element in the receiver that is equivalent to
 the argument. If the receiver does not have any elements equivalent to the
 argument, returns 0."

^ self indexOf: anObject startingAt: 1

]

{ #category : 'Searching' }
SequenceableCollection >> indexOf: anObject ifAbsent: anExceptionBlock [

"Returns the index of the first element in the receiver that is equivalent to
 the argument. If the receiver does not have any elements equivalent to the
 argument, returns the value of evaluating anExceptionBlock."

1 to: self size do:[:i |
  anObject = (self at: i) ifTrue:[ ^ i ]
].
^ anExceptionBlock value.

]

{ #category : 'Searching' }
SequenceableCollection >> indexOf: anObject startingAt: index [

"Returns the index of the first element in the receiver that is equivalent to
 the argument. If the receiver does not have any elements equivalent to the
 argument, returns 0."

index to: self size do: [:i |
  anObject = (self at: i) ifTrue:[ ^ i ]
].
^ 0

]

{ #category : 'Searching' }
SequenceableCollection >> indexOf: anObject startingAt: index ifAbsent: anExceptionBlock [

"Returns the index of the first element in the receiver that is equivalent to
 the argument. If the receiver does not have any elements equivalent to the
 argument, returns the value of evaluating anExceptionBlock"

index to: self size do: [ :i |
  (anObject = (self at: i)) ifTrue: [ ^i ].
  ].

^ anExceptionBlock value.

]

{ #category : 'Searching' }
SequenceableCollection >> indexOfIdentical: anObject [

"Private.  Returns the index of the first element in the receiver that is
 identical to the argument.  If the receiver does not have any elements that are
 identical to the argument, returns zero."

1 to: self size do: [:i |
  (anObject == (self at: i)) ifTrue: [^i]
  ].

^ 0.

]

{ #category : 'Searching' }
SequenceableCollection >> indexOfSubCollection: aSubColl [

"Returns the index of the first element of the receiver where that element and
 the subsequent ones are equal to those in aSubColl. The search is begun in the
 receiver starting at 1."

^ self indexOfSubCollection: aSubColl startingAt: 1.

]

{ #category : 'Searching' }
SequenceableCollection >> indexOfSubCollection: aSubColl startingAt: anIndex [

"Returns the index of the first element of the receiver where that element and
 the subsequent ones are equal to those in aSubColl. The search is begun in the
 receiver starting at anIndex. Returns 0 if no match is found."

^ self indexOfSubCollection: aSubColl startingAt: anIndex ifAbsent: [0].

]

{ #category : 'Searching' }
SequenceableCollection >> indexOfSubCollection: aSubColl startingAt: anIndex ifAbsent: anExceptionBlock [

"Returns the index of the first element of the receiver where that element and
 the subsequent ones are equal to those in aSubColl. The search is begun in the
 receiver starting at anIndex. Returns the value of evaluating
 anExceptionBlock if no match is found."

| subCollSize |

subCollSize := aSubColl size.
(aSubColl size == 0) ifTrue: [ ^ anExceptionBlock value ].

anIndex to: (self size - subCollSize + 1) do: [ :i |
  (self at: i equals: aSubColl) ifTrue: [ ^ i ].
  ].

^ anExceptionBlock value.

]

{ #category : 'Deprecated' }
SequenceableCollection >> indexOfValue: anObject [

self deprecated: 'SequenceableCollection>>indexOfValue: deprecated long before v3.0. Use indexOf: instead.'.
^ self indexOf: anObject.

]

{ #category : 'Deprecated' }
SequenceableCollection >> insert: aSequenceableCollection at: anIndex [

self deprecated: 'SequenceableCollection>>insert:at: deprecated long before v3.0. Use insertAll:at: instead.'.
aSequenceableCollection _validateClass: SequenceableCollection.
^ self insertAll: aSequenceableCollection at: anIndex.

]

{ #category : 'Adding' }
SequenceableCollection >> insertAll: aCollection at: anIndex [

"Inserts all the elements of aCollection into the receiver beginning at index
 anIndex.  Returns aCollection.

 The argument anIndex must be greater than or equal to one.  If anIndex is one
 greater than the size of the receiver, appends aCollection to the receiver.  If
 anIndex is more than one greater than the size of the receiver, generates an
 error."

<primitive: 606 >
| array argCls |
argCls := aCollection class .
(aCollection isKindOf: Interval) ifTrue:[
  self insertAll: aCollection asArray at: anIndex  .
  ^ aCollection
  ].
((argCls isSubclassOf: CharacterCollection )
 or:[ argCls isSubclassOf: ByteArray ]) ifTrue:[
   array := Array new: aCollection size .
   1 to: aCollection size do:[:j|
     array at: j put: (aCollection at: j)
     ].
   self insertAll: array at: anIndex .
   ^ aCollection
   ].
aCollection _validateClasses:
  { Array . OrderedCollection . SequenceableCollection }.
(anIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: anIndex].
(anIndex < 1 or: [ anIndex > (self size + 1)]) "out of bounds"
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].

^ self _primitiveFailed: #insertAll:at: args: { aCollection . anIndex }

]

{ #category : 'Adding' }
SequenceableCollection >> insertObject: anObject at: anIndex [

"Inserts anObject into the receiver at index anIndex and returns anObject.
 If receiver is a small (<= 2024 oops) committed object, does space-efficent
 tranlogging of the insertion.
"

<primitive: 398>
"primitive fails if receiver is byte format"
self insertAll: { anObject } at: anIndex .
^ anObject

]

{ #category : 'Enumerating' }
SequenceableCollection >> keysAndValuesDo: aDyadicBlock [

"The 2-argument block <aDyadicBlock> is evaluated with the index of each element
 of the receiver as the first argument and the element itself as the second argument.
 Evaluation is in indexed order with the first element being at index 1."

	^self from: 1 to: self size keysAndValuesDo: aDyadicBlock

]

{ #category : 'Accessing' }
SequenceableCollection >> last [

"Returns the last element of the receiver.  Generates an error if the receiver
 is empty."

(self size == 0)
ifTrue: "attempt to access an empty Collection"
   [ ^ self _error: #objErrCollectionEmpty].
^self at: self size

]

{ #category : 'Updating' }
SequenceableCollection >> last: anObject [

"Stores the given object in the last position in the receiver and returns
 anObject."

^ self at: (self size) put: anObject.

]

{ #category : 'Searching' }
SequenceableCollection >> occurrencesOf: anObject [

"Returns the number of the receiver's elements that are equal to anObject."

| result |

result := 0.
1 to: self size do: [ :j |
  anObject = (self at: j) ifTrue: [ result := result + 1 ]
  ].
^result

]

{ #category : 'Searching' }
SequenceableCollection >> occurrencesOfDistinctEqualSubCollection: aCollection [
    "Answer the number of *distinct* *non-overlapping* occurrences of
    aCollection in the receiver using equality not identity."

    | otherSize selfSize currentStart currentResult count |
    otherSize := aCollection size.
    selfSize := self size.
    count := 0.
    currentStart := 1.
    [(currentStart + otherSize - 1) <= selfSize] whileTrue:
        [currentResult := self indexOfSubCollection: aCollection startingAt: currentStart ifAbsent: [^count].
         count := count + 1.
         currentStart := currentResult + otherSize].
    ^count

]

{ #category : 'Json' }
SequenceableCollection >> printJsonOn: aStream [

  | delimiter |
  delimiter := ''.
  aStream nextPut: $[.
  1 to: self size do: [ :j |
    aStream nextPutAll: delimiter.
    (self at: j) printJsonOn: aStream.
    delimiter := ','.
  ].
  aStream nextPut: $].

]

{ #category : 'Streams' }
SequenceableCollection >> readStream [

"Returns a ReadStream on the receiver."

^ ReadStream on: self

]

{ #category : 'Enumerating' }
SequenceableCollection >> reject: aBlock [
  "Evaluates aBlock with each of the receiver's elements as the argument.
 Stores the values for which aBlock is false into a collection of the same
 class as the receiver, and returns the new collection.  The argument aBlock
 must be a one-argument block.

 The result preserves the ordering of the receiver.  That is, if element a
 comes before element b in the receiver, then element a is guaranteed to come
 before b in the result."

  | result |
  aBlock _isExecBlock ifFalse:[
   aBlock class == SelectBlock
    ifTrue: [
      "send value: to the block to create the 4 element
        associative access Array; the argument nil is ignored to
        create the Array, but all SelectBlocks have one argument"
      result := (QueryExecuter on: self)
        _boundQueryReject: (aBlock queryBlock value: nil).
      ^ result ]].
  result := self species new.
  1 to: self size do: [ :i | | element |
    element := self at: i.
    (aBlock value: element)
      ifFalse: [ result add: element ] ].
  ^ result

]

{ #category : 'Removing' }
SequenceableCollection >> remove: oldObject ifAbsent: anExceptionBlock [

"Removes from the receiver an object that is equivalent to oldObject.
 Returns oldObject.  If several elements of the receiver are equivalent to
 oldObject, only one instance is removed.  If oldObject has no equivalent
 elements in the receiver, evaluates anExceptionBlock and returns the result
 of the evaluation."


| index |

index := self indexOf: oldObject.
index == 0
  ifTrue: [^ anExceptionBlock value]
  ifFalse: [^ self removeAtIndex: index]

]

{ #category : 'Deprecated' }
SequenceableCollection >> removeAll: aCollection ifAbsent: errorBlock [

self deprecated: 'SequenceableCollection>>removeAll:ifAbsent: deprecated before 3.0. Use removeAllPresent:'.

"Removes all occurrences of the each of the elements of aCollection from the
 receiver.  Iterates over aCollection using do: [required].  Executes the given
 one-argument block if an object in aCollection is not in the receiver.  The
 block argument may be nil, in which case the object-not-found error will be
 ignored."

self == aCollection ifTrue:[
  self size: 0 .
  ^ aCollection
  ].
errorBlock == nil ifTrue:[
  aCollection accompaniedBy: self do: [:me :obj | | idx |
    idx := me indexOf: obj.
    [ idx ~~ 0 ] whileTrue: [
      me removeFrom: idx to: idx.
      idx := me indexOf: obj startingAt: idx
    ].
  ].
] ifFalse:[
  self _removeAll: aCollection ifAbsent: errorBlock .
].
^ aCollection

]

{ #category : 'Removing' }
SequenceableCollection >> removeAllSuchThat: aBlock [

"Removes all elements of the receiver for which aBlock returns true.  Returns
 the removed elements."

| indexes result |

indexes := { } .
result := self speciesForCollect new.
1 to: self size do: [:idx | | item |
  item := (self at: idx).
  (aBlock value: item) ifTrue: [
    result add: item.
    indexes add: idx
  ]
].

indexes reverseDo: [:each |
  self removeFrom: each to: each
].
^ result

]

{ #category : 'Removing' }
SequenceableCollection >> removeAtIndex: anIndex [

"Removes the element at the given index.  Returns the removed element."

| anObj |
anObj := self at: anIndex .
self removeFrom: anIndex to: anIndex  .
^ anObj

]

{ #category : 'Removing' }
SequenceableCollection >> removeFirst [

"Removes the first element of the receiver.  Returns the removed element."

^ self removeAtIndex: 1

]

{ #category : 'Removing' }
SequenceableCollection >> removeFrom: startIndex to: stopIndex [

"Removes the elements of the receiver from startIndex to stopIndex inclusive.
 Returns the receiver.

 The size of the receiver is decreased by stopIndex - startIndex + 1."

<primitive: 601>

(stopIndex < startIndex)
ifTrue:
   [ ^ startIndex _error: #rtErrBadCopyFromTo args: { stopIndex }].
((stopIndex > self size) or: [(stopIndex < 1)])
   ifTrue: [ ^ self _errorIndexOutOfRange: stopIndex].
(startIndex < 1)
   ifTrue: [ ^ self _errorIndexOutOfRange: startIndex].
((self isKindOf: CharacterCollection )
  or:[ self isKindOf: ByteArray ]) ifTrue:[
   | range |
   range := stopIndex - stopIndex + 1.
   startIndex to: stopIndex do: [ :i |
     self at: i put: (self at: (i + range))].
   self size: (self size - range).
   ^ self
   ].
^ self _primitiveFailed: #removeFrom:to: args: { startIndex . stopIndex }

]

{ #category : 'Removing' }
SequenceableCollection >> removeIdentical: oldObject ifAbsent: anExceptionBlock [

"Removes from the receiver an object that is identical to oldObject.
 Returns oldObject.  If several elements of the receiver are identical to
 oldObject, only one instance is removed.  If oldObject is not present
 in the receiver, evaluates anExceptionBlock and returns the result
 of the evaluation."

| idx |

idx := self indexOfIdentical: oldObject.
(idx == 0) ifTrue: [
  ^ anExceptionBlock value.
] ifFalse: [
  ^ self removeAtIndex: idx.
].

]

{ #category : 'Removing' }
SequenceableCollection >> removeIdentical: anObject otherwise: notFoundValue [

"Removes from the receiver an object that is identical to anObject and
 returns anObject.  If several elements of the receiver are identical to
 oldObject, only one instance is removed.  If anObject is not present in
 the receiver, returns notFoundValue."

| idx |

idx := self indexOfIdentical: anObject.
(idx == 0) ifTrue: [
  ^ notFoundValue
] ifFalse: [
  ^ self removeAtIndex: idx.
].

]

{ #category : 'Removing' }
SequenceableCollection >> removeIfPresent: anElement [

"Removes anObject from the receiver and returns anObject.  If anObject is
 present several times in the receiver, only one occurrence is removed.
 Returns nil if anObject is missing from the receiver."

  | index |
  index := self indexOf: anElement .
  index == 0 ifFalse:[
    self removeAtIndex: index .
    ^ anElement .
  ].
  ^ nil

]

{ #category : 'Deprecated' }
SequenceableCollection >> removeIndex: index [

self deprecated: 'SequenceableCollection>>removeIndex: Obsolete in GemStone/64.  Use the removeAtIndex: method instead.'.
^ self removeAtIndex: index.

]

{ #category : 'Removing' }
SequenceableCollection >> removeLast [

"Removes the last element of the receiver.  Returns the removed element."

| lastElement selfSize |

selfSize := self size.
lastElement := self at: selfSize.
self size: (selfSize - 1).

^ lastElement.

]

{ #category : 'Deprecated' }
SequenceableCollection >> removeValue: oldObject [

self deprecated: 'SequenceableCollection>>removeValue: Obsolete in GemStone/64.  Use the remove: method instead.'.
^ self remove: oldObject.

]

{ #category : 'Updating' }
SequenceableCollection >> replaceFrom: startIndex to: stopIndex with: aCollection [

"Replaces the elements of the receiver between the indexes startIndex and
 stopIndex with the elements of aCollection.  Returns the receiver."

| i aSize |

((aSize := aCollection size) == (stopIndex - startIndex + 1))
  ifFalse: [ ^ self errorDifferentSizeCollections].

(self == aCollection) ifTrue: [ ^ self ].

(aCollection isKindOf: SequenceableCollection) ifTrue:[
  self replaceFrom: startIndex to: startIndex + aSize - 1 with: aCollection startingAt: 1 .
  ^ self
].
startIndex > (self size + 1) ifTrue:[ self _errorIndexOutOfRange: startIndex ].
i := startIndex.
aCollection accompaniedBy: self do: [:me :element |
  me at: i put: element.
  i := i + 1.
].
^ self.

]

{ #category : 'Updating' }
SequenceableCollection >> replaceFrom: startIndex to: stopIndex with: aCollection startingAt: repIndex [

"Replaces the elements of the receiver between the indexes startIndex and
 stopIndex inclusive with the elements of aCollection starting at repIndex.
 If aCollection is identical to the receiver, the source and
 destination blocks may overlap.

 aCollection must be a kind of SequenceableCollection or a kind of IdentityBag.

 Returns the receiver."

| srcIdx srcEnd mySize |
startIndex > ((mySize := self size) + 1) ifTrue:[ self _errorIndexOutOfRange: startIndex ].
(aCollection isKindOf: SequenceableCollection) ifFalse:[
  aCollection _validateClasses: { SequenceableCollection . IdentityBag }.
].
srcIdx := repIndex .
srcEnd := srcIdx + stopIndex - startIndex .
aCollection == self ifTrue:[
  srcIdx == startIndex ifTrue:[ ^ self ].
  srcIdx < startIndex ifTrue:[ | k |  "must copy backwards"
    k := srcEnd .
    mySize < stopIndex ifTrue:[ self size: stopIndex ].
    stopIndex _downTo: startIndex do:[:j |
       self at: j put: (aCollection at:  k).
       k := k - 1 .
    ].
    ^ self
  ]
].
srcEnd > aCollection size ifTrue:[ aCollection _errorIndexOutOfRange: srcEnd].
startIndex to: stopIndex do:[:j |
  self at: j put: (aCollection at: srcIdx ) .
  srcIdx := srcIdx + 1
].
^ self

]

{ #category : 'Updating' }
SequenceableCollection >> replaceFrom: startIndex to: stopIndex withObject: anObject [

"Replaces the elements of the receiver between the indexes startIndex and
 stopIndex with anObject. Returns the receiver."

startIndex to: stopIndex do: [ :i| self at: i put: anObject].
^ self.

]

{ #category : 'Copying' }
SequenceableCollection >> reverse [

"Returns a copy of the receiver with its elements in reverse order."
| copy sz high mid |
sz := self size.
copy := self species new: sz.
copy size < sz ifTrue:[ copy size: sz "handle OrderedCollection" ].
high := sz .
mid := sz bitShift: -1 . " // 2 "
1 to: mid do:[:low |
  copy at: low put: (self at: high) .
  copy at: high put: (self at: low) .
  high := high - 1 .
] .
mid < high ifTrue:[ "middle element of odd sized Array"
  copy at: high put: (self at: high)
].
^copy

]

{ #category : 'Enumerating' }
SequenceableCollection >> reverseDo: aBlock [

"Evaluates the one-argument block aBlock using each element of the receiver,
 in reverse order, as the argument."

self size _downTo: 1 do: [ :index |
  aBlock value: (self at: index).
  ].

^ self

]

{ #category : 'Accessing' }
SequenceableCollection >> second [
	"Compatible with Squeak/Pharo"
^ self at: 2

]

{ #category : 'Enumerating' }
SequenceableCollection >> select: aBlock [
  "Evaluates aBlock with each of the receiver's elements as the argument.
 Stores the values for which aBlock is true into a collection of the same
 class as the receiver, and returns the new collection.  The argument aBlock
 must be a one-argument block.

 The result preserves the ordering of the receiver.  That is, if element a
 comes before element b in the receiver, then element a is guaranteed to come
 before b in the result."

  | result |
  aBlock _isExecBlock ifFalse:[
   aBlock class == SelectBlock
    ifTrue: [
      "send value: to the block to create the 4 element
        associative access Array; the argument nil is ignored to
        create the Array, but all SelectBlocks have one argument"
      result := (QueryExecuter on: self)
        _boundQuerySelect: (aBlock queryBlock value: nil).
      ^ result ]].
  result := self species new.
  1 to: self size do: [ :i |  | each |
    each := self at: i.
    (aBlock value: each)
      ifTrue: [ result add: each ] ].
  ^ result

]

{ #category : 'Private' }
SequenceableCollection >> verifyAreFalse [

"Returns true if all elements of the receiver are false.  Otherwise, returns
 self."

1 to: self size do: [:j |
  false == (self at:j) ifFalse: [ ^ self ]
].
^ true

]

{ #category : 'Private' }
SequenceableCollection >> verifyAreTrue [

"Returns true if all elements of the receiver are true.  Otherwise, returns
 self."

1 to: self size do: [:j |
  true == (self at:j) ifFalse: [ ^ self ]
].
^ true

]

{ #category : 'Private' }
SequenceableCollection >> verifyElementsIn: aSeqColl [

"Check for equality in corresponding elements.  Returns true if the receiver
 and the argument are equal. Otherwise returns self."

(self size == aSeqColl size)
   ifFalse: [ ^ self ].
1 to: self size do: [:i |
   (self at: i) = (aSeqColl at: i)
      ifFalse: [ ^ self ]].
^ true

]

{ #category : 'Enumerating' }
SequenceableCollection >> with: aSequenceableCollection do: aBlock [

"Evaluate a two argument block aBlock using each element of the receiver as the
 first argument and the corresponding element of aSequenceableCollection as the
 second argument.  Returns the receiver."

(self size == aSequenceableCollection size)
  ifFalse: [ ^ self errorDifferentSizeCollections ].

1 to: self size do: [ :i|
  aBlock value: (self at: i) value: (aSequenceableCollection at: i).
  ].

^ self.

]
