Extension { #name : 'ProcessorScheduler' }

{ #category : 'Debugging' }
ProcessorScheduler class >> _allProcessesTerminate9 [
  self scheduler _allProcessesTerminate9

]

{ #category : 'Private' }
ProcessorScheduler class >> _isInitialized [
  ^((System __sessionStateAt: 15) ~~ nil)

]

{ #category : 'Priorities' }
ProcessorScheduler class >> highestPriority [
  "return the highest allowable priority"
  ^ 40   "also in C code as PROCESSSCHEDULER_highestPriority"
         "also other in-line highestPriority logic"

]

{ #category : 'Priorities' }
ProcessorScheduler class >> lowestPriority [
  "return the lowest allowable priority"
  ^1   "also in C code as PROCESSSCHEDULER_lowestPriority"
       "also other inline lowestPriority logic"
]

{ #category : 'Instance Creation' }
ProcessorScheduler class >> new [
  self shouldNotImplement: #new

]

{ #category : 'Accessing' }
ProcessorScheduler class >> scheduler [
  "This is equivalent to 'Processor' in the Blue Book.  It returns
   the instance of the receiver that is responsible for scheduling
   ready processes"

<primitive: 457>

^self _halt: 'Attempt to use process scheduler before it has been initialized'

]

{ #category : 'Accessing' }
ProcessorScheduler class >> topazAllProcesses [
 "Returns an Array of all GsProcess's known to the
  scheduler , excluding the current active process."
 | set |
 set :=  self scheduler allProcesses .
 set removeIfPresent: GsProcess _current .
 ^ set asArray .

]

{ #category : 'Accessing' }
ProcessorScheduler class >> topazStatusForProcess: aGsProcess [

^ self scheduler _statusString: aGsProcess

]

{ #category : 'Private' }
ProcessorScheduler >> _add: anObject toSet: anIdentitySet [

"Returns receiver."

<primitive: 726>
self _primitiveFailed: #_add:toSet: args: { anObject . anIdentitySet }

]

{ #category : 'Private' }
ProcessorScheduler >> _addDelay: aDelayOrProcess to: aDelayQ [

"Receiver should be the ProcessorScheduler's delayQueue.
 Appends aDelayOrProcess to the receiver, using semantics of
 a SortedCollection with sortBlock   [:d1:d2 | d1 _signalTime <= d2 _signalTime] .
 Returns receiver."

<primitive: 734>  "prim fails if aDelayQ is large"
aDelayQ add: aDelayOrProcess

]

{ #category : 'Private' }
ProcessorScheduler >> _addReadyProcess: aProcess to: aReadyQ [

"Receiver should be the ProcessorScheduler's readyQueue.
 Appends aProcess to the receiver, while keeping aReadyQ sorted based in
 priorities of the processes, per semantics of a SortedCollection 
 with sort block    [:p1:p2 | p1 priority >= p2 priority] .
 Returns receiver."

<primitive: 733>  "prim fails if aReadyQ is large"
aReadyQ add: aProcess

]

{ #category : 'Accessing' }
ProcessorScheduler >> _allProcessesInto: anIdentitySet [

  "Add all of the instances of GsProcess that are not terminated
   to result and return aCollection."
  | wasInCrit |
  wasInCrit := self _enterCritical .
  anIdentitySet _addAll: readyQueue forReplay: false .
  self _waitingProcessesInto: anIdentitySet .
  anIdentitySet _addAll: suspendedSet forReplay: false .
  anIdentitySet add: self activeProcess .
  wasInCrit == 0 ifTrue:[ self _exitCritical ].
  ^ anIdentitySet

]

{ #category : 'Accessing' }
ProcessorScheduler >> _allProcessesInto: anIdentitySet inGroup: groupId [

  "Add all of the instances of GsProcess that are not terminated
   and are in specified group to aSet and return anIdentitySet."

  | set proc polled wasInCrit |
  wasInCrit := self _enterCritical .
  anIdentitySet _addAll: readyQueue forReplay: false .
  1 to: (set := readyQueue) size do:[:n|
    proc := set at: n .
    proc _groupOrNil == groupId ifTrue:[ anIdentitySet add: proc ].
  ].
  self _waitingProcessesInto: anIdentitySet inGroup: groupId .
  1 to: (set := suspendedSet) size do:[:n|
    proc := set _at: n .
    proc _groupOrNil == groupId ifTrue:[ anIdentitySet add: proc ].
  ] .
  1 to: (polled := socketsPolled) size do:[:j |
    (polled at: j) _waitingProcessesInto: anIdentitySet inGroup: groupId .
  ].
  wasInCrit == 0 ifTrue:[ self _exitCritical ].
  ^ anIdentitySet

]

{ #category : 'Debugging' }
ProcessorScheduler >> _allProcessesTerminate9 [
  "Send terminate 9 to all GsProcess's known to the
   scheduler , excluding the current active process."
  | set polled |
  set :=  self allProcesses .
  set removeIfPresent: GsProcess _current .
  set do:[:proc | self _unschedule: proc ].
  [
    set do:[:aProcess | aProcess terminate9 ].
    polled := socketsPolled shallowCopy .
    1 to: polled size do:[:j |
      (polled at: j) _disableEvent: nil ."disable all events and polling"
    ].
  ] onException: AbstractException do:[:ex |
    ex resume "ignore"
  ].

]

{ #category : 'Debugging' }
ProcessorScheduler >> _allStacksReport [
  "includes the stack of the currently executing process."
  | allP rpt |
  rpt := String new .
  allP := self allProcesses .
  allP do:[:aProc | rpt add: aProc _reportString ].
  ^ rpt

]

{ #category : 'Private' }
ProcessorScheduler >> _changePriority: aProcess from: oldPriority [
  | curr |
  (curr := self activeProcess) == aProcess ifTrue:[
    aProcess priority >= oldPriority ifTrue:[ ^ self "keep running aProcess" ].
    self yield .
  ] ifFalse:[ | procQ q |
    (procQ := aProcess onQueue) == (q := readyQueue) ifTrue:[
      q removeIdentical: aProcess otherwise: nil .
      self _addReadyProcess: aProcess to: q .  "add at new priority"
      aProcess priority > oldPriority ifTrue:[
        self yield .
      ].
    ].
    "process not on the readyQueue, leave it suspended or waiting on a Delay"
  ].
]

{ #category : 'Private' }
ProcessorScheduler >> _checkForExpiredDelays: delayQ [
    "returns number of expired delays"
    | readyCount dlysSize now |
    readyCount := 0 .
    dlysSize := delayQ size .
    now := self _timeoutNow: dlysSize .
    dlysSize ~~ 0 ifTrue:[ | idx ofs dT |
      idx := 0 .
      ofs := 1 .
      [ ofs <= dlysSize ] whileTrue:[  | aDelay "a Delay or GsProcess or Semaphore" |
        "search  delayQ for the first item not yet expired.
         For items that have expired, activate them so they will be scheduled.
        "
        aDelay := delayQ at: ofs .
        ((dT := aDelay _signalTime) ~~ nil and:[ dT > now])  ifTrue: [ "this one not expired"
	  idx := ofs .
          ofs := dlysSize + 1 . "exit loop"
        ] ifFalse: [  	"this one expired"
	  "self dbglog: ('sched: activating a ', aDelay class name , ' time: ',
		now asString, ', _signalTime: ', aDelay _signalTime asString).  "
          readyCount := readyCount + aDelay _finishDelay  .
          ofs := ofs + 1 .
        ].
      ].
      (idx == 0) ifTrue: [
        "all delays have expired, tell timeout thread to wait for more work."
	delayQ size: 0 .
	dlysSize := 0 .
        timeoutThreadActive ifTrue:[
          self _updateTimerThread: -1 priority: 15"inline userSchedulingPriority" .
          timeoutThreadActive := false .
        ].
        "self dbglog: 'sched: all delays expired' ."
      ] ifFalse:[
        idx ~~ 1 ifTrue: [
          delayQ removeFrom: 1 to: (idx - 1). "delete expired items"
	  dlysSize := delayQ size .
        ].
	dlysSize ~~ 0 ifTrue:[ | nextDly |
	  nextDly := delayQ at: 1 .
	  self _updateTimerThread: nextDly _signalTime priority: nextDly priority .
          timeoutThreadActive := true .
        ].
      ].
    ] .
    ^ readyCount
]

{ #category : 'In-Process Debugger Support' }
ProcessorScheduler >> _criticalMethods [

"Returns an IdentitySet of GsNMethods."

^ criticalMethods ifNil:[ self _initCriticalMethods ]

]

{ #category : 'Delay Access' }
ProcessorScheduler >> _delaySchedule: aProcessOrDelay [
  "Add aProcessOrDelay to the delayQueue ."

  self _addDelay: aProcessOrDelay to: delayQueue .

]

{ #category : 'Delay Access' }
ProcessorScheduler >> _delayUnschedule: aDelay [
  "Unschedule aDelay in the queue."

  delayQueue removeIdentical: aDelay otherwise: nil

]

{ #category : 'Private' }
ProcessorScheduler >> _doPoll: resultArray yield: yieldInt [

"Wait for a GsSocket to be ready or for a timeout to expire.
 yieldInt > 0 means yield , i.e. just poll for GsSocket(s) ready ,
 yieldInt == 0 means poll until GsSocket(s) ready or until
 timer C thread delivers EINTR to main thread when a timeout expires.

 resultArray must be an Array. It will have stored into it, each
 instance of GsSocket that had an event detected on it.
 Returns nil if there was a socket error, otherwise returns number of
 elements in resultArray, or zero if the primitive timed out."

<primitive: 192>
self _uncontinuableError

]

{ #category : 'Private' }
ProcessorScheduler >> _enableEvents [
  "perform the enable interrupts part of _gsReturnNothingEnableEvents"
  <primitive: 141>
  self _uncontinuableError
]

{ #category : 'Private' }
ProcessorScheduler >> _enterCritical [
"enter critcal region. 
 returns a SmallInteger 0==entered critical region, 1==already in critical region"

<primitive: 169> 
self _primitiveFailed: #_enterCritical
]

{ #category : 'Private' }
ProcessorScheduler >> _exitCritical [

"clears OM.schedulerInReapEvents, which is also cleared by prim 2010"

<primitive: 737>
self _primitiveFailed: #_exitCritical

]

{ #category : 'Private' }
ProcessorScheduler >> _findNextProcess [
  [ self _findReadyProcess  ifNotNil:[:newProc| 
      ^ newProc
    ] ifNil:[
      self _rescheduleContinue: nil .
      false
    ]
  ] whileFalse .
]

{ #category : 'Private' }
ProcessorScheduler >> _findReadyProcess [

  "Find another process to run, waiting for events if no process is
   ready to run.  Signals a ThreadError if a deadlock would occur because
   no other processes are ready nor waiting.
   Returns the next process to run or returns nil if interrupted by OOB."

  <primitive: 736> "prim always fails, sets OM.reapEventsCount "
  | proc err oldProc rpt res |
  res := self _reapEvents: 0 .
  res ~~ self ifTrue:[
    res == 0 ifTrue:[ "no events but interrupted by OOB from stone"
      ^ nil
    ].
    "we have  res == nil"
    oldProc := self activeProcess .
    oldProc ifNotNil:[ | q |  "normally oldProc is the active process"
        (q := oldProc onQueue) ifNotNil:[
           q ~~ readyQueue ifTrue:[
             oldProc _onQueue: nil .
             self _remove: oldProc fromSet: q .  "waitingSet or suspendedSet"
           ].
        ]
    ].
    (rpt := '----- ProcessorScheduler deadlock ' copy) add: ' current process ' ;
                           add: oldProc asOop asString ; lf .
    rpt addAll: self _allStacksReport .
    GsFile gciLogServer: rpt .
    (err := ThreadError new) _number: 2366 ; reason: #rtErrSchedulerDeadlocked;
	 args: { self . oldProc . rpt } ;
         details: 'The ProcessorScheduler is deadlocked' .
    self _exitCritical .
    err signal .
  ].
  proc := readyQueue _removeAtOrNil: 1 . "removeFirst"
  proc ifNotNil:[
    proc _onQueue: nil .
    "caller must send _switchFrom:to: or _exitCritical"
  ].
  ^ proc

]

{ #category : 'Private' }
ProcessorScheduler >> _findReadyProcessForExit: oldProc [
  "Returns either nil or the next GsProcess to run.
   If result is nil, caller must signal an appropriate exception .
   If result is not nil, caller must send
    _switchFrom:to:  to the scheduler to switch to returned GsProcess."

  "caller has entered critical region"
  | proc status |
  status := self _reapEvents: 0 .
  status == 0 ifTrue:[ status := self _reapEvents: 0  ].
  status ~~ self ifTrue:[
     oldProc ifNotNil:[ | q |  "normally oldProc is the active process"
       (q := oldProc onQueue) ifNotNil:[
          q ~~ readyQueue ifTrue:[
            oldProc _onQueue: nil .
            self _remove: oldProc fromSet: q .  "waitingSet or suspendedSet"
          ].
       ]
     ].
     ^ nil   "no other processes available to run"
  ].
  proc := readyQueue _removeAtOrNil: 1 .
  proc ifNotNil:[ proc _onQueue: nil ].
  ^ proc

]

{ #category : 'Private' }
ProcessorScheduler >> _initCriticalMethods [

  "Methods which an in-process debugger should not set breakpoints in nor
    single step through.  

  The method compiler omits check interrupt bytecodes that would otherwise be 
  generated when compiling an instance method in ProcessorScheduler.
  Interrupt check on backwards branches are generated in those instance methods.
"
  | arr set |
  arr := {
    "primitive 457"
    #CriticalSection  . #_scheduler .
    Delay  . #_scheduler .
    Delay class  . #_scheduler .
    GsProcess  . #_scheduler .
    GsProcess class  . #_scheduler .
    GsSocket  . #_scheduler .
    Processor class  . #_scheduler .
    ProcessorScheduler class  . #scheduler .
    Semaphore  . #_scheduler .

    "senders scheduler"
    AbstractException  . #_signalTimeout: .
    GcFinalizeNotification . #_signalGcFinalize: .
    AbstractException . #_signalAsync: .

    ProcessorScheduler class  . #topazAllProcesses .
    ProcessorScheduler class  . #topazStatusForProcess: .
    ProcessorScheduler class  . #_allProcessesTerminate9 .

    "senders _scheduler"
    #CriticalSection  . #critical: .
    Delay  . #signal .
    Delay  . #_cancel: .
    Delay  . #_finishDelay .
    Delay  . #_wait: .
    Delay class  . #waitForMilliseconds: .
    Delay class  . #waitForSeconds: .
    Delay class  . #waitUntilMilliseconds: .
    Delay class  . #_highPriorityWaitForMilliseconds: .
    GsNMethod class  . #_terminateNonForkedProcess .
    GsProcess  . #priority: .
    GsProcess  . #resume .
    GsProcess  . #suspend .
    GsProcess  . #terminate .
    GsProcess  . #_finishDelay .
    GsProcess  . #_newGroup .
    GsProcess  . #_raisePriority .
    GsProcess  . #_reapSignal: .
    GsProcess  . #_signalAll .
    GsProcess  . #_signalJoiners .
    GsProcess  . #_start .
    GsProcess  . #_statusString .
    GsProcess  . #_stepOverInFrame:mode:replace:tos:breakpointLevel: .
    GsProcess  . #_unscheduleProcess .
    GsProcess  . #_wait .
    GsProcess class  . #_forkBlock:with:env:prio: .
    GsProcess class  . #_newForBlock:with:env: .
    GsSocket  . #readWillNotBlockWithin: .
    GsSocket  . #writeWillNotBlockWithin: .
    GsSocket  . #_waitForReadReady .
    GsSocket  . #_waitForWriteReady .
    Processor class  . #activePriority .
    Processor class  . #activeProcess .
    Processor class  . #allProcesses .
    Processor class  . #cancelWhenReadable:signal: .
    Processor class  . #cancelWhenWritable:signal: .
    Processor class  . #highestPriority .
    Processor class  . #highIOPriority .
    Processor class  . #lowestPriority .
    Processor class  . #lowIOPriority .
    Processor class  . #readyProcesses .
    Processor class  . #suspendedProcesses .
    Processor class  . #systemBackgroundPriority .
    Processor class  . #timingPriority .
    Processor class  . #userBackgroundPriority .
    Processor class  . #userInterruptPriority .
    Processor class  . #userSchedulingPriority .
    Processor class  . #waitingProcesses .
    Processor class  . #whenReadable:signal: .
    Processor class  . #whenWritable:signal: .
    Processor class  . #yield .
    Semaphore  . #highPriorityWaitForMilliseconds: .
    Semaphore  . #signal .
    Semaphore  . #signalAll .
    Semaphore  . #wait .
    Semaphore  . #waitForMilliseconds: .

    "implem _exitCritical"
    ProcessorScheduler  . #_exitCritical .
    Semaphore  . #_exitCritical .

    "senders _exitCritical"
    Delay  . #_wait: .
    Delay class  . #_highPriorityWaitForMilliseconds: .
    GsProcess  . #_signalJoiners .
    GsProcess  . #_stepOverInFrame:mode:replace:tos:breakpointLevel: .
    GsProcess  . #_trimStackToLevel:abandonActiveUnwind:numTries:timeout: .
    GsProcess  . #_finishTermination: .
    GsProcess  . #_executeUnwindBlk:numTries:timeout:stack:topFp:resumeCnt: .
    GsProcess  . #_waitingOn: .
    GsProcess  . #_waitForActiveUnwind:timeoutMs:numTries:resumeCnt: .

    ProcessorScheduler  . #waitingProcesses .
    ProcessorScheduler  . #_allProcessesInto: .
    ProcessorScheduler  . #_allProcessesInto:inGroup: .
    ProcessorScheduler  . #_terminateScheduledProcess: .
    ProcessorScheduler  . #_waitForMilliseconds: .
    ProcessorScheduler  . #_waitForSocket:timeout:forWrite: .
    ProcessorScheduler  . #yield .
    ProcessorScheduler  . #_findReadyProcess .
    ProcessorScheduler  . #_findReadyProcessForExit: .
    ProcessorScheduler  . #_runNextProcess .
    ProcessorScheduler  . #_yieldForTimeout .
    Semaphore  . #highPriorityWaitForMilliseconds: .
    Semaphore  . #signal .
    Semaphore  . #signalAll .
    Semaphore  . #wait .
    Semaphore  . #waitForMilliseconds: .

    "primitive 737"
    ProcessorScheduler  . #_exitCritical .

    "primitive 693"
    GsProcess . #_current  .
    GsProcess class . #_current  .
    ProcessorScheduler . #activeProcess  .

    "primitive 169"
    ProcessorScheduler . #_enterCritical  .

    "primitive 736"
    ProcessorScheduler . #waitingProcesses  .
    ProcessorScheduler . #yield  .
    ProcessorScheduler . #_allProcessesInto:  .
    ProcessorScheduler . #_allProcessesInto:inGroup:  .
    ProcessorScheduler . #_runNextProcess  .
    ProcessorScheduler . #_terminateScheduledProcess:  .
    ProcessorScheduler . #_yieldForTimeout  .
    ProcessorScheduler . #readyProcesses  .
    Semaphore . #signal  .
    Semaphore . #signalAll  .
    Semaphore . #wait  .

   "senders _reschedule"
    ProcessorScheduler . #_suspendProcess:  .
    ProcessorScheduler . #_waitForMilliseconds:  .
    ProcessorScheduler . #_waitForSocket:  .
    ProcessorScheduler . #_waitForSocket:timeout:forWrite:  .
    ProcessorScheduler . #_waitOnSema:  .


    "other"
    ProcessorScheduler  . #_reapEvents: .
    ProcessorScheduler  . #_checkForExpiredDelays: .
    ProcessorScheduler . #_now  .
    ProcessorScheduler . #_timeoutNow:  .
    ProcessorScheduler . #_delayUnschedule:  .
    ProcessorScheduler . #_switchFrom:to:  .
    ProcessorScheduler . #_runNextProcess  .
    ProcessorScheduler . #_rescheduleContinue: .
    ProcessorScheduler . #_reschedule  .
    ProcessorScheduler . #_doPoll:yield:  .
    ProcessorScheduler . #_isWhenReadableSet:signal:  .
    ProcessorScheduler . #_isWhenWritableSet:signal:  .
    ProcessorScheduler . #whenReadable:signal:  .
    ProcessorScheduler . #whenWritable:signal:  .
    ProcessorScheduler . #cancelWhenReadable:signal:  .
    ProcessorScheduler . #cancelWhenWritable:signal:  .
    ProcessorScheduler . #_unschedule:  .
    ProcessorScheduler . #_suspendNewProcess:  .
    ProcessorScheduler . #_resumeProcess:  .
    ProcessorScheduler . #_resumeForTermination:resumeCnt:  .
    ProcessorScheduler . #suspendedProcesses  .
    ProcessorScheduler . #_add:toSet:  .
    ProcessorScheduler . #_remove:fromSet:  .
    ProcessorScheduler . #_changePriority:from:  .
    ProcessorScheduler . #_addReadyProcess:to:  .
    ProcessorScheduler . #_addDelay:to:  .
    ProcessorScheduler . #_updateTimerThread:priority:  .
    ProcessorScheduler . #_sizeNoShrink:  .
    ProcessorScheduler . #_allStacksReport .

    GsProcess .  #resume  .

    GsProcess class . #_newForBlock:with:env:  .
    GsProcess . #_init:args:env:stackSerial:forked:  .
    GsProcess class . #_forkBlock:with:env:prio:  .
    ProcessorScheduler . #_scheduleNewProcess:prio:
  }.
  set := IdentitySet new .
  1 to: arr size by: 2 do:[:n |
    | cls sel meth |
    cls := arr at: n .      "tolerate execution before end of slow filein / image upgrade"
    cls _isSymbol ifTrue:[ cls := Globals at: cls otherwise: nil ].
    sel := arr at: n + 1 .
    cls ifNotNil:[  
      meth := cls compiledMethodAt: sel otherwise: nil .
      meth ifNotNil:[ set add: meth ] .
    ].
  ].
  criticalMethods := set .
  ^ set

]

{ #category : 'Initialization' }
ProcessorScheduler >> _initialize: initSessionArg [
  "This method is invoked once by the VM during session initialization.
   The single instances of ProcessorScheduler and GsCurrentSession
   are created by the VM before invoking this method.

   If initSessionArg~~nil , invokes GsCurrentSession(C) initialize

   In the following discussion, the single instance of GsCurrentSession
   is 'currSess' .

   Before invoking this method, the VM has completed these steps :
     1) default timeZone is fetched from (Globals at:#TimeZone) and
     installed in session state.   This wil be either the
     value of the class instance variable 'default' in TimeZone ,
     or the class variable #Default in the ObsoleteTimeZone class
     (if the 2007 upgrade to TimeZone has not been installed).

   Initializes the receiver's process queues.

   Invokes GsCurrentSession >> initialize  to execute any application
   defined initialization."

  readyQueue := SortedCollection sortBlock: [:p1:p2 | p1 priority >= p2 priority].
		"sort block must agree with prim 733"
  activeProcess := nil.
  delayQueue := SortedCollection sortBlock: [:d1:d2 | d1 _signalTime <= d2 _signalTime].
		"sort block must agree with prim 734"
  suspendedSet := IdentitySet new: 4 .
  waitingSet := IdentitySet new: 4 .
  lastGroup := 0.
  pollResults := Array new: 5 . pollResults size: 0 .
  socketsPolled := Array new: 5 . socketsPolled size: 0 .
  timeoutThreadActive := false .
  self _setInstVarsAlwaysValid .      "fix 44837"

  System __sessionStateAt: 15 put: self.
  (Globals at: #CharacterDataTables otherwise: nil) ifNotNil:[
    [
      (System gemEnvironmentVariable:'GS_DISABLE_CHARACTER_TABLE_LOAD') ifNil:[
        Character _loadCharTables.
      ]
    ] onException: AbstractException do:[:ex | "ignore" ].
  ].
  initSessionArg ifNotNil:[
    GsCurrentSession initialize
  ].
  "No user or application changes here.
   Customization is to be done in GsCurrentSession >> initialize"

]

{ #category : 'Accessing' }
ProcessorScheduler >> _isSuspended: aGsProcess [
  "Returns a Boolean "

  ^ suspendedSet includesIdentical: aGsProcess

]

{ #category : 'Private' }
ProcessorScheduler >> _isWhenReadableSet: aGsSocket signal: objToNotify [
  "Returns true if read event is still enabled.
   Returns false if read event was never set or has happened and been
   automatically unset."

  ^ aGsSocket hasReadWaiter: objToNotify

]

{ #category : 'Private' }
ProcessorScheduler >> _isWhenWritableSet: aGsSocket signal: objToNotify [
  "Returns true if write envent is still enabled.
   Returns false if write event was never set or has happened and been
   automatically unset."
  ^ aGsSocket hasWriteWaiter: objToNotify

]

{ #category : 'Private' }
ProcessorScheduler >> _newGroup [
  lastGroup := lastGroup + 1.
  ^lastGroup

]

{ #category : 'Delay Access' }
ProcessorScheduler >> _now [
  "Returns a SmallInteger, the time that should be used by a Delay to represent to
   current time in milliseconds."

<primitive: 651>
^ self _primitiveFailed: #_now

]

{ #category : 'Private' }
ProcessorScheduler >> _reapEvents: yieldInt [

  "Reaps any pending events that will cause a thread to be ready to run.
   yieldInt should be 0 to wait for an event, or 1 for  yielding.
   Returns nil if scheduler would deadlock,
   returns 0 if an OOB byte from stone has interrupted the waiting,
   returns self otherwise.
  "
  "senders must enter critical region by using prim 736"

  | delayQ readyQ pollCount |
  delayQ := delayQueue .
  readyQ := readyQueue .
  pollCount := 0 .
  [
    | readyCount dlysSize |
    self _checkForExpiredDelays: delayQ .
    readyCount := readyQ size .
    dlysSize := delayQ size .

   "  uncomment for tracing _reapEvents"
   "GsFile gciLogServer: '_reapEvents yieldInt ', yieldInt asString,' readyCount ', 
         readyCount asString,' dlysSize ', dlysSize asString,' polledSize ', 
         socketsPolled size asString, ' loopCount ', pollCount asString ."

    (dlysSize + socketsPolled size ) ~~ 0 ifTrue: [
      | resSize resArray |
         "yield use case:  poll for socket activity, sleepVal== 0
          otherwise:  waitforever for socket activity, or wakeup by timeout thread."

      "  uncomment for tracing _reapEvents"
      "GsFile gciLogServer: '_doPoll  ', ((readyCount + yieldInt) > 0 
               ifTrue:['yield'] ifFalse:['poll']). "

      resSize := self _doPoll: (resArray := pollResults) yield: readyCount + yieldInt .
      pollCount := pollCount + 1 .
      resSize ~~ 0 ifTrue:[
	1 to: resSize do:[:k | | aSocket |
	  aSocket := resArray at: k .
	  aSocket _reapEvents .
        ].
        self _sizeNoShrink: resArray .
        readyCount := readyQ size.
      ].
      yieldInt ~~ 0 ifTrue:[ ^ self ].
      readyCount ~~ 0 ifTrue:[ ^ self ]
        ifFalse:[
          (self _checkForExpiredDelays: delayQ) ~~ 0 ifTrue:[ ^ self ]
      ].
      "self dbglog: 'sched: no events but interrupted' ."
      ^ 0 "no events but interrupted"
    ] ifFalse:[
      readyCount := readyQ size .
      (readyCount + yieldInt) ~~ 0 ifTrue: [
        ^ self
      ] ifFalse:[
        "readyCount == 0 ifTrue:[  self pause ]." "uncomment to debug deadlock"
        ^ nil .  "would deadlock if caller is not yield"
      ].
    ] .
    pollCount > 100 ifTrue:[  | err |
      (err := ThreadError new) _number: 2366 ; reason: #rtErrSchedulerDeadlocked;
	  args: { self . nil } ;
          details: 'The ProcessorScheduler is deadlocked, infinite loop in _reapEvents: ' .
       self _exitCritical .
       err signal
    ].
    true
  ] whileTrue "forever loop"

]

{ #category : 'Private' }
ProcessorScheduler >> _remove: anObject fromSet: anIdentitySet [

"Returns receiver. If size of anIdentitySet goes to zero
and it is committed, regenerates the in-memory state as not committed."

<primitive: 727>
self _primitiveFailed: #_remove:fromSet: args: { anObject . anIdentitySet }

]

{ #category : 'Private' }
ProcessorScheduler >> _reschedule [
  "Find a process that is ready to run. Once found suspend originalOldProcess
   and activate the new process."
  | newProc oldProc |
  oldProc := self activeProcess .
  [ (newProc := self _findReadyProcess ) ifNotNil:[
      self _switchFrom: oldProc to: newProc . "exits critical"
      "We only get here once the old process is once again active."
      true
    ] ifNil:[
      self _rescheduleContinue: oldProc .
      false
    ]
  ] whileFalse .
]

{ #category : 'Private' }
ProcessorScheduler >> _rescheduleContinue: oldProc [
  | noEx |
  [
    self _exitCritical .
    1 timesRepeat:[ self class ] . "honor interrupts"
    noEx := true .
    self _enterCritical .
  ] ensure: [  "if an Exception is not resumed, unschedule oldProc"
    noEx ifNil:[ | wasInCrit |
      wasInCrit := self _enterCritical .
      oldProc ifNotNil:[ self _unschedule: oldProc ].
      wasInCrit == 0 ifTrue:[ self  _exitCritical ].
    ].
  ].
]

{ #category : 'Private' }
ProcessorScheduler >> _resumeForTermination: aProcess resumeCnt: resumeCntArray [
  "Put a suspended process back to the state it was in when it was suspended."
  "| trace | trace := true."  
  "trace ifNotNil:[ GsFile gciLogServer:'_resumeForTermination proc ', aProcess asOop asString,
                     ' status ' , (self _statusString: aProcess) ]."
  (self _resumeProcess: aProcess) ifFalse:[ | readyQ |
    aProcess waitingOn ifNotNil:[:x |
      (x isKindOf: Semaphore) ifTrue:[ | proc |
        "trace ifNotNil:[GsFile gciLogServer:'resuming from a Semaphore', aProcess asOop asString]."
        proc := waitingSet removeIdentical: aProcess otherwise: nil .
        proc == aProcess ifTrue:[ 
				  aProcess _waitingOn: nil .
				  self _addReadyProcess: proc to: (readyQ := readyQueue) .
				  proc _onQueue: readyQ .
        ] ifFalse:[
          Error signal:'Failure in _resumeForTermination'.
        ]
      ] ifFalse: [
        "trace ifNotNil:[GsFile gciLogServer:' Waiting on a ', x class name, ' proc ',
               aProcess asOop asString ]."
      ]
    ] ifNil:[
      aProcess isForked ifFalse:[
        (resumeCntArray at: 1) == 1 ifTrue:[
          "A process which was returned to the GCI as context of an Exception"
          "trace ifNotNil:[GsFile gciLogServer:'resuming from GCI', aProcess asOop asString]."
			    self _addReadyProcess: aProcess to: (readyQ := readyQueue) .
			    aProcess _onQueue: readyQ .
        ].
      ].
    ].
  ].
]

{ #category : 'Private' }
ProcessorScheduler >> _resumeProcess: aProcess [
  "Put a suspended process back to the state it was in when it was
   suspended. Returns true if the process was made ready to run
   false otherwise. "

  | q readyQ proc "trace" |
  "trace := true ."
  q := aProcess onQueue .
	q == suspendedSet ifTrue:[
		suspendedSet removeIdentical: aProcess otherwise: nil .
		"trace ifNotNil:[GsFile gciLogServer:'resuming from suspended ', aProcess asOop asString]."
		aProcess waitingOn ifNotNil:[ | ws |
			self _add: aProcess toSet: (ws := waitingSet) .
			aProcess _onQueue: ws .
      ^ false
		] ifNil:[
			self _addReadyProcess: aProcess to: (readyQ := readyQueue) .
			aProcess _onQueue: readyQ .
			^ true
		].
  ]. 
  q == (readyQ := readyQueue) ifTrue:[
    "already on readyQ"
		"trace ifNotNil:[GsFile gciLogServer:'_resumeProc already on readyQ ', aProcess asOop asString]."
    ^ true .
  ].
	proc := delayQueue removeIdentical: aProcess otherwise: nil .
	proc == aProcess ifTrue:[
    "proc waitingOn  may be non-nil for a process waiting on a socket with a timeout."
		"trace ifNotNil:[GsFile gciLogServer:'resuming from delayQ ', aProcess asOop asString]."
		proc _signalTime: nil .
		self _addReadyProcess: proc to: (readyQ := readyQueue) .
		proc _onQueue: readyQ .
		^ true.
  ].
	aProcess waitingOn ifNotNil:[:x | | aProc |
		(x isKindOf: Delay) ifTrue:[ | aDelay |
			aDelay := delayQueue removeIdentical: x otherwise: nil .
			aDelay == x ifTrue:[
				"trace ifNotNil:[ GsFile gciLogServer:'resumeA from a Delay ', aProcess asOop asString]."
				aProcess _signalTime: nil .
				aProcess _waitingOn: nil .
				self _addReadyProcess: aProcess to: readyQ .
				aProcess _onQueue: readyQ .
				^ true.
			].
			UncontinuableError signal:'bad Delay state'. 
		].
    "x should be either a GsSocket or a Semaphore"
		aProc := waitingSet removeIdentical: aProcess otherwise: nil .
		aProc == aProcess ifTrue:[
				"trace ifNotNil:[GsFile gciLogServer:'resume from a GsSocket ', aProcess asOop asString]."
				aProcess _signalTime: nil .
				aProcess _waitingOn: nil .
				self _addReadyProcess: aProcess to: readyQ .
				aProcess _onQueue: readyQ .
				^ true
		].
		UncontinuableError signal:'bad waiting state'. 
  ].
  "could have been active GsProcess that signalled error to GCI, and is not in scheduler state
   and is being terminated."
	"trace ifNotNil:[GsFile gciLogServer:'_resumeProcess:', aProcess asOop asString,
               ' about to add to readyQueue C']."
  aProcess _signalTime: nil .
  self _addReadyProcess: aProcess to: readyQ .
  aProcess _onQueue: readyQ .
  ^ true
]

{ #category : 'Private' }
ProcessorScheduler >> _runNextProcess [
  "Used when a blocked thread exits to kick the scheduler so it will run
   another process."
  <primitive: 736> "prim always fails, sets OM.reapEventsCount "

  | newProc ex oldProc |
  [ newProc := self _findNextProcess .
    (newProc == nil or:[
      newProc == (oldProc := self activeProcess)]) ifTrue:[
      (ex := ThreadError new) details:
          'ProcessScheduler _runNextProcess failed, new ' , newProc asOop asString,
			' old ' , oldProc asOop asString .
      self _exitCritical .
      ex signalNotTrappable
    ].
    newProc _isTerminated
  ] whileTrue .

  self _switchFrom: nil to: newProc .  "primitive updates activeProcess instVar"
  "Should never return"

  GsFile gciLogServer: 'ERROR in _runNextProcess , no process to run'.
  self _uncontinuableError

]

{ #category : 'Private' }
ProcessorScheduler >> _scheduleNewProcess: aProcess prio: aPriority [
  "Put a newly created process on the ready queue"
  | readyQ p |
  aPriority ifNil:[ p := self activePriority ]
    ifNotNil:[ 
     p := aPriority .
     p < 1 ifTrue:[ OutOfRange new name: 'priority' min: 1 actual: p ; signal].
  ].
  aProcess _setPriority: p .
  self _addReadyProcess: aProcess to: (readyQ := readyQueue) .
  aProcess _onQueue: readyQ .
]

{ #category : 'Private' }
ProcessorScheduler >> _scheduleProcess: aProcess [
  "add the given process to the correct ready-queue"

  | q readyQ |
  readyQ := readyQueue .
  q := aProcess onQueue .
  q ~~ readyQ ifTrue:[
    q ifNotNil:[
      aProcess _onQueue: nil .
      self _remove: aProcess fromSet: q .  "waitingSet or suspendedSet"
    ].
    self _addReadyProcess: aProcess to: readyQ .
    aProcess _onQueue: readyQ  .
  ].

]

{ #category : 'Private' }
ProcessorScheduler >> _setInstVarsAlwaysValid [

"Returns receiver.  Prevent values of instVars from being changed by abort
if they do get committed."

<primitive: 995>
self _primitiveFailed: #_setInstVarsAlwaysValid

]

{ #category : 'Private' }
ProcessorScheduler >> _sizeNoShrink: anArray [

"Sets logical size of anArray to zero; if anArray is a small object,
 the operation does not shrink the physical size of anArray .
 Returns receiver. "

<primitive: 739>
self _primitiveFailed: #_sizeNoShrink: args: { anArray }

]

{ #category : 'Private' }
ProcessorScheduler >> _statusString: aGsProcess [
  "Returns a string that describes aGsProcess's status."

  | result q waitOn |
  aGsProcess == self activeProcess ifTrue: [
     ^ 'active'
  ].
  q := aGsProcess onQueue .
  q == readyQueue ifTrue: [
    ^ 'ready'
  ].
  q == suspendedSet ifTrue: [
    ^ 'suspended'
  ].
  waitOn := aGsProcess waitingOn .
  (q == waitingSet or:[ waitOn ~~ nil]) ifTrue:[
    result := String new.
    waitOn _isSymbol ifTrue:[
      result add: 'waiting on ' ; add: waitOn . "usually for Ruby IO.select"
    ] ifFalse:[
      result add: 'waiting on a '; add: waitOn class name .
    ].
    aGsProcess _signalTime ifNotNil:[
        result addLast: ', with a timeout'
    ].
  ] ifFalse:[
    q ifNotNil:[
      self halt:'invalid process state'.
    ].
    (aGsProcess _isTerminated) ifTrue: [
      result := 'terminated'.
    ] ifFalse: [
      aGsProcess _terminationStarted ifTrue:[
       result := 'terminationStarted' .
      ] ifFalse:[
        (delayQueue indexOfIdentical: aGsProcess) ~~ 0 ifTrue:[
          result := 'on delayQueue' .
        ] ifFalse:[
          "GCI application must have it in the debug state"
          result := 'debug'.
        ].
      ].
    ].
  ].
  ^ result

]

{ #category : 'Private' }
ProcessorScheduler >> _suspendNewProcess: aProcess [

 "put a newly created Process in the suspendedSet ."

  | suspSet |
  aProcess _setPriority: self activePriority .
  self _add: aProcess toSet: (suspSet := suspendedSet) .
  aProcess _onQueue: suspSet .

]

{ #category : 'Private' }
ProcessorScheduler >> _suspendProcess: aProcess [
  "suspend the process and do not schedule it for further processing"

  | suspSet |
  (aProcess == self activeProcess) ifTrue:[
    self _add: aProcess toSet: (suspSet := suspendedSet) .
    aProcess _onQueue: suspSet  .
    self _reschedule .
  ] ifFalse:[ 
    suspSet := suspendedSet .
    aProcess onQueue ifNotNil:[ :onQ |
      onQ == suspSet ifTrue:[ ^ self "done" ].
      onQ removeIdentical: aProcess otherwise: nil .
    ].
    self _add: aProcess toSet: suspSet .
    aProcess _onQueue: suspSet .
  ].

]

{ #category : 'Private' }
ProcessorScheduler >> _switchFrom: oldProcess to: newProcess [
  "Suspend the currentProcess and activate the receiver"

<primitive: 2010>
"prim clears OM.schedulerInReapEvents .
 No primitive failure code. Execution resumes upon resumption
 of oldProcess "
^ self

]

{ #category : 'Process Access' }
ProcessorScheduler >> _terminateScheduledProcess: aGsProcess [
  "remove the given process from its queues it is in"

  <primitive: 736> "enter critical region, prim always fails"
  (aGsProcess == self activeProcess) ifTrue: [ | nxt |
    nxt := self _findReadyProcessForExit: aGsProcess .
    (nxt ~~ nil and:[ nxt ~~ aGsProcess]) ifTrue:[
      self _switchFrom: nil to: nxt .       "exits critical"
      self _uncontinuableError . "should not reach here"
    ].
  ] ifFalse: [
    aGsProcess _unscheduleProcess .
  ].
  self _exitCritical .
]

{ #category : 'Delay Access' }
ProcessorScheduler >> _timeoutNow: delayQsize [

  "Returns current time in milliseconds consistent with most recently
   expired timeout.  If delayQsize == 0 , result may always be zero "
<primitive: 735>
^ self _primitiveFailed: #_timeoutNow:

]

{ #category : 'Private' }
ProcessorScheduler >> _unschedule: aGsProcess [

  "remove the given process from any scheduler state it is in."
  | q |
  q := aGsProcess onQueue .
  q ifNotNil:[
    aGsProcess _onQueue: nil .
    q == readyQueue ifTrue:[
      q removeIdentical: aGsProcess otherwise: nil.
    ] ifFalse:[
      self _remove: aGsProcess fromSet: q . "suspendedSet or waitingSet"
    ].
  ] ifNil:[
    self _remove: aGsProcess fromSet: waitingSet . 
  ]

]

{ #category : 'Private' }
ProcessorScheduler >> _updateTimerThread: timeMs priority: anInt [

"Restart timer thread so it will wakeup at the specified signal time.
 aSignalTime == -1 means  timer thread should wait forever for more work."

<primitive: 738>
self _primitiveFailed: #_updateTimerThread: args: { timeMs . anInt }

]

{ #category : 'Private' }
ProcessorScheduler >> _waitForMilliseconds: intervalMs [
  "Causes the current thread to suspend for the specified time .
   Returns true if the timeout expired. Returns false if the thread
   woke up early for some reason."
  "Sender has done _enterCritical "
  intervalMs _isSmallInteger ifFalse:[
    intervalMs _validateClass: SmallInteger.
  ].
  intervalMs <= 0 ifTrue:[
    self yield .
    ^ false
  ] ifFalse:[ | nowMs endMs aProc proc res wasInCrit "trace" |
    "trace := true."
    nowMs := self _now .
    endMs := 16rfffffffffffffff "max SmallInteger" .
    (endMs - intervalMs) > nowMs ifTrue:[
       endMs := nowMs + intervalMs
    ].
    (proc := self activeProcess) _signalTime: endMs  .
    self _addDelay: proc to: delayQueue .
    self _reschedule .
    wasInCrit := self _enterCritical .
    wasInCrit == 0 ifFalse:[ UncontinuableError signal:'scheduler already in critical']. 

    "we have woken up."
    aProc := self activeProcess .
    aProc ~~ proc ifTrue:[
      self _exitCritical .
      Error signal:'Unexpected: activeProcess changed during _waitForMilliseconds:'.
    ].
    res := true .
    aProc _signalTime ifNotNil:[
      "Woke up without the delay having been activated, so clean it up."
      delayQueue removeIdentical: aProc otherwise: nil . "inline _delayUnschedule:"
      aProc  _signalTime: nil.
       res := false
    ].
    self _exitCritical .
    ^ res .
  ].

]

{ #category : 'Private' }
ProcessorScheduler >> _waitForSocket: aGsSocket [
  "Causes the current thread to suspend until aGsSocket is read/writable,
   or has an error.  Timeout is infinite.  Returns true if socket is ready."

  | proc ws |
  (proc := self activeProcess) _waitingOn: aGsSocket .  "clear done by _reapSignal:"
  "assertion code"
  "true ifTrue:[
    (socketsPolled includesIdentical: aGsSocket) ifFalse:[ self pause ].
  ]."

  self _add: proc toSet: (ws := waitingSet) .
  proc _onQueue: ws .

  self _reschedule .

]

{ #category : 'Private' }
ProcessorScheduler >> _waitForSocket: aGsSocket timeout: msToWait forWrite: writeBool [

  "Causes the current thread to suspend until aGsSocket is read/writable,
   has an error, or the specified timeout occurs.
   msToWait must be > 0 . Returns receiver.
   Caller must have done _enableWrite or _enableRead for some GsSocket,
   and added current process to the socket's waiters list.
   writeBool should be nil or true .
   Caller has sent _enterCritical .
  "

  | proc aProc wasInCrit |
  (proc := self activeProcess) _waitingOn: aGsSocket .  "clear done by _reapSignal:"
  "assertion code"
  "true ifTrue:[
    (socketsPolled includesIdentical: aGsSocket) ifFalse:[ self pause ].
  ]."

  proc _signalTime: (self _now) + msToWait.
  self _addDelay: proc to: delayQueue .

  self _reschedule .
  wasInCrit := self _enterCritical .
  wasInCrit == 0 ifFalse:[ UncontinuableError signal:'scheduler already in critical']. 
  aProc := self activeProcess.
  aProc ~~ proc ifTrue:[
    self _exitCritical .
    Error signal:'Unexpected: activeProcess changed during _waitForSocket:timeout:'.
  ].
  proc _signalTime ifNotNil: [
    delayQueue removeIdentical: proc otherwise: nil . "inline _delayUnschedule:"
    proc _signalTime: nil.
    proc _waitingOn: nil .
    writeBool ifNotNil:[ aGsSocket _cancelWriteEventFor: proc ]
		 ifNil:[ aGsSocket _cancelReadEventFor: proc ].
  ].
  self _exitCritical .

]

{ #category : 'Accessing' }
ProcessorScheduler >> _waitingProcessesInto: anIdentitySet [

  "Add all of the instances of GsProcess that
   are waiting for an event to anIdentitySet and return anIdentitySet"

  | que |
  anIdentitySet _addAll: waitingSet forReplay: false .
  (que := { } ) addAll: delayQueue .
  1 to: que size do: [ :idx |
     (que atOrNil: idx) ifNotNil:[:p |
        p _targetProcess ifNotNil:[:t| 
          t _isArray ifTrue:[ anIdentitySet addAll: t ]
                 ifFalse:[ anIdentitySet add: t ].
        ]
     ]
  ].
  ^ anIdentitySet
]

{ #category : 'Accessing' }
ProcessorScheduler >> _waitingProcessesInto: aCollection inGroup: groupId [

  "Add all of the instances of GsProcess that
   are waiting for an event and are in specified group
   to aCollection and return aCollection"

  | proc set |
  1 to: (set := waitingSet) size do:[:n |
    proc := set _at: n .
    proc _groupOrNil == groupId ifTrue:[ aCollection add: proc ].
  ].
  set := delayQueue .
  1 to: set size do: [ :n |
    proc := (set at: n) _targetProcess .
    proc ifNotNil:[
      proc _isArray ifTrue:[ 
        proc do:[:aProc | proc _groupOrNil == groupId ifTrue:[ aCollection add: aProc ]].
      ] ifFalse:[
        proc _groupOrNil == groupId ifTrue:[ aCollection add: proc ].
      ]
    ].
  ].
  ^ aCollection
]

{ #category : 'Private' }
ProcessorScheduler >> _waitOnSema: aSemaphore [

  | proc ws |
  (proc := self activeProcess) _waitingOn: aSemaphore .
  self _add: proc toSet: (ws := waitingSet) .
  proc _onQueue: ws .

  self _reschedule .

]

{ #category : 'Process State Change' }
ProcessorScheduler >> _yieldForTimeout [
  "schedule a different active process, if any are
   available at a higher priority, and put the current process in the
   appropriate ready queue."

  | readyQ wasInCrit |
  wasInCrit := self _enterCritical .
  self _reapEvents: 1 .
  ((readyQ := readyQueue) atOrNil: 1) ifNotNil:[:readyProc |
    readyProc priority > self activePriority  ifTrue: [ | oldProc newProc |
      oldProc := self activeProcess .

      newProc := readyQ _removeAtOrNil: 1 . 
      newProc _onQueue: nil .

      self _addReadyProcess: oldProc to: readyQ .  "active process never on ready que"
      oldProc _onQueue: readyQ .

      "tracing code [
        GsFile gciLogServer:'yield old clientData=' , oldProc _clientData asString ,
		' new clientData=', newProc _clientData asString .
      ]."

      self _enableEvents .
      self _switchFrom: oldProc to: newProc .   "prim does _exitCritical, updates activeProcess iv"
      ^ self
    ].
  ].
  wasInCrit == 0 ifTrue:[ self _exitCritical] .
]

{ #category : 'Accessing' }
ProcessorScheduler >> activePriority [
  "return the priority of the active process"
  | ap |
  ^ (ap := activeProcess) ifNil:  [ 15 "inline userSchedulingPriority" ]
                          ifNotNil: [ ap priority ]

]

{ #category : 'Accessing' }
ProcessorScheduler >> activeProcess [

"Returns the currently active process.
 Creates a new GsProcess if needed and stores it into the
  activeProcess instVar of the scheduler instance."
<primitive: 693>
self _primitiveFailed: #_currentProcess

]

{ #category : 'Accessing' }
ProcessorScheduler >> allProcesses [
  "Return an IdentitySet of all of the instances of GsProcess
   that are not terminated."

  ^ self _allProcessesInto: (IdentitySet new: 4)

]

{ #category : 'Event Scheduling' }
ProcessorScheduler >> cancelWhenReadable: aGsSocket signal: objToNotify [
  "Cancels the event specified by aGsSocket and objToNotify.
   The event is created using the whenReadable:signal: method.
   Returns true if the event has already been cancelled, reaped, or does not exist.
   Returns false if the event is cancelled by this invocation."

  ^ aGsSocket _cancelReadEventFor: objToNotify .

]

{ #category : 'Event Scheduling' }
ProcessorScheduler >> cancelWhenWritable: aGsSocket signal: objToNotify [
  "Cancels the event specified by aGsSocket and objToNotify.
   The event is created using the whenWritable:signal: method.
   Returns true if the event has already been cancelled, reaped, or does not exist.
   Returns false if the event is cancelled by this invocation."

  ^ aGsSocket _cancelWriteEventFor: objToNotify .

]

{ #category : 'Logging' }
ProcessorScheduler >> dbglog: aString [
  "log a debugging string"
  GsFile gciLogServer: aString

]

{ #category : 'Priorities' }
ProcessorScheduler >> highestPriority [
  "return the highest allowable priority"
  ^ 40   "also in C code as PROCESSSCHEDULER_highestPriority"
         "also other in-line highestPriority logic"

]

{ #category : 'Priorities' }
ProcessorScheduler >> highIOPriority [
  "Answer the priority at which the most time
   critical input/output processes should run. [Blue Book]"
  ^30

]

{ #category : 'Priorities' }
ProcessorScheduler >> lowestPriority [
  "return the lowest allowable priority"
  ^1   "also in C code as PROCESSSCHEDULER_lowestPriority"
       "also other inline lowestPriority logic"
]

{ #category : 'Priorities' }
ProcessorScheduler >> lowIOPriority [
  "Answer the priority at which most input/output
   processes should run. [Blue Book]"
  ^25

]

{ #category : 'Accessing' }
ProcessorScheduler >> readyProcesses [
  "Return all of the instances of GsProcess that are ready to run but
   are not active."
  | s wasInCrit |
  wasInCrit := self _enterCritical .
  (s := IdentitySet new: 4) _addAll: readyQueue  forReplay: false .
  wasInCrit == 0 ifTrue:[ self _exitCritical] .
  ^ s

]

{ #category : 'Accessing' }
ProcessorScheduler >> suspendedProcesses [
  "Return all of the instances of GsProcess that were suspended using
   the GsProcess>>suspend method
   or created with the ExecBlock>>newProcess method and not yet
   sent resume."

  ^ suspendedSet shallowCopy

]

{ #category : 'Priorities' }
ProcessorScheduler >> systemBackgroundPriority [
  "Answer the priority at which system background
   processes should run. [Blue Book]"
  ^5

]

{ #category : 'Priorities' }
ProcessorScheduler >> timingPriority [
  "Answer the priority at which the system processes
   keeping track of real time should run. [Blue Book]"
  ^35

]

{ #category : 'Priorities' }
ProcessorScheduler >> userBackgroundPriority [
  "Answer the priority at which background
   processes created by the user should run. [Blue Book]"
  ^10

]

{ #category : 'Priorities' }
ProcessorScheduler >> userInterruptPriority [
  "Answer the priority at which processes created
   by the user and desiring immediate service
   should run. [Blue Book]"
  ^20

]

{ #category : 'Priorities' }
ProcessorScheduler >> userSchedulingPriority [
  "Answer the priority at which the user inter-
   face processes should run. [Blue Book]"
  ^15  "this value inlined in some methods here and in GsProcess, and
	in C as PROCESSSCHEDULER_userPriority"

]

{ #category : 'Accessing' }
ProcessorScheduler >> waitingProcesses [
  "Return an IdentitySet of all of the instances of GsProcess that
   are waiting for an event."
  | res wasInCrit |
  wasInCrit := self _enterCritical .
  res := self _waitingProcessesInto: (IdentitySet new: 4) .
  wasInCrit == 0 ifTrue:[ self _exitCritical] .
  ^ res

]

{ #category : 'Event Scheduling' }
ProcessorScheduler >> whenReadable: aGsSocket signal: objToNotify [
  "Causes the receiver to signal objToNotify once when aGsSocket is
   readable or has an exceptional condition.
   Once the receiver signals the object for this event the event is
   canceled so you need to call this method again to reenable the event.
   objToNotify is signalled by sending it the message '_reapSignal: source'
   with source being the socket that is readable.
   The kernel classes that respond to _reapSignal: are:
     Semaphore, SharedQueue, and GsProcess.
   Returns self."

  aGsSocket _whenReadableNotify: objToNotify .
  "true ifTrue:[  assertion code
    (socketsPolled includesIdentical: aGsSocket) ifFalse:[ self pause ].
  ]."

]

{ #category : 'Event Scheduling' }
ProcessorScheduler >> whenWritable: aGsSocket signal: objToNotify [
  "Causes the receiver to signal objToNotify once when aGsSocket is
   writable or has an exceptional condition.
   Once the receiver signals the object for this event the event is
   canceled so you need to call this method again to reenable the event.
   objToNotify is signalled by sending it the message '_reapSignal: source'
   with source being the socket that is writable.
   The kernel classes that respond to _reapSignal: are:
     Semaphore, SharedQueue, and GsProcess.
   Returns self."

  aGsSocket _whenWritableNotify: objToNotify .
  "true ifTrue:[ assertion code
    (socketsPolled includesIdentical: aGsSocket) ifFalse:[ self pause ].
  ]."

]

{ #category : 'Process State Change' }
ProcessorScheduler >> yield [
  "schedule a different active process, if any are
   available, and put the current process in the
   appropriate ready queue."

  | readyQ wasInCrit |
  wasInCrit := self _enterCritical .
  self _reapEvents: 1 .
  ((readyQ := readyQueue) atOrNil: 1) ifNotNil:[:readyProc |
    readyProc priority >= self activePriority  ifTrue: [ | oldProc newProc |
      oldProc := self activeProcess .

      newProc := readyQ _removeAtOrNil:1 .
      newProc _onQueue: nil .

      self _addReadyProcess: oldProc to: readyQ .  "active process never on ready que"
      oldProc _onQueue: readyQ .

      "tracing code [
        GsFile gciLogServer:'yield old clientData=' , oldProc _clientData asString ,
		' new clientData=', newProc _clientData asString .
      ]."

      self _switchFrom: oldProc to: newProc .   "prim does _exitCritical, updates activeProcess iv"

      ^ self
    ].
  ].
  wasInCrit == 0 ifTrue:[ self _exitCritical] .
]
