Extension { #name : 'MultiByteString' }

{ #category : 'Formatting' }
MultiByteString class >> charSize [

"Returns number of bytes that make up a character for this string class."

^ self subclassResponsibility: #charSize

]

{ #category : 'Storing and Loading' }
MultiByteString class >> loadFrom: passiveObj [

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into a new instance of the receiver.
 Returns the new instance."

| inst |

inst := String new.  "yes, make a String !!"
inst _loadFrom: passiveObj size: passiveObj readSize .
inst _changeClassToMultiByte: self .  "Now change to double/quad byte"
inst class == self ifFalse:[ Error signal:'wrong class from loadFrom: , ', inst class name].
passiveObj hasRead: inst .
^inst

]

{ #category : 'Instance Creation' }
MultiByteString class >> new [

^ self subclassResponsibility: #new

]

{ #category : 'Instance Creation' }
MultiByteString class >> new: aSize [

^ self subclassResponsibility: #new:

]

{ #category : 'Decompiling without Sources' }
MultiByteString >> _asSource [

^ self quoted

]

{ #category : 'Private' }
MultiByteString >> _asString [

"Allowed codePoints are 0..255 .
 Returns a (single byte) String representation of the receiver if all of the
 Characters in the receiver have codePoints in allowed range .
 Returns nil if any of the Characters in the receiver cannot be represented as
 single-byte Characters in allowed code point range."

<primitive: 229>

^ self _primitiveFailed: #_asString .

]

{ #category : 'Converting' }
MultiByteString >> _asUnicode16 [

"Return an instance of Unicode7 or Unicode16 using the
minimum bytes per character required to represent the receiver.
Return nil if the receiver is not representable as Unicode7 nor Unicode16."

<primitive: 926>

self _primitiveFailed:#_asUnicode16

]

{ #category : 'Converting' }
MultiByteString >> _asUnicode7 [

<primitive: 925>
"Return an instance of Unicode7 if receiver can be
represented as an Unicode7 string, else return nil."

self _primitiveFailed:#_asUnicode7

]

{ #category : 'Compatibility' }
MultiByteString >> _at: anIndex equals: aCharCollection ignoreCase: aBoolean [

"Returns true if aCharCollection is contained in the receiver, starting at
 anIndex.  Returns false otherwise.  The comparison is case-insensitive."

^ aBoolean ifTrue:[ self at: anIndex equalsNoCase: aCharCollection ]
	  ifFalse:[ self at: anIndex equals: aCharCollection ]

]

{ #category : 'Indexing Support' }
MultiByteString >> _changingSizeTo: newSize [

"Notifies any modification tracking objects that the receiver (an indexable
 object) is having its size changed."

"the object manager reports size changes in number of bytes; convert this
 to number of characters."

^ super _changingSizeTo: newSize // (self charSize)

]

{ #category : 'Execution' }
MultiByteString >> _compileInContext: anObject symbolList: aSymbolList [

"Compiles the receiver as an instance method for the class of
 anObject, using aSymbolList as the symbol list, and using the
 default compilation environment.
 The receiver is parsed as a doit, i.e. it does not have a selector.

 Returns a GsMethod that is not in the method dictionary of any class, or
 else generates an error if there are compilation errors."

^ self _compileInContext: anObject symbolList: aSymbolList oldLitVars: nil
      environmentId: 0 flags: 0

]

{ #category : 'Execution' }
MultiByteString >> _compileInContext: anObject symbolList: aSymbolList
oldLitVars: litVarArray  environmentId: anEnvironmentId flags: flagsInt [

"Compiles the receiver as an instance method for the class of
 anObject, using aSymbolList as the symbol list.
 The receiver is parsed as a doit, i.e. it does not have a selector.

 If litVarArray is not nil, it must be an Array of Symbol,SymbolAssociation pairs
 and this Array will be searched prior to searching aSymbolList to
 resolve literal variables within the method.

 anEnvironmentId must be a SmallInteger >= 0 and <= 255
 specifying a compilation environment.

 flagsInt must be a SmallInteger >= 0 and <= 255 and contains the bits
 specified by GCI_COMPILE_* in $GEMSTONE/include/gcicmn.ht .

 Returns a GsMethod that is not in the method dictionary of any class, or
 else generates an error if there are compilation errors."

<primitive: 150>
aSymbolList _validateClass: SymbolList .
^ self _primitiveFailed: #_compileInContext:symbolList:oldLitVars:environmentId:flags:
       args: { anObject . aSymbolList . litVarArray . anEnvironmentId . flagsInt}

]

{ #category : 'Private' }
MultiByteString >> _findLastString: subString startingAt: startIndex ignoreCase: aBoolean [

"Searches backwards through receiver    starting at startIndex  .
 Returns the index at which subString begins.  If the
 receiver does not contain subString, this returns 0."

<primitive: 451>

startIndex _validateClass: SmallInteger.
(startIndex < 1) | (startIndex > self size)
  ifTrue: [ ^ self _error: #objErrBadOffsetIncomplete args: { startIndex } ].
subString _validateClasses:{ String }.
aBoolean _validateClass: Boolean .
^ self _primitiveFailed: #_findLastString:startingAt:ignoreCase:
       args: { subString . startIndex . aBoolean }

]

{ #category : 'Private' }
MultiByteString >> _findString: subString startingAt: startIndex ignoreCase: aBoolean [

"If a receiver contains subString beginning at some point at or after
 startIndex, this returns the index at which subString begins.  If the
 receiver does not contain subString, this returns 0."

<primitive: 30>
startIndex _validateClass: SmallInteger.
(startIndex < 1)
  ifTrue: [ ^ self _error: #objErrBadOffsetIncomplete args: { startIndex } ].
subString _validateClasses:{ String }.
aBoolean _validateClass: Boolean .
^ self _primitiveFailed: #_findString:startingAt:ignoreCase:
       args: { subString . startIndex . aBoolean }

]

{ #category : 'New Indexing Comparison - prims' }
MultiByteString >> _idxPrimCompareEqualTo: aCharCollection [
  "This comparison operation is used for the indexing subsystem to
 determine an ordering for insertion into indexing objects.

 This method collates letters AaBb..Zz."

  "The comparison should be compatible with the case-insensitive semantics
 of the String method with selector #= .
 Same primitive as String>>lessThan: "

  <primitive: 981>
  aCharCollection isUnicodeString
    ifTrue: [
      "Unicode"
      ArgumentError signal:'Unicode argument disallowed in String comparison' ]
    ifFalse: [
      aCharCollection ifNil:[ ^ false ].
      aCharCollection _validateClass: CharacterCollection.
      "JapaneseString or MixedString"
      ^ self = aCharCollection ]

]

{ #category : 'New Indexing Comparison - prims' }
MultiByteString >> _idxPrimCompareGreaterThan: aCharCollection [

"This comparison operation is used for the indexing subsystem to
 determine an ordering for insertion into indexing objects such as B-trees.

 This method collates letters AaBb..Zz."

"The comparison should be compatible with the case-insensitive semantics
 of the String method with selector #> .
 Same primitive as String>>greaterThan: ."

<primitive: 443>
aCharCollection _validateClass: CharacterCollection.
^ aCharCollection isUnicodeString
    ifTrue: [ ArgumentError signal:'Unicode argument disallowed in String comparison' ]
    ifFalse: [
      "JapaneseString or MixedString"
      self > aCharCollection ]

]

{ #category : 'New Indexing Comparison - prims' }
MultiByteString >> _idxPrimCompareLessThan: aCharCollection [

"This comparison operation is used for the indexing subsystem to
 determine an ordering for insertion into indexing objects.

 This method collates letters AaBb..Zz."

"The comparison should be compatible with the case-insensitive semantics
 of the String method with selector #< .
 Same primitive as String>>lessThan: "

<primitive: 441>

aCharCollection _validateClass: CharacterCollection.
^ aCharCollection isUnicodeString
    ifTrue: [ ArgumentError signal:'Unicode argument disallowed in String comparison' ]
    ifFalse: [
      "JapaneseString or MixedString"
      self < aCharCollection ]

]

{ #category : 'Deprecated' }
MultiByteString >> _primEquals: aString collatingTable: aTable [

"Deprecated. Returns true if the receiver collates the same as the argument.
 See lessThan:collatingTable: for description of aTable.
 The collating sequence is defined by aTable."

<primitive: 446 >
aTable _validateSameCharSizeAs: self .
aString _validateClasses:{ String }.
self _primitiveFailed: #equals:collatingTable: args: { aString . aTable }

]

{ #category : 'Deprecated' }
MultiByteString >> _primGreaterThan: aString collatingTable: aTable [

"Deprecated. Returns true if the receiver collates after the argument.
 See lessThan:collatingTable: for description of aTable."

<primitive: 433 >
aTable _validateSameCharSizeAs: self .
aString _validateClasses:{ String }.
self _primitiveFailed: #greaterThan:collatingTable: args: { aString . aTable }

]

{ #category : 'Deprecated' }
MultiByteString >> _primLessThan: aString collatingTable: aTable [

"Deprecated. Returns true if the receiver collates before the argument.
 The collating sequence is defined by aTable which must be
 a string with the same character size as the receiver."

<primitive: 359 >
aTable _validateSameCharSizeAs: self .
aString _validateClasses:{ String }.
self _primitiveFailed: #lessThan:collatingTable: args: { aString . aTable }

]

{ #category : 'Copying' }
MultiByteString >> _reverseFrom: aString [

"copies elements of aString into receiver in reverse order.
 Size of aString must be same as size of receiver.
 Returns receiver.
 aString==self is allowed. "

<primitive: 793>
aString stringCharSize == self stringCharSize ifFalse:[
  aString _error: #rtErrBadArgKind args:{ self class }.
].
aString size == self size ifFalse:[
  aString _error: #rtErrBadSize args:{ self size . aString size }.
].
self _primitiveFailed: #_reverseFrom args: { aString }

]

{ #category : 'Private' }
MultiByteString >> _unicodeCompareEnabled [
  "Returns a Boolean, value of
    (Globals at:#StringConfiguration) == Unicode16 
   as of login.
   optimization to avoid dictionary lookup"
 <primitive: 251>
 self _primitiveFailed: #_unicodeCompareEnabled

]

{ #category : 'Private' }
MultiByteString >> _validateSameCharSizeAs: aString [

"Returns nil if self has same character size as aString,
 otherwise generates an error."

| cSize cls |
self stringCharSize ~~ (cSize := aString stringCharSize) ifTrue:[
  cSize == 1 ifTrue:[ cls := String ].
  cSize == 2 ifTrue:[ cls := DoubleByteString ].
  cSize == 4 ifTrue:[ cls := QuadByteString ].
  self _errorExpectedClass: cls
].
^ nil

]

{ #category : 'Case-Insensitive Comparisons' }
MultiByteString >> < aCharCollection [

"Returns true if the receiver collates before the argument.
 Returns false otherwise.

 The comparison is case-insensitive.
 Signals an Error if argument is a kind of Unicode7, Unicode16 or Unicode32 .

 The ASCII letters A and a collate the same.
   'A' < 'a' returns false.
   'A' < 'b' returns true. "

<primitive: 440>
(aCharCollection _stringCharSize bitAnd: 16r7) ~~ 0 ifTrue:[
  ^ aCharCollection > self
].
aCharCollection _validateClass: CharacterCollection .
^ aCharCollection > self

]

{ #category : 'Case-Insensitive Comparisons' }
MultiByteString >> <= aCharCollection [

"Returns true if the receiver collates before the argument or if all of the
 corresponding Characters in the receiver and argument are equal.
 Returns false otherwise.

 The comparison is consistent with that defined for the < method."

^(self > aCharCollection) == false

]

{ #category : 'Case-Sensitive Comparisons' }
MultiByteString >> = aCharCollection [

"Returns true if corresponding Characters in the receiver and argument are equal
 and aCharCollection is comparable with the receiver, and aCharCollection is
 not a kind of Symbol.  Returns false otherwise.

 The comparison for 'equal' is case-sensitive .
 Signals an Error if argument is a kind of Unicode7, Unicode16 or Unicode32 .

 Note that 'kind of Symbol' means either an instance of Symbol or instance of
 MultiByteSymbol."

<primitive: 449>
| argInfo |
argInfo := aCharCollection _stringCharSize .
(argInfo bitAnd: 16r8) ~~ 0 ifTrue:[  "arg is Unicode"
  ^ (self compareTo: aCharCollection collator: nil useMinSize: false) == 0
].
(aCharCollection isKindOf: Utf8) ifTrue:[
  ^ (self compareTo: aCharCollection collator: nil useMinSize: false) == 0
].
(aCharCollection isKindOf: CharacterCollection) ifTrue:[ ^ aCharCollection = self].
^ false

]

{ #category : 'Case-Insensitive Comparisons' }
MultiByteString >> > aCharCollection [

"Returns true if the receiver collates after the argument.  Returns false
 otherwise.

 The comparison is case-insensitive.
 Signals an Error if argument is a kind of Unicode7, Unicode16 or Unicode32."

<primitive: 442>

aCharCollection _validateClass: CharacterCollection.
^ aCharCollection < self

]

{ #category : 'Case-Insensitive Comparisons' }
MultiByteString >> >= aCharCollection [

"Returns true if the receiver collates after the argument or if all of the
 corresponding Characters in the receiver and argument are equal.
 Returns false otherwise.

 The comparison is consistent with that defined for the < method."

^ (self < aCharCollection) == false

]

{ #category : 'Converting' }
MultiByteString >> asByteArray [

	^ByteArray fromString: self

]

{ #category : 'Deprecated' }
MultiByteString >> asciiLessThan: aString [

^ self subclassResponsibility: #asciiLessThan:

]

{ #category : 'Deprecated' }
MultiByteString >> asDoubleByteSymbol [

"Returns a Symbol, DoubleByteSymbol or QuadByteSymbol
    representation of the receiver.
 Deprecated."

self deprecated:'MultiByteString >> asDoubleByteSymbol is deprecated, use asSymbol' .
^ self asSymbol .

]

{ #category : 'Converting' }
MultiByteString >> asLowercase [

"Returns a new instance of the receiver's class, with all upper-case Characters
 in the receiver changed to lower-case.
 If receiver is a DoubleByteSymbol, returns a DoubleByteString.
 If receiver is a QuadByteSymbol, returns a QuadByteString.

 Lower-case Characters are defined by the Unicode standard and
 u_tolower from libicu is called for each character in the receiver."

<primitive: 821>
self _primitiveFailed: #asLowercase .
self _uncontinuableError

]

{ #category : 'Converting' }
MultiByteString >> asLowercaseOld [

"Returns a new instance of the receiver's class, with all upper-case Characters
 in the receiver changed to lower-case.
 If receiver is a DoubleByteSymbol, returns a DoubleByteString.
 If receiver is a QuadByteSymbol, returns a QuadByteString.

 Lower-case Characters are the a to z, including letters with diacriticals, 
 up to codePoint 255. Conversion of codepoints above 255 , requires that
 (deprecated)character data tables be installed."

<primitive: 984>
self _primitiveFailed: #asLowercaseOld .
self _uncontinuableError

]

{ #category : 'Converting' }
MultiByteString >> asString [

"Returns a (single byte) String representation of the receiver if all of the
 Characters in the receiver can be represented as single-byte Characters.
 Otherwise, returns the receiver."

| result |
(result := self _asString) ifNil:[ ^ self ].
^ result

]

{ #category : 'Converting' }
MultiByteString >> asSymbol [

"Returns a canonical Symbol representation of the receiver."

^ Symbol withAll: self .

]

{ #category : 'Converting' }
MultiByteString >> asSymbolKind [

"Returns a (single byte) Symbol representation of the receiver."

^ Symbol withAll: self

]

{ #category : 'Converting' }
MultiByteString >> asUnicodeString [

"If receiver is an Unicode16 or Unicode32, return the receiver.
 Otherwise return an instance of Unicode7 , Unicode16 or Unicode32 ,
 using the class with smallest character size needed to
 represent a copy of the receiver.
 Raises an ArgumentError if receiver contains a codePoint > 16r10FFFF,
 or a codePoint in the range 16rD800 to 16rDFFF "

<primitive: 927>

self _primitiveFailed:#asUnicodeString

]

{ #category : 'Converting' }
MultiByteString >> asUppercase [

"Returns a new instance of the receiver's class, with all lower-case Characters
 in the receiver changed to upper-case.
 If receiver is a DoubleByteSymbol, returns a DoubleByteString.
 If receiver is a QuadByteSymbol, returns a QuadByteString.

 Upper-case Characters are defined by the Unicode standard and
 u_toupper from libicu is called for each character in the receiver."

<primitive: 455>
self _primitiveFailed: #asUppercase .
self _uncontinuableError

]

{ #category : 'Converting' }
MultiByteString >> asUppercaseOld [

"Returns a new instance of the receiver's class, with all lower-case Characters
 in the receiver changed to upper-case.
 If receiver is a DoubleByteSymbol, returns a DoubleByteString.
 If receiver is a QuadByteSymbol, returns a QuadByteString.

 Upper-case Characters are the A to Z, including letters with diacriticals, 
 up to codePoint 255.
 Conversion of codepoints above 255 , requires that (deprecated)character data 
 tables be installed."

<primitive: 983>
self _primitiveFailed: #asUppercaseOld .
self _uncontinuableError

]

{ #category : 'Accessing' }
MultiByteString >> at: anIndex [

"Returns the Character at anIndex."

^ self subclassResponsibility: #at:

]

{ #category : 'Case-Sensitive Comparisons' }
MultiByteString >> at: anIndex equals: aCharCollection [

"Returns true if aCharCollection is contained in the receiver, starting at
 anIndex.  Returns false otherwise.  The comparison is case-sensitive."

<primitive: 668>
anIndex _validateClass: SmallInteger.
((anIndex <= 0) or: [(anIndex > self size)])
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex ].
aCharCollection _validateClass: CharacterCollection.
^ self _primitiveFailed: #at:equals: args: { anIndex . aCharCollection }

]

{ #category : 'Case-Insensitive Comparisons' }
MultiByteString >> at: anIndex equalsNoCase: aCharCollection [

"Returns true if aCharCollection is contained in the receiver, starting at
 anIndex.  Returns false otherwise.  The comparison is case-insensitive."

<primitive: 438>
anIndex _validateClass: SmallInteger.
((anIndex <= 0) or: [(anIndex > self size)])
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex ].
aCharCollection _validateClass: CharacterCollection.
^ self _primitiveFailed: #at:equalsNoCase: args: { anIndex . aCharCollection }

]

{ #category : 'Updating' }
MultiByteString >> at: anIndex put: aChar [

") Stores aChar at the specified location and returns aChar."

^ self subclassResponsibility: #at:put:

]

{ #category : 'Formatting' }
MultiByteString >> charSize [

"Returns number of bytes that make up a character for this string class."

^ self subclassResponsibility: #charSize

]

{ #category : 'Comparing' }
MultiByteString >> codePointCompareTo: aString [

"aString must be a CharacterCollection or ByteArray,
 and must not be a Utf8 or Utf16 .
 Return -1 , 0, or 1  based on how self compares to aString
 using codePoint comparison of each character. "

<primitive: 844>
^ self _primitiveFailed: #codePointCompareTo: args: { aString }

]

{ #category : 'Compression' }
MultiByteString >> compressWithLz4: opCode [

"Compress the receiver with lz4 compression and answer a ByteArray.
 There are 2 lz4 compression modes available:
   opCode 0 - frame mode
   opCode 1 - block mode

 Frame mode should almost always be used. Block mode requires the
 receiver to be completely faulted into memory which can cause out of 
 memory errors for large objects. Frame mode has no such limitation
 as the complete contents of the receiver need not be in memory
 during compression. Also the lz4 command line program compresses
 data using frame mode.

 Symbols, DoubleByteSymbols and QuadByteSymbols may not be compressed.
 Attempting to do so raises an exception.

 For multi-character byte objects, the contents are first converted to 
 little endian format before compression if necessary.

 Returns a ByteArray on success or raises an exception on error."

<primitive: 241>
opCode _validateClass: SmallInteger .
((opCode < 0) or:[ opCode > 1])
  ifTrue:[ opCode _error: #rtErrArgOutOfRange ] .
^ self _primitiveFailed: #compressWithLz4:.

]

{ #category : 'Testing' }
MultiByteString >> containsSeparator [

"Returns true if the receiver contains a separator Character."

1 to: self size do: [:i |
  (self at: i) isSeparator ifTrue: [^true].
].
^false

]

{ #category : 'Copying' }
MultiByteString >> copyFrom: startIndex to: stopIndex [

"Returns a new Array containing the elements of the receiver
 between startIndex and stopIndex, inclusive.  The result is of the same class
 as the receiver.

 If startIndex > stopIndex then an empty collection is returned.
 Otherwise both startIndex and stopIndex must be positive integers not larger than the
 size of the receiver, with startIndex <= stopIndex.
"

<primitive: 818>
(startIndex < 1) ifTrue: [ ^ self _errorIndexOutOfRange: startIndex].

((stopIndex > self size) or: [(stopIndex < 0)])
   ifTrue: [ ^ self _errorIndexOutOfRange: stopIndex].

self _primitiveFailed: #copyFrom:to: args: { startIndex . stopIndex }

]

{ #category : 'Encoding' }
MultiByteString >> encodeUsing: encodingArray [

  "A new instance of the class of the receiver will be returned where
   each character in the receiver is replaced by the character or string
   from encodingArray that is located at the position in encodingArray for
   the character.  The codePoint of the character is used as an index
   into the encodingArray,  value 16r0 accesses   encodingArray at:1 .
   An Error is signaled if any of the following are true
    the codePoint is >= size of the encodingArray

    the value from the encodingArray is a Character with codePoint too large
    to fit in an instance of the receiver's class.

    the value from the encodingArray is a CharacterCollection with
    bytes-per-character bigger than bytes-per-character of the receiver

    the value from the encodingArray is neither a String, MultiByteString,
    nor Character . "
  <primitive: 654>
  encodingArray _validateInstanceOf: Array  .
  self _primitiveFailed: #encodeUsing: args: { encodingArray }

]

{ #category : 'Deprecated' }
MultiByteString >> equals: aString collatingTable: aTable [

"Deprecated. Returns true if the receiver collates the same as the argument.
 See lessThan:collatingTable: for description of aTable.
 The collating sequence is defined by aTable."

self deprecated: 'MultiByteString>>equals:collatingTable: deprecated in 3.2'.
^ self _primEquals: aString collatingTable: aTable

]

{ #category : 'Case-Insensitive Comparisons' }
MultiByteString >> equalsNoCase: aCharCollection [

"Returns true if corresponding Characters in the receiver and argument are equal
 and aCharCollection is comparable with the receiver, and aCharCollection is
 not a kind of Symbol.  Returns false otherwise.

 The comparison for equal is case-insensitive.

 Note that 'kind of Symbol' means either an instance of Symbol or instance of
 DoubleByteSymbol."

<primitive: 450>
aCharCollection _stringCharSize ~~ 0 ifTrue:[
  self _primitiveFailed: #equalsNoCase:  args: { aCharCollection } .
  ^ self _uncontinuableError
].
(aCharCollection isKindOf: CharacterCollection) ifTrue: [
  ^ self equalsNoCase: (String withAll: aCharCollection)
] ifFalse:[
  ^ false
].

]

{ #category : 'Execution' }
MultiByteString >> evaluate [

"Compiles the receiver as an unbound method and executes it using the current
 default symbol list."

^ (self _compileInContext: nil
       symbolList: GsSession currentSession symbolList)
  _executeInContext: nil

]

{ #category : 'Execution' }
MultiByteString >> evaluateInContext: anObject [
  ^ self evaluateInContext: anObject symbolList: GsSession currentSession symbolList
]

{ #category : 'Execution' }
MultiByteString >> evaluateInContext: anObject symbolList: aSymbolList [

"Compiles the receiver as an instance method for the class of anObject, using
 aSymbolList as the symbol list.  Executes the resulting GsMethod using anObject
 as self and returns the result of the execution.  Generates an error if
 compilation errors occur."

^ (self _compileInContext: anObject symbolList: aSymbolList)
    _executeInContext: anObject

]

{ #category : 'Deprecated' }
MultiByteString >> greaterThan: aString collatingTable: aTable [

"Deprecated. Returns true if the receiver collates after the argument.
 See lessThan:collatingTable: for description of aTable."

self deprecated: 'MultiByteString>>greaterThan:collatingTable: deprecated in 3.2'.
^self _primGreaterThan: aString collatingTable: aTable

]

{ #category : 'Deprecated' }
MultiByteString >> greaterThanOrEqual: aString collatingTable: aTable [

"Deprecated. Returns true if the receiver collates after or the same as the argument.
 The collating sequence is defined by aTable."

self deprecated: 'MultiByteString>>greaterThanOrEqual:collatingTable: deprecated in 3.2'.
^ (self lessThan: aString collatingTable: aTable) not

]

{ #category : 'Hashing' }
MultiByteString >> hash [

"Returns a positive Integer based on a case-sensitive hash of the contents
 of the receiver."

"The algorithm implemented is described in:

 [Pearson 90]
 Pearson, Peter K., Fast Hashing of Variable-Length Text Strings,
 Communications of the ACM 33, 6, (June 1990), 677-680."

<primitive: 282>
self _primitiveFailed: #hash .
self _uncontinuableError

]

{ #category : 'Case-Sensitive Searching' }
MultiByteString >> includes: aCharacter [

"Returns true if the receiver contains aCharacter, false otherwise.
 The search is case-sensitive."

^(aCharacter class == Character) and:
	[self includesValue: aCharacter].

]

{ #category : 'Case-Sensitive Searching' }
MultiByteString >> includesValue: aCharacter [

"Returns true if the receiver contains aCharacter.  The search is
 case-sensitive."

<primitive: 452>

aCharacter _validateClass: Character .
]

{ #category : 'Case-Sensitive Searching' }
MultiByteString >> indexOf: aCharacter startingAt: startIndex [

"Returns the index of the first occurrence of aCharacter in the receiver,
 not preceding startIndex.  If the receiver does not contain aCharacter,
 returns zero.

 The search is case sensitive."

<primitive: 453>
startIndex _validateClass: Integer .
startIndex < 1 ifTrue:[ ^ self _errorIndexOutOfRange: startIndex].
^ self indexOf: aCharacter asCharacter startingAt: startIndex

]

{ #category : 'Case-Insensitive Comparisons' }
MultiByteString >> isEquivalent: aCharCollection [
    "Returns true if the receiver is equivalent to aCharCollection.
    The receiver is equivalent to aCharCollection if the receiver
    contains the same Characters as aCharCollection regardless of case
    or internal representation.  For example, if $a is in
    aCharCollection, it is equivalent to any representation of an 'a'
    in the receiver's character set."

<primitive: 450>
aCharCollection _stringCharSize ~~ 0 ifTrue:[
  self _primitiveFailed: #isEquivalent:  args: { aCharCollection } .
  ^ self _uncontinuableError
].
(aCharCollection isKindOf: CharacterCollection) ifTrue: [
  ^ self isEquivalent: (String withAll: aCharCollection)
] ifFalse:[
  ^ false
].

]

{ #category : 'Encoding' }
MultiByteString >> javaScriptEncode [
  "Returns a String.
   The primitive encapulates all of the encoding logic
   from Seaside JSStream>>encodeString:on: .
   Assumes Character>>greaseInteger always equivalent to Character>>codePoint
  "
<primitive: 1046>
^ self _primitiveFailed: #javaScriptEncode

]

{ #category : 'Deprecated' }
MultiByteString >> lessThan: aString collatingTable: aTable [

"Deprecated. Returns true if the receiver collates before the argument.
 The collating sequence is defined by aTable which must be
 a string with the same character size as the receiver."

self deprecated: 'MultiByteString>>lessThan:collatingTable: deprecated in 3.2'.
^self _primLessThan: aString collatingTable: aTable

]

{ #category : 'Deprecated' }
MultiByteString >> lessThanOrEqual: aString collatingTable: aTable [

"Deprecated. Returns true if the receiver collates before or the same as the argument.
 The collating sequence is defined by aTable."

self deprecated: 'MultiByteString>>lessThanOrEqual:collatingTable: deprecated in 3.2'.
^ (self greaterThan: aString collatingTable: aTable) not

]

{ #category : 'Accessing' }
MultiByteString >> numArgs [

"Returns the number of arguments the receiver would take, if the receiver were
 a message selector."

self isInfix ifTrue: [ ^1 ].
self isKeyword ifTrue: [ ^self identicalOccurrencesOf: $: ].
^0

]

{ #category : 'Formatting' }
MultiByteString >> printOn: aStream [

"Puts a displayable representation of the receiver on the given stream."

"For classes whose instances can be literals, the result should contain
 formatting information.  For example, the following expression should
 evaluate to true:

 #abc asString = String withAll: '#abc'"

 | idx lastIdx sz |
 aStream nextPut: $' .
 idx := self indexOf: $' startingAt: 1 .
 idx == 0 ifTrue:[
   aStream nextPutAll: self .
 ] ifFalse:[
   lastIdx := 1.
   [ (idx == 0) or: [aStream isFull]] whileFalse: [
     aStream nextPutAll: (self copyFrom: lastIdx to: idx) .
     aStream nextPut: $' .
     lastIdx := idx + 1 .
     idx := self indexOf: $' startingAt: lastIdx .
   ].
   sz := self size .
   ((lastIdx <= sz) and: [aStream isFull == false ]) ifTrue: [
     aStream nextPutAll: (self copyFrom: lastIdx to: sz)  .
   ].
 ].
 aStream nextPut: $' .

]

{ #category : 'Removing' }
MultiByteString >> removeFrom: startIndex to: stopIndex [

"Deletes the elements of the receiver from startIndex to stopIndex.  The
 size of the receiver is decreased by stopIndex - startIndex + 1.

 Both startIndex and stopIndex must be positive integers not larger than the
 size of the receiver, with startIndex <= stopIndex."

<primitive: 271>
(stopIndex < startIndex)
ifTrue:
   [ ^ startIndex _error: #rtErrBadCopyFromTo args: { stopIndex }].
((stopIndex > self size) or: [(stopIndex < 1)])
   ifTrue: [ ^ self _errorIndexOutOfRange: stopIndex].
(startIndex < 1)
   ifTrue: [ ^ self _errorIndexOutOfRange: startIndex].
^ self _primitiveFailed: #removeFrom:to: args: { startIndex . stopIndex }

]

{ #category : 'Copying' }
MultiByteString >> reverse [

"Returns a copy of the receiver with its elements in reverse order."
| copy |
copy := self species new: self size .
^ copy _reverseFrom: self

]

{ #category : 'Accessing' }
MultiByteString >> size [

"Returns the number of Characters in the receiver"

^ self subclassResponsibility: #size

]

{ #category : 'Updating' }
MultiByteString >> size: anInteger [


"Changes the size of the receiver to anInteger Characters."

^ self subclassResponsibility: #size:

]

{ #category : 'Encoding' }
MultiByteString >> sizeForEncodeAsUTF8 [

"Returns a SmallInteger, the size of the result of encodeAsUTF8."

<primitive: 520>
self _primitiveFailed: #sizeForEncodeAsUTF8

]

{ #category : 'Accessing' }
MultiByteString >> valueAt: anIndex [

"Returns the value of the (multi-byte) Character at anIndex."

^ (self at: anIndex) codePoint

]

{ #category : 'Copying' }
MultiByteString >> withLFs [

"Returns a copy of the receiver with all back-slashes replaced by line-feeds."

^self copyReplaceAll: '\' with: (self class with: Character lf)

]

{ #category : 'Accessing' }
MultiByteString >> wordAt: anIndex [

"Returns the integer value of the Character at anIndex in the receiver."

^ (self at: anIndex) codePoint

]

{ #category : 'Updating' }
MultiByteString >> wordAt: anIndex put: aValue [

"Stores the integer value aValue in the character cell of the receiver
 specified by anIndex. Return aValue."

self at: anIndex put: (Character withValue: aValue) .
^ aValue

]

{ #category : 'Storing and Loading' }
MultiByteString >> writeTo: passiveObj [

"Converts the receiver to its passive form and writes that information on
 passiveObj."

passiveObj writeClass: self class;
  writeSize: self _basicSize;
    nextPutAllBytes: self; space

]
