Extension { #name : 'Metaclass3' }

{ #category : 'Accessing' }
Metaclass3 >> _classCategory [
  ^ self thisClass _classCategory

]

{ #category : 'Accessing' }
Metaclass3 >> _gbsTraversalCallback [

"Private.  When GemBuilder Smalltalk traverses a metaclass, this method
 is called to return a description of the metaclass."

 self isMeta ifTrue:[ ^ self printString ].
 ^ super _gbsTraversalCallback  "probably MNU"

]

{ #category : 'Class Instance Variables' }
Metaclass3 >> _inheritCIVs: ivnames at: inheritedIVIndex [
	"Adds the instance variables specified in the argument to the receiver at the
 given location.

 Instance variables that are added to a metaclass are called
 Class Instance Variables."

	| names vname subClss tCls numNewIvs |
	self isMeta ifFalse: [^self error: 'receiver is not a metaClass'].
	(numNewIvs := ivnames size) == 0 ifTrue: [^self].
	numNewIvs + self instSize > GEN_MAX_INSTANCE_VARS
		ifTrue: [self error: 'new instVars would exceed max number of instVars'].
	names := Array withAll: instVarNames.
	tCls := self thisClass.
	1 to: numNewIvs
		do:
			[:i |
			vname := (ivnames at: i) asSymbol.
			names insertObject: vname at: inheritedIVIndex + i - 1.
			self _incrementInstVars: 1.
			self _unsafeAt: 4 put: (InvariantArray withAll: names).
			"force the class to increase in size by storing one position off the end"
			self _refreshClassCache: false.
			tCls _insertCivAt: inheritedIVIndex + i - 1].
	tCls _refreshClassCache: false.
	tCls subclassesDisallowed
		ifFalse:
			[subClss := self _subclasses.
			subClss ~~ nil
				ifTrue: [subClss do: [:e | e _inheritCIVs: ivnames at: inheritedIVIndex]]].
	"finally, recompile my methods to ensure that the instance variable indexes are
 correct"
	self _recompileMethodsAfterNewIvOffset: inheritedIVIndex

]

{ #category : 'Accessing' }
Metaclass3 >> _name [
 | cls |
 self isMeta ifFalse:[ ^ super _name ].
 (cls := destClass) ifNotNil:[ ^ cls _name , ' class' ].
 ^ 'unknownMetaclass3'

]

{ #category : 'Private' }
Metaclass3 >> _primSubclass: nameSym
instVarNames: arrayOfIvNames
format: anInteger
classVars: aSymbolDictionary
poolDictionaries: anArrayOfPoolDicts
classInstanceVars: arrayOfCivNames [

	"subclass creation primitive.
 if aSymbolDictionary is empty or nil , the resulting class will have
  classVars == nil .
 if anArrayOfPoolDicts is empty or nil , the resulting class will have
   poolDictionaries == nil

 Gemstone64, this primitive enforces the restriction that
 a class cannot be more than 450 classes below Object when
 following the superClass chain upwards ."
	<primitive: 1042>

	nameSym _validateClass: Symbol.
	anInteger _validateClass: SmallInteger.
	arrayOfIvNames _validateClass: Array.
	arrayOfCivNames _validateClass: Array.
	aSymbolDictionary
		ifNotNil: [aSymbolDictionary _validateClass: SymbolDictionary].
	anArrayOfPoolDicts ifNotNil: [anArrayOfPoolDicts _validateClass: Array].
	^self
		_primitiveFailed: #_primSubclass:instVarNames:format:classVars:poolDictionaries:
		args:
			{nameSym.
			arrayOfIvNames.
			anInteger.
			aSymbolDictionary.
			anArrayOfPoolDicts.
			arrayOfCivNames}

]

{ #category : 'Modifying Classes' }
Metaclass3 >> _setClassVars: aDict old: previousDict [

  classVars ~~ aDict ifTrue:[
    previousDict ~~ classVars ifTrue:[ self error:'invalid store to classVars'].
    classVars := aDict .
    self thisClass _setClassVars: aDict old: previousDict
  ]

]

{ #category : 'Private' }
Metaclass3 >> _subclass: nameSym
instVarNames: arrayOfIvNames
format: anInteger
classVars: aSymbolDictionary
poolDictionaries: anArrayOfPoolDicts
classInstanceVars: arrayOfCivNames [
  "enforcment that nameSym is a valid Identifier is done in caller."
	^self
		_primSubclass: nameSym
		instVarNames: arrayOfIvNames
		format: anInteger
		classVars: aSymbolDictionary
		poolDictionaries: anArrayOfPoolDicts
		classInstanceVars: arrayOfCivNames

]

{ #category : 'Accessing' }
Metaclass3 >> _subclasses [

"Returns the class variable that is the list of subclasses, or
 nil if the receiver does not keep track of subclasses."

| subclss result |

self isMeta ifTrue:[
  subclss := self thisClass _subclasses .
  subclss == nil ifTrue:[ ^ nil ].
  result := IdentitySet new .
  subclss do:[ :aClass | result add: aClass class ].
  ^ result
].
^ super _subclasses

]

{ #category : 'Class Instance Variables' }
Metaclass3 >> _validateNewClassInstVar: ivname [

"Returns true if the name passes all tests.  Generates errors if the name fails
 due to a) being an invalid identifier or b) being a duplicate of an existing
 instance variable, either in this metaclass or in one of this metaclass's
 instance's subclasses metaclasses."

| subs |
((ivname size == 0
  or: [ivname isValidIdentifier not])
  or:[ ivname _isSymbol not]) ifTrue: [
  self _error: #classErrBadIdentifier args: { ivname }.
  ^false
  ].
(instVarNames includesIdentical: ivname) ifTrue: [
  self _error: #rtErrAddDupInstvar args: { ivname }.
  ^false
  ].
self thisClass isModifiable ifTrue: [
  subs := self _subclasses .
  subs ~~ nil ifTrue: [
    subs do: [:sub |
      (sub _validateNewClassInstVar: ivname) ifFalse: [ ^false ]
      ].
    ].
  ].
^true

]

{ #category : 'Modifying Classes' }
Metaclass3 >> _validateNewClassName: aName [
  "Return true if aName is a legal class name, otherwise signal an error.
   If execution continued from the error, return false. 
   Caller responsible for sending aSymbol to aName after this method returns true."

  aName isValidIdentifier ifFalse:[ 
    ( Exception _new: 2149 args: { aName } ) signal .
    ^ false
  ].
  ^ true .
]

{ #category : 'Class Instance Variables' }
Metaclass3 >> addInstVarNames: instVarNamesArray [

"Adds the instance variables specified in the argument to the receiver and any
 of its subclasses.  Generates an error upon encountering a name that is not a
 valid instance variable name or that is already an instance variable of the
 receiver.

 Instance variables that are added to a metaclass are called
 Class Instance Variables."

| ivnames oldEnd tCls |

self isMeta ifFalse:[ ^ self error:'receiver is not a metaClass' ].
tCls  := self thisClass .
(tCls isInvariant not or: [tCls subclassesDisallowed]) ifFalse: [
  ^ self _error: #rtErrClassNotModifiable
  ].
instVarNamesArray size == 0 ifTrue: [ ^self ].
"allow an error handler to proceed from the errors signaled by the validation
 method and thus skip offending instance variable names"
ivnames := { } .
instVarNamesArray do:[ :aStr | ivnames add: aStr asSymbol ].
ivnames := ivnames select: [:e |
  self _validateNewClassInstVar: e
  ].
oldEnd := self instSize + 1.
self _inheritCIVs: ivnames at: oldEnd.

]

{ #category : 'Accessing' }
Metaclass3 >> classHistory [
  self isMeta ifTrue:[ ^ destClass classHistory ].
  ^ super classHistory

]

{ #category : 'Updating the Method Dictionary' }
Metaclass3 >> compileAccessingMethodsFor: anArrayOfSymbols [

"Reimplemented to treat class instance variables specially,
 if isMeta is true for receiver .
 The new methods have environmentId == 0 .  "

| allVarNames varName symlst |
self isMeta ifFalse:[
  ^ super compileAccessingMethodsFor: anArrayOfSymbols
].
symlst := SymbolList new.
varName := 'newValue'.
allVarNames := self allInstVarNames collect:[:s | s asString ].
[allVarNames includesValue: varName] whileTrue: [
  varName :=  'z' , varName .
].
anArrayOfSymbols do: [ :var | | methodtext lf tab |
  lf := Character lf .
  tab := Character tab .
  (methodtext := String new) add: var; add: lf; add: tab; add: '^' ; add: var ; add: lf .
  [ self compileMethod: methodtext dictionaries: symlst
       category: #Accessing environmentId: 0
  ] onException: CompileError do:[:ex |
     self _error: #classErrNotAVar args: { var }
  ].
  (allVarNames indexOf: var) > Metaclass3 instSize ifTrue:[
    "compile a method that lets the variable be modified if the user has
     the proper authority"
    (methodtext := String new)
      add: var; add: ': '; add: varName; add: lf;
      add: tab; add: 'self atClassInstVar: #'''; add: var;
      add: ''' put: '; add: varName; add: lf .
    [ self compileMethod: methodtext dictionaries: symlst
           category: #Updating environmentId: 0
    ] onException: CompileError do:[:ex |
      self _error: #classErrNotAVar args: { var }
    ].
  ] ifFalse: [
    (methodtext := String new ) add: var; add: ': ' ; add: varName ; add: lf ;
      add: tab; add: var ; add: ' := ' ; add: varName ; add: lf .
    [ self compileMethod: methodtext dictionaries: symlst
             category: #Updating environmentId: 0
    ] onException: CompileError do:[:ex |
        self _error: #classErrNotAVar args: { var }
    ].
  ].
].

]

{ #category : 'Browser Methods' }
Metaclass3 >> definition [
	"We are created as a side-effect of creating the class!"

	^self thisClass definition.

]

{ #category : 'Accessing' }
Metaclass3 >> extraDict [
  self isMeta ifTrue:[ ^ destClass extraDict ].
  ^ nil

]

{ #category : 'Accessing' }
Metaclass3 >> extraDictForStore [
  self isMeta ifTrue:[ ^ destClass extraDictForStore ].
  ^ nil

]

{ #category : 'Accessing' }
Metaclass3 >> instanceString [
  self isMeta ifTrue:[ ^ destClass name ] ifFalse:[ ^ name ]

]

{ #category : 'Accessing' }
Metaclass3 >> instanceSymbol [
  self isMeta ifTrue:[ ^ destClass name ] ifFalse:[ ^ name ]

]

{ #category : 'Accessing' }
Metaclass3 >> isMeta [

"Returns true if the receiver is a meta class "

 ^ format ifNil:[ false "cope with malformed classes" ]
      ifNotNil:[:fmt| (fmt bitAnd: GC_Metaclass3_metaClass) ~~ 0]

]

{ #category : 'Accessing' }
Metaclass3 >> methodStampDictName [
  self isMeta ifTrue:[ ^ #GSMetaMethodStampDict ].
  ^ super methodStampDictName.

]

{ #category : 'Accessing' }
Metaclass3 >> name [
 self isMeta ifTrue:[ ^ destClass name , ' class' ]
           ifFalse:[ ^ super name ]

]

{ #category : 'Instance Creation' }
Metaclass3 >> new [
  "Disallowed.  To create a new class or metaclass, use
   Class | subclass:instVarNames:... instead."
  self shouldNotImplement: #new

]

{ #category : 'Instance Creation' }
Metaclass3 >> new: aSize [
  "Disallowed.  To create a new class or metaclass, use
   Class | subclass:instVarNames:... instead."
  self shouldNotImplement: #new:

]

{ #category : 'Accessing' }
Metaclass3 >> pragmaDictName [

  self isMeta ifTrue:[ ^ #GSMetaMethodPragmaDict ].
  ^ super pragmaDictName

]

{ #category : 'Accessing' }
Metaclass3 >> theNonMetaClass [
  "used by Monticello"

  "inline  self thisClass"
  (format bitAnd: GC_Metaclass3_metaClass) ~~ 0 ifTrue:["a meta class"  ^ destClass ] .
  ^ self

]

{ #category : 'Accessing' }
Metaclass3 >> thisClass [

"If receiver is a meta class, returns the class, else returns self."

(format bitAnd: GC_Metaclass3_metaClass) ~~ 0 ifTrue:["a meta class"  ^ destClass ] .
^ self

]

{ #category : 'Accessing' }
Metaclass3 >> versionedName [
 self isMeta ifTrue:[ ^ destClass versionedName , ' class' ]
           ifFalse:[ ^ super name ]

]
