Extension { #name : 'InterSessionSignal' }

{ #category : 'Signalling' }
InterSessionSignal class >> disableSignalling [

"Prevent instances of InterSessionSignal from being signalled by the VM"

 System _updateSignalErrorStatus: 3 toState: false

]

{ #category : 'Signalling' }
InterSessionSignal class >> enableSignalling [

"Enable instances of InterSessionSignal to be signalled by the VM .
 must be resent by exception handling code after handling each
 instance of InterSessionSignal that was signalled by the VM."

 System _updateSignalErrorStatus: 3 toState: true

]

{ #category : 'Signalling' }
InterSessionSignal class >> poll [
"Returns a new instance of InterSessionSignal if a signal is
 available from another session, or nil ."

| sigArray ex |
sigArray := System _signalFromGemStoneSession .
sigArray ifNil:[ ^ nil ].
(ex := self new) fromSession:  (sigArray at: 1)
              signal: (sigArray at: 2) message: (sigArray at: 3)
              numPending: ( sigArray at: 4).
^ ex

]

{ #category : 'Signalling' }
InterSessionSignal class >> pollAndSignal [
  "Signal a new instance of InterSessionSignal if a signal is
    available from another session, otherwise do nothing."

  self poll ifNotNil:[ :ex | ex signal ].

]

{ #category : 'Signalling' }
InterSessionSignal class >> sendSignal: aSmallInt to: aSessionId withMessage: aString [

"Send a signal to another VM which can result in an InterSessionSignal
 being signalled within that VM."

^ System sendSignal: aSmallInt to: aSessionId withMessage: aString

]

{ #category : 'Signalling' }
InterSessionSignal class >> signallingEnabled [

"Return true if signaling enabled, return false otherwise  .
 See InterSessionSignal(C)>>enableSignalling "

^ System _signalErrorStatus: 3

]

{ #category : 'Formatting' }
InterSessionSignal >> _legacyHandlerArgs [
  | arr |
  (arr := { sendingSession . signal } ) addAll: super _legacyHandlerArgs .
  ^ arr

]

{ #category : 'Formatting' }
InterSessionSignal >> buildMessageText [
^ self buildMessageText:(
  [ | str |
    (str := String withAll:'fromSession=') add: sendingSession asString;
      add: ' signal:'; add: signal asString ;
      add: ' message: ' ; add: (gsArgs atOrNil: 1) asString .
    gsDetails ifNotNil:[:d| str add:', '; add: d  asString ].
    str
  ] onException: Error do:[:ex |
    ex return: nil
  ]
)

]

{ #category : 'Handling' }
InterSessionSignal >> defaultAction [

"Return an error to the controlling debugger or GCI client .
 See AbstractException>>defaultAction for documentation."

^ self _signalToDebugger 

]

{ #category : 'Handling' }
InterSessionSignal >> fromSession: sessionIdInt signal: signalInt message: msgString [
  gsNumber := ERR_InterSessionSignal .
  sendingSession := sessionIdInt .
  signal := signalInt .
  gsArgs := { msgString . 0} .

]

{ #category : 'Handling' }
InterSessionSignal >> fromSession: sessionIdInt signal: signalInt message: msgString numPending: anInt [
  gsNumber := ERR_InterSessionSignal .
  sendingSession := sessionIdInt .
  signal := signalInt .
  gsArgs := { msgString . anInt } .

]

{ #category : 'Instance initialization' }
InterSessionSignal >> initialize [
  gsNumber := ERR_InterSessionSignal.
  gsResumable := true .
  gsTrappable := true .

]

{ #category : 'Accessing' }
InterSessionSignal >> numPending [
  "Returns a SmallInteger the number pending in stone beyond this one
   at the point this one was obtained from stone."
  | res |
  res := gsArgs ifNotNil:[:a | a atOrNil: 2 ].
  res ifNil:[ res := 0 ].
  ^ res

]

{ #category : 'Signalling' }
InterSessionSignal >> replyToSenderWithSignal: aSmallInteger withString: aString [

"Reply to the VM which caused this instance "

^ System sendSignal: aSmallInteger to: self sendingSessionId withMessage: aString

]

{ #category : 'Deprecated' }
InterSessionSignal >> sendingSession [
  self deprecated: 'Exception>>sendingSession deprecatedi v3.0, use #sendingSessionId or #sendingSessionSerial'.
  ^ sendingSession

]

{ #category : 'Accessing' }
InterSessionSignal >> sendingSessionId [
  ^ GsSession sessionIdOfSerial: sendingSession

]

{ #category : 'Accessing' }
InterSessionSignal >> sendingSessionSerial [
  ^ sendingSession

]

{ #category : 'Accessing' }
InterSessionSignal >> sentInt [
  ^ signal

]

{ #category : 'Accessing' }
InterSessionSignal >> sentMessage [
  ^ gsArgs ifNotNil:[:a | a atOrNil: 1 ].

]
