Extension { #name : 'Integer' }

{ #category : 'Private' }
Integer class >> _finishFromStream: aStream signFound: aSignFound factor: aFactor radix: radix [
  "BinaryExponent = ( 'e' | 'E' | 'd' | 'D' | 'q' ) ['-' | '+'] Digits
   DecimalExponent = ( 'f' | 'F' ) ['-' | '+'] Digits
   Digit = '0' | '1' | '2' | ... | '9'
   Digits = Digit {Digit}
   Exponent = BinaryExponent | DecimalExponent | ScaledDecimalExponent | FixedPointExponent
   FractionalPart = '.' Digits [Exponent]
   FixedPointExponent = 'p' [ ['-' | '+'] Digits ]
   Number = RadixedLiteral | NumericLiteral
   Numeric = Digit | 'A' | 'B' | ... | 'Z'
   NumericLiteral = Digits ( [FractionalPart] | [Exponent] )
   Numerics = Numeric { Numeric }
   RadixedLiteral = Digits ( '#' | 'r' ) ['-'] Numerics
   ScaledDecimalExponent = 's' [ ['-' | '+'] Digits ]
"
  "parse Numerics (plus option sign) for RadixedLiteral"

  | number ch signFound factor |
  signFound := aSignFound.
  factor := aFactor.
  number := 0.
  (aStream peek == $- )
    ifTrue: [
      signFound
        ifTrue: [ ^ self _errIncorrectFormat: aStream ].
      aStream next.
      signFound := true.
      factor := -1 ]
    ifFalse: [
      (aStream peek == $+ )
        ifTrue: [
          signFound
            ifTrue: [ ^ self _errIncorrectFormat: aStream ].
          aStream next.
          signFound := true ] ].
  [ aStream peek ifNil:[ false ] ifNotNil:[:x | x isSeparator == false] ]
    whileTrue: [
      ch := aStream next.
      number := number * radix + (ch digitValueInRadix: radix) ].
  ^ number * factor

]

{ #category : 'Private' }
Integer class >> _fromStream: aStream [
  "parse after leading whitespace has been removed :
     Number = [ - ] RadixedLiteral | [ '-' ] Digits | RadixedLiteralNegBody
     RadixedLiteral = Digits ( '#' | 'r' ) Numerics
     RadixedLiteralNegBody =  Digits ( '#' | 'r' ) '-' Numerics "
	| ch result sign dCount |
	(aStream peek == $- ) 
    ifTrue:[ aStream next.  sign := $-.  ]
		ifFalse:[(aStream peek == $+ ) ifTrue: [aStream next.  sign := $+ ]].

	result := 0.
  dCount := 0 .
	[ (ch := aStream peek) ~~ nil and:[ ch isDigit] ] whileTrue:[
     aStream next.
	 	 result := result * 10 + ch digitValue .
     dCount := dCount + 1 
  ].
  dCount == 0 ifTrue:[ ^self _errIncorrectFormat: aStream].
	(ch == $r or: [ch == $#]) ifTrue:[ "found a radix separator"
    aStream next.
    ^ self _fromStream: aStream radix: result sign: sign .
  ].
  sign == $- ifTrue:[ ^ 0 - result ].
	^ result .

]

{ #category : 'Private' }
Integer class >> _fromStream: aStream radix: aRadix sign: signArg [
  "parse 
      [ '-' ] Numerics
   portions of
     RadixedLiteral = Digits ( '#' | 'r' ) Numerics
     RadixedLiteralNegBody =  Digits ( '#' | 'r' ) '-' Numerics 

   a Sign in the body is illegal if a sign came before Digits "
  | result sign ch digit |
	result := 0.
  sign := signArg .
	(ch := aStream peek == $- ) ifTrue:[
    sign ifNotNil: [^self _errIncorrectFormat: aStream].
		aStream next.
		sign := $-
  ] ifFalse:[
    (ch == $+ ) ifTrue:[ 
      sign ifNotNil: [^self _errIncorrectFormat: aStream].
			aStream next.
			sign := $+
    ]
  ].
  [ (ch := aStream peek) ~~ nil and:[ (digit := ch digitValueInRadix: aRadix) ~~ nil ] ] whileTrue:[
     ch := aStream next.
		result := (result * aRadix) + digit .
  ].
  sign == $- ifTrue:[ ^ 0 - result ].
  ^ result
]

{ #category : 'Instance Creation' }
Integer class >> _new: length [

"Returns a instance of LargeInteger with specified size and value = 0"

<primitive: 266>
length _validateClass: SmallInteger .
length _error: #rtErrArgOutOfRange

]

{ #category : 'Instance Creation' }
Integer class >> _new: length neg: aBoolean [

| res |
res := self _new: length .
aBoolean ifTrue:[ res _digitAt:0 put: 1 "set sign bit" ].
^ res

]

{ #category : 'Private' }
Integer class >> _newWithValue: aSmallInt [

"Provided for use in testing the image .
 Returns a subnormal LargeInteger equal to the argument.

 The absolute value of aSmallInt must be a SmallInteger . "

| res arg isNeg idx |
isNeg := aSmallInt < 0 .
arg := aSmallInt .
isNeg ifTrue:[ arg := arg negated ].
arg _isSmallInteger ifFalse:[ arg _validateClass: SmallInteger] .
res := self _new: 2 neg: isNeg .
idx := 1 .
[arg == 0] whileFalse:[ | aDig |
  aDig := arg bitAnd: 16rffffffff .
  res _digitAt: idx put: aDig .
  idx := idx + 1 .
  arg := arg bitShift: -32 .
].
^ res immediateInvariant .

]

{ #category : 'Constants' }
Integer class >> _selectedPrimeGreaterThan: anInteger [
  ^ self _selectedPrimeGreaterThan: anInteger limit: 160481189
]

{ #category : 'Constants' }
Integer class >> _selectedPrimeGreaterThan: anInteger limit: aLimit [

"Returns a prime number that is greater than or equal to anInteger.
 The prime number is not necessarily the smallest prime number that is
 greater than or equal to anInteger.  For anInteger greater than our
 highest selected prime, returns an odd number."

| primes prime hi lo mid |
primes := #( 3 5 7 11 17 19 23 29 37 53 73 107 157 233 347 503 751
853 907 929 947 977 991 1009 1013 1019 1021 1039 
1511 
1723 1829 1901 1933 1969 1979 1987 2003 
2011 2017 2027 2029
3001 4001 5003 6007 7001 8009 9001
10007 11003 12007 13001 14009 15013 16001 17011 18013 19001
20011 21001 22003 23003 24001 25013 26003 27011 28001 29009
30011 31013 32003 33013 34019 35023 36007 37003 38011 39019
40009 41011 42013 43003 44017 45007 46021 47017 48017 49003
50021 51001 52009 53003 54001 55001 56003 57037 58013 59009
60013 61001 62003 63029 64007 65003 66029 67003 68023 69001
70001 71011 72019 73009 74017 75011 76001 77003 78007 79031
80021 81001 82003 83003 84011 85009 86011 87011 88001 89003
90001 91009 92003 93001 94007 95003 96001 97001 98009 99013
100003 101009 102001 103001 104003 224737 350377 479909
611953 746773 882377 1020379 1159523 1299709 2750159 4256233
5800079 7368787 8960453 10570841 12195257 13834103 15485863
32452843 49979687 67867967 86028121 104395301 122949823
141650939 160481183 ).

hi := primes size.
(anInteger > (primes at: hi))
  ifTrue: [
    ^ anInteger odd
      ifTrue: [ anInteger ]
      ifFalse: [ anInteger + 1 ]
  ].

lo := 0.
[       "Binary search for closest selected prime that is >= anInteger"
  mid := (hi + lo) // 2.
  prime := primes at: mid.
  anInteger <= prime
    ifTrue: [ hi := mid ]
    ifFalse: [ lo := mid ].
  hi - lo > 1
] untilFalse.

prime := primes at: hi .
prime > aLimit ifTrue:[ ^ primes at: hi - 1 ].
^ prime 
]

{ #category : 'Instance Creation' }
Integer class >> fromCompleteString: aString [

"Returns an instance of the appropriate subclass, reconstructed from aString.
 Leading blanks and trailing blanks are permitted.  Trailing non-digits
 generate an error.

 Smalltalk radix syntax for non-base-10 numbers is supported."

<primitive: 343>

" non-primitive implementation handles error conditions and arguments that
  are not a kind of String "
| s result |

s := ReadStreamPortable on: aString.
result := self fromStream: s.
[ s atEnd ]
whileFalse:
  [ (s next == $  )
    ifFalse:
      [ ^ self _errIncorrectFormat: aString ]
  ].
^ result


]

{ #category : 'Instance Creation' }
Integer class >> fromHexString: aString [

"Returns an instance of the appropriate subclass of the receiver whose
 value is read from the given string."

| ch factor result idx size digit legalChars |

idx := 1.
size := aString size.
size ~~ 0 ifTrue: [
  ch := aString at: idx.
  [ ch == $   ] whileTrue: [
    idx := idx + 1.
    ch := aString at: idx
  ].

  ch == $- ifTrue: [
    idx := idx + 1.
    factor := -1
  ]
  ifFalse: [
    ch == $+ ifTrue: [
      idx := idx + 1.
    ]
  ].
].

result := 0.

legalChars := #($a $b $c $d $e $f $A $B $C $D $E $F) .
(idx > size or: [(ch := aString at: idx) isDigit == false
  and: [ (legalChars includesIdentical: ch) == false]]) ifTrue: [
  ^ self _errIncorrectFormat: aString
].

[ idx > size or: [ (ch := aString at: idx) isDigit == false
  and: [(legalChars includesIdentical: ch) == false]
  ] ] whileFalse: [
  digit := ch > $9 ifTrue: [ch asUppercase codePoint - $A codePoint + 10]
                    ifFalse: [ ch codePoint - $0 codePoint ].
  result := result * 16 + digit.
  idx := idx + 1.
].

(ch isLetter
  and: [(legalChars includesIdentical: ch) == false])
  ifTrue: [
  ^ self _errIncorrectFormat: aString
].

factor ~~ nil ifTrue: [
  result := result * factor
].

^result

]

{ #category : 'Instance Creation' }
Integer class >> fromStream: aStream [
	"Reads bytes from aStream and returns an instance of the appropriate subclass.
	 Starting at the current position of aStream, leading blanks and trailing blanks
	 are permitted.  Trailing non-digits terminate the conversion without raising any errors.

	 ANSI and Smalltalk-80 radix syntax for is supported (sign before or after the radix, respectively).

   parse this subset of the BNF: 
     Number = [ - ] RadixedLiteral | [ '-' ] Digits | RadixedLiteralNegBody
     RadixedLiteral = Digits ( '#' | 'r' ) Numerics
     RadixedLiteralNegBody =  Digits ( '#' | 'r' ) '-' Numerics "

	self _checkReadStream: aStream forClass: CharacterCollection.
  aStream peek ifNotNil:[:ch | 
    ch unicodeIsWhitespace ifTrue:[
      [ aStream next unicodeIsWhitespace ] whileTrue.
      aStream skip: -1.
    ].
  ].
  ^ self _fromStream: aStream

]

{ #category : 'Instance Creation' }
Integer class >> fromString: aString [

"Returns an instance of the appropriate subclass, reconstructed from aString.
 Leading blanks and trailing blanks are permitted.  Trailing non-digits
 terminate the conversion without raising any errors.

 Smalltalk radix syntax for non-base-10 numbers is supported.
 parse this subset of the BNF: 
     Number = [ - ] RadixedLiteral | [ '-' ] Digits | RadixedLiteralNegBody
     RadixedLiteral = Digits ( '#' | 'r' ) Numerics
     RadixedLiteralNegBody =  Digits ( '#' | 'r' ) '-' Numerics "

<primitive: 267>

" non-primitive implementation handles error conditions and arguments that
  are not a kind of String "
| s result |

s := ReadStreamPortable on: aString.
result := self fromStream: s.
^ result

]

{ #category : 'Converting' }
Integer >> _asHexString: hexChars [

"Returns a String representing the receiver as a base-16 number."

| val len dig mask shift str |

self < 0 ifTrue:[
  val := 0 - self . str := '-' copy .
] ifFalse:[
  self == 0 ifTrue:[ ^ '0' ].
  str := String new .
  val := self .
].
len := val _digitLength .
dig := val _digitAt: len . "most significant 32bit digit"
mask := 16rF0000000 .
shift := -28 .
[ mask ~~ 0 and:[ (dig bitAnd: mask) == 0] ] whileTrue:[
   mask := mask bitShift: -4 .  "skip leading zeros"
   shift := shift + 4 .
].
[ mask ~~ 0 ] whileTrue:[
  str add: (hexChars at: 1 + (( dig bitShift: shift ) bitAnd: 16rF)).
  mask := mask bitShift: -4 .
  shift := shift + 4 .
].
len - 1 _downTo: 1 do:[:n |
  dig := val _digitAt: n .
  mask := 16rF0000000 .
  shift := -28 .
  [ mask ~~ 0 ] whileTrue:[
    str add: (hexChars at: 1 + (( dig bitShift: shift ) bitAnd: 16rF)).
    mask := mask bitShift: -4 .
    shift := shift + 4 .
  ].
].
str size == 0 ifTrue: [
  str add: $0 .
].
^ str

]

{ #category : 'Private' }
Integer >> _bitShift: aSmallInteger [
  <primitive: 742>
  aSmallInteger _validateClass: SmallInteger.
  self _primitiveFailed: #_bitShift: args: { aSmallInteger }

]

{ #category : 'Private' }
Integer >> _bottomBit [
 "sender must ensure receiver is positive,
  used by raisedTo methods"

  ^ (self _digitAt: 1) bitAnd: 1

]

{ #category : 'Converting' }
Integer >> _coerce: aNumber [

"Coerces an Integer."

aNumber _isSmallInteger ifTrue:[
  ^ Integer _newWithValue: aNumber
].
^ aNumber truncated

]

{ #category : 'Accessing' }
Integer >> _decimalDigitsLength: useApproximationBoolean [

"Returns the number of decimal digits required to represent the absolute value of
 the receiver, ignoring any leading zero bits to left of least significant bit.
 Result is always >= 1

 If useApproximationBoolean == true, the result is approximated for
 receivers not representable as a SmallInteger.
 The approximation is
    nd := ((self _lastDigit _decimalDigitsLength: false)
           + (self _digitLength - 1) * 9.64 ) truncated .
 Where log base 10(16rFFFFFFFF) == 9.63...  .
 The approximate result is within +/- 1 of the exact result for results
 <= 2860 ."

<primitive: 852>
self _primitiveFailed: #_decimalDigitsLength: args: { useApproximationBoolean }

]

{ #category : 'Accessing' }
Integer >> _digitAt: anIndex [

"Returns the specified 32bit unsigned word from a sign-magnitude representation
 of the receiver.
 anIndex should be a SmallInteger >= 0 .
 _digitAt:0 is the sign word , which contains either 0 or 1, 1 means negative.
 anIndex beyond most significant digit will return 0.
 The result will be a SmallInteger >= 0 "

<primitive: 18>

anIndex _validateClass: SmallInteger.
anIndex < 0 ifTrue:[ anIndex _error: #errArgTooSmall args:{ 0 } ].
self _primitiveFailed: #_digitAt: args: { anIndex } .
self _uncontinuableError

]

{ #category : 'Accessing' }
Integer >> _digitLength [

"Returns the number of base 4294967296 digits in the receiver, without counting
 any leading zeros or the sign digit."

<primitive: 19>

self _primitiveFailed: #_digitLength .
self _uncontinuableError

]

{ #category : 'Converting' }
Integer >> _generality [

"Returns the generality of Integer."

^ 40

]

{ #category : 'Private' }
Integer >> _isLeapYear [

"Returns true if the year represented by the value of the receiver is
 a leap year, false otherwise.  Valid at least for years 1900 and later."

 "Useful in debugging Date and DateTime."

(self \\ 100) = 0 ifTrue:[
   ^ (self \\ 400) = 0
   ].
^ (self \\ 4) = 0

]

{ #category : 'Accessing' }
Integer >> _lastDigit [

"Returns the last base 4294967296 digit of the receiver."

^self _digitAt: self _digitLength

]

{ #category : 'Testing' }
Integer >> _validateIsPowerOf2 [

self isPowerOf2
  ifFalse:[ self _error: #rtErrNotPowerOfTwo args: { self } ] .

]

{ #category : 'Arithmetic' }
Integer >> - anInteger [

"Returns the difference between the receiver and anInteger."

<primitive: 259>

^ self _retry: #- coercing: anInteger

]

{ #category : 'Arithmetic' }
Integer >> * anInteger [

"Returns the product of the receiver and anInteger."

<primitive: 260>

^ self _retry: #* coercing: anInteger

]

{ #category : 'Arithmetic' }
Integer >> / anInteger [

"Returns the result of dividing the receiver by anInteger."

<primitive: 261>
(anInteger == 0)
  ifTrue: [^ self _errorDivideByZero].
(anInteger _isInteger)
  ifTrue: [^(Fraction numerator: self denominator: anInteger)]
  ifFalse: [^ self _retry: #/ coercing: anInteger]

]

{ #category : 'Arithmetic' }
Integer >> // anInteger [

"Divides the receiver by anInteger.  Returns the integer quotient, with
 truncation toward negative infinity.  For example,

 9//4 = 2
 -9//4 = -3

 The selector \\ returns the remainder from this division."

<primitive: 263>
(anInteger == 0) ifTrue: [^ self _errorDivideByZero].
^ self _retry: #// coercing: anInteger

]

{ #category : 'Arithmetic' }
Integer >> \\ anInteger [

"Returns the modulus defined in terms of //.  Returns a Number with the same
 sign as the argument anInteger.  For example,

 9\\4 = 1
 -9\\4 = 3
 9\\-4 = -3"

<primitive: 264>
(anInteger == 0)
  ifTrue: [^ self _errorDivideByZero].
^ super \\ anInteger

]

{ #category : 'Arithmetic' }
Integer >> + anInteger [

"Returns the sum of the receiver and anInteger."

<primitive: 258>

^ self _retry: #+ coercing: anInteger

]

{ #category : 'Comparing' }
Integer >> < aNumber [
	"Returns true if the receiver is less than aNumber; returns false otherwise."

	<primitive: 20>
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ ArgumentTypeError signal: 'Expected a Number' ].
	^ (AbstractFraction _coerce: self) < aNumber
]

{ #category : 'Comparing' }
Integer >> <= aNumber [
	"Returns true if the receiver is less than or equal to aNumber; returns false
 otherwise."

	<primitive: 21>
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ ArgumentTypeError signal: 'Expected a Number' ].
	^ (AbstractFraction _coerce: self) <= aNumber
]

{ #category : 'Comparing' }
Integer >> = aNumber [
	"Returns true if the receiver is equal to aNumber; returns false otherwise."

	"aNumber _isFloat ifTrue:[ ^ self asFloat = aNumber ]."

	<primitive: 24>
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ false ].
	^ (AbstractFraction _coerce: self) = aNumber
]

{ #category : 'Comparing' }
Integer >> > aNumber [
	"Returns true if the receiver is greater than aNumber; returns false
 otherwise."

	<primitive: 22>
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ ArgumentTypeError signal: 'Expected a Number' ].
	^ (AbstractFraction _coerce: self) > aNumber
]

{ #category : 'Comparing' }
Integer >> >= aNumber [
	"Returns true if the receive is greater than aNumber; returns false
 otherwise."

	<primitive: 23>
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ ArgumentTypeError signal: 'Expected a Number' ].
	^ (AbstractFraction _coerce: self) >= aNumber
]

{ #category : 'Comparing' }
Integer >> ~= aNumber [
	"Returns true if the receiver is not equal to aNumber; returns false
 otherwise."

	<primitive: 25>
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ true ].
	^ (AbstractFraction _coerce: self) ~= aNumber
]

{ #category : 'Bit Manipulation' }
Integer >> allMask: anInteger [

"Treats the argument anInteger as a bit mask.  Returns true if all of the bits
 that are 1 in the argument are 1 in the receiver; returns false otherwise."

^ anInteger = (self bitAnd: anInteger)

]

{ #category : 'Bit Manipulation' }
Integer >> anyMask: anInteger [

"Treats the argument anInteger as a bit mask.  Returns true if any of the bits
 that are 1 in the argument are 1 in the receiver; returns false otherwise."

^ 0 ~= (self bitAnd: anInteger )

]

{ #category : 'Converting' }
Integer >> asCharacter [

"Returns the Character whose value equals the receiver.  Allowable range
 for the receiver is 0 to 65535, inclusive."

^ Character withValue: self

]

{ #category : 'Converting' }
Integer >> asDecimalFloat [
"Returns a DecimalFloat representing the receiver."

<primitive: 137>

| factor sum radix |

sum := 0.0F0 .
radix := 4294967295.0F0 .
factor := self sign asFloat.
1 to: self size do: [:i |
  sum := (self _digitAt: i) * factor + sum.
  factor := factor * radix
].
^sum

]

{ #category : 'Converting' }
Integer >> asFloat [

"Returns a SmallDouble or Float representing the receiver."

<primitive: 138>

self _primitiveFailed: #asFloat .
self _uncontinuableError

]

{ #category : 'Converting' }
Integer >> asFraction [

"Returns self, per ANSI standard"

^ self

]

{ #category : 'Converting' }
Integer >> asHexString [

^ self _asHexString: '0123456789abcdef'

]

{ #category : 'Converting' }
Integer >> asHexStringWithLength: length [

"Behaves like the asHexString method except that the result
 string is padded with zeros if it is shorter than length
 characters until its size reaches the specified length."

| string diff |
string :=  self asHexString .
diff := length - string size .
diff > 0
  ifTrue:[ | result |
       result := String new.
       diff timesRepeat:[ result add: $0 ].
       result addAll: string.
       ^ result
].
^ string

]

{ #category : 'Converting' }
Integer >> asInteger [

"Returns the receiver."

^ self

]

{ #category : 'Formatting' }
Integer >> asString [

"Returns a string representing the receiver.  Positive
 values do not include a leading + ."


<primitive: 265>

self _primitiveFailed: #asString .
self _uncontinuableError

]

{ #category : 'Bit Manipulation' }
Integer >> bitAnd: anInteger [

"Returns an Integer whose bits are the logical and of the receiver's bits and
 the bits of anInteger."

<primitive: 740>
anInteger _validateClass: Integer .
self _primitiveFailed: #bitAnd: args: { anInteger }

]

{ #category : 'Bit Manipulation' }
Integer >> bitAt: argInt [

"Returns the bit at the ith positin of the receiver, where 1 is the least
 significant bit.
 Gs64 v3.0 changed from 0 based to 1 based argument to meet ANSI standard."

| word digitIndex i |
argInt _isSmallInteger ifFalse:[ argInt _validateClass: SmallInteger ].
argInt < 1 ifTrue:[ argInt _error: #errArgTooSmall args:{ 1 } ].
i := argInt - 1 . "i is zero based"
digitIndex := (i bitShift: -5) + 1.  "32 bits per digit, get a small integer"
word := self _digitAt: digitIndex.
(self _digitAt:0) ~~ 0 ifTrue:[  | index | "rcvr is negative"
  "convert word from ones-complement to twos-complement"
  word := 16rFFFFFFFF - word.
  index := 1.
  [ (index < digitIndex) and:[(self _digitAt: index) == 0] ] whileTrue: [
    index := index + 1
  ].
  index == digitIndex ifTrue: [ word := word + 1 ].
].

^ (word bitShift: 0 - (i bitAnd: 16r1F) ) bitAnd: 1

]

{ #category : 'Bit Manipulation' }
Integer >> bitAt: index put: bitValue [
	"Return an integer whose binary representation is identical to the receiver with the exception that
	 the value of the bit at position index is equal to the low order bit of bitValue.
	 The least significant bit of the receiver is designated as position 1, with indices increasing to the left.
	 ANSI says the result is undefined if either the receiver or bitValue is a negative integer.
	 GemStone uses two-complement negative numbers and accepts either or both being negative.
	 It is an error if index is less than or equal to zero.
	 Gs64 v3.0 changed from 0 based to 1 based argument to meet ANSI standard."

	| mask |
	index _isSmallInteger ifFalse: [index _validateClass: SmallInteger].
	index < 1 ifTrue: [^index _error: #errArgTooSmall args: {1}].
	mask := 1 bitShift: index - 1.
	bitValue == 1 ifTrue: [^self bitOr: mask].
	bitValue == 0 ifTrue: [^self bitAnd: -1 - mask	"inline bitInvert"].
	^self bitAt: index put: (bitValue bitAt: 1).

]

{ #category : 'Bit Manipulation' }
Integer >> bitInvert [

"Returns an Integer whose bits are the one's-complement of the receiver."

^ -1 - self

]

{ #category : 'Bit Manipulation' }
Integer >> bitOr: anInteger [

"Returns an Integer whose bits are the logical or of the receiver's bits and
 the bits of anInteger."

<primitive: 741>
anInteger _validateClass: Integer.
self _primitiveFailed: #bitOr: args: { anInteger }

]

{ #category : 'Bit Manipulation' }
Integer >> bitShift: aSmallInteger [

"Returns an Integer whose value (in two's-complement representation) is the
 receiver's value (also in two's-complement representation) shifted by
 anInteger bits.

 If anInteger is positive, the shift is to the left, and zero-bits enter at
 the right.  If anInteger is negative, the shift is to the right, and the sign
 bit is repeated at the left."

 | res |
 res := self _bitShift: aSmallInteger .
 (aSmallInteger < 0 and:[ self < 0]) ifTrue:[
   (self bitAnd: (1 bitShift: 0 - aSmallInteger) - 1 ) == 0 ifFalse:[
     res := res - 1
   ].
 ].
 ^ res

]

{ #category : 'Bit Manipulation' }
Integer >> bitXor: anInteger [

"Returns an Integer whose bits are the logical xor of the receiver's bits and
 the bits of anInteger."

<primitive: 743>
anInteger _validateClass: Integer .
self _primitiveFailed: #bitXor: args: { anInteger }

]

{ #category : 'Truncation and Rounding' }
Integer >> ceiling [

"Returns the receiver."

^ self

]

{ #category : 'Accessing' }
Integer >> denominator [

"For an Integer, always returns 1."

^1

]

{ #category : 'Arithmetic' }
Integer >> divMod: aNumber into: anArray [

"anArray should be an Array of size 2.
 Performs
   anArray at: 1 put: (self // aNumber) .
   anArray at: 2 put: (self \\ aNumber) .
 Uses the large integer divide C code which produces both quotient
 and remainder in a single pass .
 Returns anArray .
"
<primitive: 949>
(aNumber = 0) ifTrue:[^ self _errorDivideByZero].
anArray _isArray ifFalse:[ anArray _validateClass: Array ].
^ self _retry: #divMod:into: coercing: aNumber into: anArray

]

{ #category : 'Testing' }
Integer >> even [
  ^ (self bitAt: 1) == 0

]

{ #category : 'Arithmetic' }
Integer >> factorial [

"Returns the factorial of the receiver.  Returns 1 if the
 receiver is less than or equal to 1."

| x result |
result := 1  .
x := result .
self timesRepeat:[ result := result * x .  x := x + 1 ] .
^ result .

]

{ #category : 'Truncation and Rounding' }
Integer >> floor [

"Returns the receiver."

^ self

]

{ #category : 'Truncation and Rounding' }
Integer >> fractionPart [
  "Returns the fraction remaining after the receiver is truncated toward zero."
  ^ 0

]

{ #category : 'Divisibility' }
Integer >> gcd: anInteger [

"Answer the greatest common divisor of the receiver and anInteger."

"Answer the largest non-negative integer that divides both the receiver
and anInteger with no remainder.
Answer 0 if the receiver and anInteger are both zero."

| divisor dividend remainder |
anInteger _isInteger ifFalse:[ anInteger  _validateClass: Integer ].
dividend := self negative ifTrue:[ 0 - self] ifFalse:[ self ].
divisor := anInteger negative ifTrue:[ 0 - anInteger ] ifFalse:[ anInteger].
remainder := divisor .
[remainder = 0] whileFalse: [
  remainder := dividend \\ divisor.
  dividend := divisor.
  divisor := remainder
].
^dividend

]

{ #category : 'Comparing' }
Integer >> hash [

"Returns a numeric hash index.  For an Integer representable in 30 bits,
 returns a SmallInteger equal to the receiver, otherwise converts
 receiver to a Float and returns the hash of the Float. "

<primitive: 530>
self _primitiveFailed: #hash .
self _uncontinuableError

]

{ #category : 'Bit Manipulation' }
Integer >> highBit [

"(Subclass responsibility.)  Returns the index of the high-order bit that is set
 in the binary representation of the receiver.  (If the receiver is a negative
 integer, takes its absolute value first.) 
 As of v3.7, if the receiver is zero, this returns zero; in previous releases
 this returned nil.
 As of Gs64 v3.2  the least significant bit has index 1 ,
 in previous versions the least significant bit had index 0 .

 Example:   4 highBit == 3  "

Integer subclassResponsibility: #highBit

]

{ #category : 'Testing' }
Integer >> isPowerOf2 [

"Answer true if the receiver is a power of 2.  Otherwise answer false."
self == 0
  ifTrue:[^ false].

^ (self bitAnd: (self - 1)) == 0

]

{ #category : 'Testing' }
Integer >> isZero [
  ^ self = 0
]

{ #category : 'Divisibility' }
Integer >> lcm: anInteger [
	"Answer the least common multiple of the receiver and anInteger.
	 Answer the smallest non-negative integer which is evenly divided by
	 both the receiver and operand. Answer 0 if both are zero."

	^(self = 0 and: [anInteger = 0])
		ifTrue: [0]
		ifFalse: [(self // (self gcd: anInteger) * anInteger) abs].

]

{ #category : 'Bit Manipulation' }
Integer >> noMask: anInteger [

"Treats the argument anInteger as a bit mask.  Returns true if none of the bits
 that are 1 in the argument are 1 in the receiver; returns false otherwise."

^ 0 = (self bitAnd: anInteger)

]

{ #category : 'Accessing' }
Integer >> numerator [

"For an Integer, always returns the receiver."

^self

]

{ #category : 'Testing' }
Integer >> odd [
  ^ (self bitAt: 1) == 1

]

{ #category : 'Json' }
Integer >> printJsonOn: aStream [

	self printOn: aStream.

]

{ #category : 'Formatting' }
Integer >> printOn: aStream base: b [

"Prints a representation of the receiver on aStream in base b.  The base b
 must be 2 <= b <= 36.  Returns the receiver."

^ self printOn: aStream base: b showRadix: true.

]

{ #category : 'Formatting' }
Integer >> printOn: aStream base: b showRadix: aBoolean [

"Prints a representation of the receiver on aStream in base b.  The base b
 must be 2 <= b <= 36.  Returns the receiver."

aStream nextPutAll: (self printStringRadix: b showRadix: aBoolean)

]

{ #category : 'Formatting' }
Integer >> printStringRadix: base [
	"Return a string containing a sequence of characters that represents the numeric value of the
	 receiver in the radix specified by the argument. The sequence of characters must be
	 recognizable using the radixDigits production of the Smalltalk Lexical Grammar as if the numeric
	 value of the radixSpecifier was base. If the receiver is negative, a minus sign ('-') is prepended
	 to the sequence of characters. The result is undefined if base is less than two or greater than 36."

	^self printStringRadix: base showRadix: false

]

{ #category : 'Formatting' }
Integer >> printStringRadix: base showRadix: aBoolean [
	"Returns a String that describes the receiver in the specified radix.
	 The base b must be 2 <= b <= 36."

	| str b |
	self < 0
		ifTrue: [^'-' , (self negated printStringRadix: base showRadix: aBoolean)].
	b := base.
	(b between: 2 and: 36)
		ifFalse: [^self _error: #rtErrInvalidArgument args: {b}].
	b == 10
		ifTrue: [str := self asString]
		ifFalse:
			[b == 16
				ifTrue: [str := self _asHexString: '0123456789ABCDEF']
				ifFalse:
					[| abs digits quorem |
					abs := self.
					digits := #($0 $1 $2 $3 $4 $5 $6 $7 $8 $9 $A $B $C $D $E $F $G $H $I $J $K $L $M $N $O $P $Q $R $S $T $U $V $W $X $Y $Z).
					quorem := {nil. nil}.
					str := String new.
					abs == 0
						ifTrue: [str add: $0]
						ifFalse:
							[[abs > 0] whileTrue:
									[abs quoRem: b into: quorem.
									str add: (digits at: (quorem at: 2) + 1).
									abs := quorem at: 1]].
					str := str reverse]].
	aBoolean
		ifTrue:
			[| res |
			(res := b asString)
				add: $r;
				addAll: str.
			^res].
	^str

]

{ #category : 'Arithmetic' }
Integer >> quo: anInteger [

"Divides the receiver by anInteger.  Returns the integer quotient, with
 truncation toward zero.  For example,
   -9 quo: 4 = -2
 The selector rem: returns the remainder from this division.
 If you need both quotient and remainder, use  Integer>>quoRem:into:  "

<primitive: 262>
(anInteger == 0) ifTrue: [^ self _errorDivideByZero].
^ self _retry: #quo: coercing: anInteger

]

{ #category : 'Arithmetic' }
Integer >> quoRem: aNumber into: anArray [

"anArray should be an Array of size 2.
 Performs
   anArray at: 1 put: (self quo: aNumber) .
   anArray at: 2 put: (self rem: aNumber) .
 Uses the large integer divide C code which produces both quotient
 and remainder in a single pass .
 Returns anArray .
"
<primitive: 789>
(aNumber = 0) ifTrue: [^ self _errorDivideByZero].
anArray _isArray ifFalse:[ anArray _validateClass: Array ].
^ self _retry: #quoRem:into: coercing: aNumber into: anArray

]

{ #category : 'Arithmetic' }
Integer >> rem: aNumber [

"Returns the integer remainder defined in terms of quo: (division
 of the receiver by aNumber, with truncation toward zero)."

aNumber _isInteger ifTrue:[ | arr |
  (aNumber == 0) ifTrue: [^ self _errorDivideByZero].
  arr := { nil . nil }.
  self quoRem: aNumber into: arr .
  ^ arr at: 2
].
^ super rem: aNumber

]

{ #category : 'Truncation and Rounding' }
Integer >> roundAndCoerceTo: aNumber [

"Returns the multiple of aNumber that is nearest in value to the receiver."

aNumber = 0
  ifTrue: [^0]
  ifFalse: [^(self / aNumber) rounded * aNumber]
]

{ #category : 'Truncation and Rounding' }
Integer >> rounded [

"Returns the receiver."

^ self

]

{ #category : 'Truncation and Rounding' }
Integer >> roundedHalfToEven [
	"Returns the integer nearest in value to the receiver. If the receiver is
	exactly halfway between two integers, return the even one."

	^self

]

{ #category : 'Accessing' }
Integer >> size: anInteger [

"Disallowed.  You may not change the size of an Integer."

self shouldNotImplement: #size:

]

{ #category : 'Truncation and Rounding' }
Integer >> truncateAndCoerceTo: aNumber [

"Returns the multiple of aNumber that is closest to the receiver, on
 the same side of the receiver as zero is located.  In particular,
 returns the receiver if the receiver is a multiple of aNumber."

aNumber = 0
  ifTrue: [^0]
  ifFalse: [^(self quo: aNumber) * aNumber]
]

{ #category : 'Truncation and Rounding' }
Integer >> truncated [

"Returns the receiver."

^ self
]
