Extension { #name : 'IdentityKeyValueDictionary' }

{ #category : 'Private' }
IdentityKeyValueDictionary >> _at: aKey otherwise: defaultValue [

"Returns the value that corresponds to aKey.  If no such key/value pair
 exists, returns the given alternate value."

| hofs hashKey collisionBkt |

aKey ifNotNil:[
  hofs := (aKey identityHash \\ tableSize) . hofs := hofs + hofs .
  hashKey := self _atZ: hofs  .
  hashKey ifNil: [
    collisionBkt := self _atZ: hofs + 1 .
    collisionBkt ifNotNil: [ ^collisionBkt at: aKey otherwise: defaultValue ]
  ] ifNotNil: [
    (aKey == hashKey) ifTrue: [ ^ self _atZ: hofs + 1 ]
  ].
].
^ defaultValue

]

{ #category : 'Private' }
IdentityKeyValueDictionary >> _at: aKey put: aValue [

"Stores the aKey/aValue pair in the hash dictionary.  Rebuilds the hash table
 if the addition caused the number of collisions to exceed the limit allowed."

| hofs hashKey hashValue collisionBkt cbStatus |
aKey ifNil: [ ^ self _error: #rtErrNilKey ].
hofs := (aKey identityHash \\ tableSize) . hofs := hofs + hofs .
hashKey := self _atZ: hofs  .
hashKey ifNil: [
  collisionBkt := self _atZ: hofs + 1 .
  collisionBkt ifNil: [
    self _atZ: hofs putKey: aKey value: aValue .
    numElements := numElements + 1
  ] ifNotNil: [
    cbStatus := collisionBkt at: aKey put: aValue keyValDict_coll: self.
    cbStatus ~~ 0 ifTrue:[
      numElements := numElements + 1.
      cbStatus ~~ 1 ifTrue:[
        numCollisions := numCollisions + 1
      ].
    ].
  ]
] ifNotNil: [
  aKey == hashKey ifTrue: [
    self _atZ: hofs  putKey: aKey value: aValue .
  ] ifFalse: [
    hashValue := self _atZ: hofs + 1 .
    collisionBkt := self collisionBucketClass new.
    collisionBkt objectSecurityPolicy: self objectSecurityPolicy .
    collisionBkt keyValueDictionary: self.
    self _atZ: hofs putKey: nil value: collisionBkt .
    collisionBkt at: hashKey put: hashValue keyValDict_coll: self.
    numElements := numElements + 1.
    cbStatus := collisionBkt at: aKey put: aValue keyValDict_coll: self.
    cbStatus > 1 ifTrue:[
      numCollisions := numCollisions + 1
    ]
  ]
].
(numCollisions > collisionLimit) ifTrue: [
  self rebuildTable: self _nextTableSize 
].
^aValue

]

{ #category : 'Private' }
IdentityKeyValueDictionary >> _initializeWithoutClear: newSize [

"Private. Initializes the instance variables of the receiver to be an empty
 KeyValueDictionary of the specified size. Does not clear the contents
 of the receiver - assumes they are all nil."

tableSize := newSize.
numElements := 0.
numCollisions := 0.
collisionLimit := newSize  * 500 .
  "This is where it differs from KeySoftValueDictionary. By setting the default
   collision limit to be high in comparison to the size of the dictionary,
   the dictionary will only be rebuilt when the collision buckets have
   become big."
^self

]

{ #category : 'Private' }
IdentityKeyValueDictionary >> _removeKey: aKey ifAbsent: aBlock [

"Removes the key/value pair with key aKey from the receiver and returns
 the value.  If no key/value pair is present with key aKey, evaluates
 the zero-argument block aBlock and returns the result of that evaluation."

| hofs thisKey collisionBkt aValue lastPair |

aKey ifNil:[ ^ self _error: #rtErrNilKey ].
hofs := (aKey identityHash \\ tableSize) . hofs := hofs + hofs .
(thisKey := (self _atZ: hofs )) ifNil:[ "It might be in a collision bucket"
      collisionBkt := self _atZ: hofs + 1.
      collisionBkt ifNil:[
        aBlock ifNil:[^ self _errorKeyNotFound: aKey ] .
        ^ aBlock value
      ].
      aValue := collisionBkt removeKey: aKey ifAbsent: [
        aBlock ifNil:[^ self _errorKeyNotFound: aKey ] .
        ^ aBlock value
      ].
      "If we got this far, a key/value pair was removed from collisionBkt"
      numCollisions := numCollisions - 1.
      (collisionBkt size <= 1) ifTrue: [
         "We don't need the collision bucket anymore.
          Put its last key/value pair in our table."
         lastPair := collisionBkt _firstPair.
         self _atZ: hofs  putKey: (lastPair at:1) value: (lastPair at:2)
      ]
] ifNotNil: [ "There is a key/value pair in this slot"
      (aKey == thisKey) ifFalse: [ ^ self _reportKeyNotFound: aKey with: aBlock ] .
      "The key matches - remove it"
      aValue := self _atZ: hofs + 1.
      self _atZ: hofs putKey: nil value: nil .
].
numElements := numElements - 1.
^aValue

]

{ #category : 'Canonical Symbol Support' }
IdentityKeyValueDictionary >> _selectiveAbort [

""

| aKey collisionBkt |

1 to: tableSize * 2 by: 2 do: [ :tableIndex |
  nil == (aKey := self _at: tableIndex)
    ifTrue: [ nil == (collisionBkt := self _at: (tableIndex + 1))
      ifFalse: [collisionBkt _selectiveAbort]].
  ].

super _selectiveAbort

]

{ #category : 'Accessing' }
IdentityKeyValueDictionary >> at: aKey ifAbsent: aBlock [

"Returns the value that corresponds to aKey.  If no such key/value pair
 exists, returns the given alternate value."

| hofs hashKey collisionBkt |

aKey ifNotNil:[
  hofs := (aKey identityHash \\ tableSize) . hofs := hofs + hofs .
  hashKey := self _atZ: hofs  .
  hashKey ifNil: [
      collisionBkt := self _atZ: hofs + 1 .
      collisionBkt ifNotNil: [ ^ collisionBkt at: aKey ifAbsent: aBlock ]
  ] ifNotNil: [
    (aKey == hashKey) ifTrue: [ ^ self _atZ: hofs + 1 ]
  ].
].
aBlock ifNil:[^ self _errorKeyNotFound: aKey ] .
^ aBlock value

]

{ #category : 'Accessing' }
IdentityKeyValueDictionary >> at: aKey otherwise: defaultValue [

"Returns the value that corresponds to aKey.  If no such key/value pair
 exists, returns the given alternate value."

| hofs hashKey collisionBkt |

aKey ifNotNil:[
  hofs := (aKey identityHash \\ tableSize) . hofs := hofs + hofs .
  hashKey := self _atZ: hofs  .
  hashKey ifNil: [
    collisionBkt := self _atZ: hofs + 1 .
    collisionBkt ifNotNil: [ ^collisionBkt at: aKey otherwise: defaultValue ]
  ] ifNotNil: [
    (aKey == hashKey) ifTrue: [ ^ self _atZ: hofs + 1 ]
  ].
].
^ defaultValue

]

{ #category : 'Updating' }
IdentityKeyValueDictionary >> at: aKey put: aValue [

^ self _at: aKey put: aValue

]

{ #category : 'Private' }
IdentityKeyValueDictionary >> collisionBucketClass [

"Returns the class of object to create when keys collide."

^ IdentityCollisionBucket

]

{ #category : 'Hashing' }
IdentityKeyValueDictionary >> compareKey: key1 with: key2 [

"No longer used by implementation of IdentityKeyValueDictionary"

self shouldNotImplement: #compareKey:with:

]

{ #category : 'Comparing' }
IdentityKeyValueDictionary >> hash [

"Returns a numeric hash key for the receiver."

| hashValue |

hashValue := 97633 bitXor: (self size).
"For large dictionaries, the hash value is just a function of its size"
(self size > 64) ifTrue: [ ^ hashValue abs ].
self keysDo: [ :aKey |
   "Skip if the key is a dictionary."
   (aKey isKindOf: AbstractDictionary)
     ifFalse: [
       hashValue := hashValue bitXor: aKey identityHash
       ]
     ].
^ hashValue abs

]

{ #category : 'Hashing' }
IdentityKeyValueDictionary >> hashFunction: aKey [

"No longer used by implementation of IdentityKeyValueDictionary"

self shouldNotImplement: #hashFunction:

]

{ #category : 'Accessing' }
IdentityKeyValueDictionary >> keys [

"Returns an IdentitySet containing the receiver's keys."

| result |
result := IdentitySet new .
self keysDo:[ :aKey | result add: aKey ].
^ result

]

{ #category : 'Hashing' }
IdentityKeyValueDictionary >> rebuildTable: newSize [

super rebuildTable: newSize .
self collisionLimit: newSize * 500 . "differs from KeyValueDictionary"

]

{ #category : 'Hashing' }
IdentityKeyValueDictionary >> rehash [
	"Re-establish any hash invariants of the receiver.
	 Identity hashes cannot change."

]

{ #category : 'Removing' }
IdentityKeyValueDictionary >> removeKey: aKey ifAbsent: aBlock [
  ^ self _removeKey: aKey ifAbsent: aBlock

]

{ #category : 'Removing' }
IdentityKeyValueDictionary >> removeKey: aKey otherwise: defaultValue [

  ^ self _removeKey: aKey ifAbsent:[ ^ defaultValue ]

]
