Extension { #name : 'GsObjectSecurityPolicy' }

{ #category : 'Instance Creation' }
GsObjectSecurityPolicy class >> newInRepository: aRepository [

"Returns a new GsObjectSecurityPolicy in the Repository aRepository.  If the
 maximum number (65534) of ObjectSecurityPolicies has already been created for
 aRepository, this generates an error.
 The new GsObjectSecurityPolicy is owned by the UserProfile of the current session,
 and has the default authorization of W----- (owner can read and write).

 After execution of this method, the current transaction must be committed
 before objects can be created in the GsObjectSecurityPolicy, object assigned to the
 objectSecurityPolicy, or the GsObjectSecurityPolicy used as argument to a
 defaultObjectSecurityPolicy: keyword in creation of UserProfiles.

 Requires the ObjectSecurityPolicyCreation privilege and WriteAccess to
 DataCuratorObjectSecurityPolicy.
 "

<primitive: 291>
aRepository _validateClass: Repository.
self _primitiveFailed: #newInRepository: args: { aRepository } .
self _uncontinuableError

]

{ #category : 'Execution' }
GsObjectSecurityPolicy class >> setCurrent: anObjectSecurityPolicy while: aBlock [

| oldPolicy |

oldPolicy := System currentObjectSecurityPolicy.
oldPolicy == anObjectSecurityPolicy ifTrue: [ "no need to change/restore policy "
  ^ aBlock value
].
System currentObjectSecurityPolicy: anObjectSecurityPolicy .
^ [
    [
      aBlock value
    ] onException: Error do:[:ex |
      (oldPolicy == nil or:[ oldPolicy isKindOf: GsObjectSecurityPolicy]) ifTrue: [
        System currentObjectSecurityPolicy: oldPolicy .
      ].
      ex outer
    ]
  ] ensure:[
    System currentObjectSecurityPolicy: oldPolicy
  ].

]

{ #category : 'Private' }
GsObjectSecurityPolicy >> _authAt: anOffset put: aValue [

"offset   action
 ------   ---------
  5       assign to ownerAuthorization instance variable
  6       assign to worldAuthorization instance variable
  3       add to groupsRead IdentitySet, create set if needed.
  4       add to groupsWrite IdentitySet, create set if needed.
 -3       remove from groupsRead IdentitySet
 -4       remove from groupsWrite IdentitySet
  7       assign new value to groupsRead (for repository conversion)
  8       assign new value to groupsWrite (for repository conversion)

  Gs64 v2.2 ,  offsets 7 and 8 not supported, was used by _migrateGroups.
"

<primitive: 292>
self _primitiveFailed: #_authAt:put: args: { anOffset . aValue }.
self _uncontinuableError

]

{ #category : 'Private' }
GsObjectSecurityPolicy >> _currentUserAccess [

"returns current user's access to objects in the receiver  ,
 0 == none,  1 == read , 2 == write
"
<primitive: 591>
self _primitiveFailed: #_currentUserAccess.
self _uncontinuableError

]

{ #category : 'Copying' }
GsObjectSecurityPolicy >> _deepCopyWith: copiedObjDict [

"Private. Used internally to implement deepCopy."

^ self.

]

{ #category : 'Repository Conversion' }
GsObjectSecurityPolicy >> _migrateGroups: migrateBlock [

"Converts the groups collections from IdentitySets of Strings to
 IdentitySets of UserProfileGroups."

(Globals at:#DataCuratorObjectSecurityPolicy) setCurrentWhile:[
  migrateBlock value: groupsRead ; value: groupsWrite
].

]

{ #category : 'Accessing' }
GsObjectSecurityPolicy >> _name [
  ^ self dynamicInstVarAt: #name

]

{ #category : 'Formatting' }
GsObjectSecurityPolicy >> asString [

"Returns a formatted String that contains the following information:

 The name of the Repository containing the receiver and the index of
 the receiver in that Repository.

 * The user ID of the receiver's owner.

 * The Symbol that defines each user group which is authorized to read or
   write in the receiver.

 * Information about whether the receiver's owner, each group, or all
   users are authorized to read or write in the receiver.

 For example, the message 'System myUserProfile defaultObjectSecurityPolicy asString'
 returns a String that resembles the following:

 anObjectSecurityPolicy, Number 1 in Repository SystemRepository, Owner SystemUser write, World read

 If the receiver is not a GsObjectSecurityPolicy in the SystemRepository, the String contains
 'NOT IN REPOSITORY' in place of the index."

| result theGroupAuthString |
theGroupAuthString := String new.  " string of group authorizations"
"groups and build a single string to describe group authorizations."
groupsRead size ~~ 0 ifTrue:[
  (SortedCollection withAll: groupsRead ) do: [:groupName |
     theGroupAuthString addAll: ' Group '; addAll: (groupName asString);
                 add: ' '; addAll: 'read' ; add: $, .
  ].
].
groupsWrite size ~~ 0 ifTrue:[
  (SortedCollection withAll: groupsWrite ) do: [:groupName |
     theGroupAuthString addAll: ' Group '; addAll: (groupName asString);
        add: ' '; addAll: 'write' ; add: $, .
  ].
].
result := String new.
self _name ifNotNil:[:nam| result addAll: nam ]
           ifNil:[ result addAll: 'anObjectSecurityPolicy'].
result addAll: '(#';
       addAll: self objectSecurityPolicyId  asString ;
       addAll: ' in Repository ';
       addAll: self repository name ; add: $, ;
       addAll: ' Owner ';
       addAll: self owner userId ;   add: $   ;
       addAll: self ownerAuthorization ; add: $, ;
       addAll: theGroupAuthString;
       addAll: ' World ';
       addAll: self worldAuthorization .
self trackReads ifTrue:[ result addAll:', trackReads' ].
result  add: $) .
^ result
]

{ #category : 'Accessing Authorization' }
GsObjectSecurityPolicy >> authorizationForGroup: groupOrString [

"Returns a Symbol that defines whether the specified group is authorized to
 write (and read) in this GsObjectSecurityPolicy (#write), to read only
 (#read), or neither (#none)."

| theGroupObj |

"Find the UserProfileGroup object and handle the case where it was passed to us
 instead of the name of the group."
theGroupObj := (groupOrString class == UserProfileGroup)
  ifTrue: [ groupOrString ]
  ifFalse: [ UserProfileGroup groupWithName: groupOrString otherwise: nil ].
theGroupObj == nil ifTrue:[ ^ #none ].

groupsWrite size ~~ 0 ifTrue:[
  (groupsWrite includesIdentical: theGroupObj ) ifTrue:[ ^ #write ]
  ].
groupsRead size ~~ 0 ifTrue:[
  (groupsRead includesIdentical: theGroupObj) ifTrue:[ ^ #read ]
  ].
^ #none

]

{ #category : 'Accessing Authorization' }
GsObjectSecurityPolicy >> authorizationForUser: aUserProfile [

"Returns a Symbol that describes the authorization that the given user has for
 the receiver."

| userGroups curAuth |

"Consider world authorization"
curAuth := worldAuthorization.
curAuth == 2 ifTrue: [ ^ #write ].

"Owner authorization applicable?"
itsOwner == aUserProfile ifTrue: [
  ownerAuthorization == 2 ifTrue:[ ^ #write ].
  curAuth := curAuth bitOr: ownerAuthorization .
  ].

"Consider group authorizations..."
userGroups := aUserProfile groups .
userGroups size ~~ 0 ifTrue:[
  groupsWrite size ~~ 0 ifTrue:[
    (groupsWrite * userGroups) size ~~ 0 ifTrue:[ ^ #write ].
    ].
  groupsRead size ~~ 0 ifTrue:[
    (groupsRead * userGroups) size ~~ 0 ifTrue:[ curAuth := curAuth bitOr: 1 ].
    ].
  ].

curAuth == 1 ifTrue:[ ^ #read ].
^#none

]

{ #category : 'Clustering' }
GsObjectSecurityPolicy >> clusterDepthFirst [

"This method clusters the receiver.  (Overrides the inherited method, which
 also clusters all instance variables.)  Returns true if the receiver has
 already been clustered during the current transaction, false otherwise."

  self cluster ifTrue:[ ^ true ].
  groupsRead cluster .
  groupsWrite cluster .
^ false

]

{ #category : 'Accessing Authorization' }
GsObjectSecurityPolicy >> currentUserCanRead [

"Returns true if the current user has read authorization for the
 receiver, false if not."

^ self _currentUserAccess >= 1

]

{ #category : 'Accessing Authorization' }
GsObjectSecurityPolicy >> currentUserCanWrite [

"Returns true if the current user has write authorization for the
 receiver, false if not."

^ self _currentUserAccess == 2

]

{ #category : 'Updating Authorization' }
GsObjectSecurityPolicy >> group: groupOrString authorization: anAuthorizationSymbol [

"Redefines the authorization for the specified group to one of the following
 authorization Symbols:

 * #write (members of the group can both read and write in this GsObjectSecurityPolicy).
 * #read (read only).
 * #none (neither read nor write permission).

 This method generates an error if groupOrString is not either a UserProfileGroup or the
 name of a UserProfileGroup in global AllGroups, or if anAuthorizationSymbol is not one
 of (#read, #write, #none).

 Requires the ObjectSecurityPolicyProtection privilege, if the objectSecurityPolicy is not owned
 by the GemStone UserProfile under which this session is running."

| theGroupObj |

"Test for errInvalidAuthorization "
( #( #write #read #none) includesValue: anAuthorizationSymbol)
ifFalse: [  ^ self _error: #segErrBadAuthKind args: { anAuthorizationSymbol } ].

"Find the UserProfileGroup object and handle the case where it was passed to us
 instead of the name of the group."
theGroupObj := (groupOrString class == UserProfileGroup)
  ifTrue: [ groupOrString ]
  ifFalse: [ UserProfileGroup _validateGroupString: groupOrString ].

"modify the group collections for this objectSecurityPolicy as appropriate."
anAuthorizationSymbol == #write ifTrue:[
  self _authAt: -3 put: theGroupObj "readGroups remove: theGroupObj ifAbsent:[]" .
  self _authAt: 4 put: theGroupObj  "writeGroups _addSymbol: theGroupObj " .
  ]
ifFalse:[
  anAuthorizationSymbol == #read ifTrue:[
    self _authAt: -4 put: theGroupObj "writeGroups remove: theGroupObj ifAbsent:[]" .
    self _authAt: 3 put: theGroupObj  "readGroups _addSymbol: theGroupObj " .
    ]
  ifFalse:[ "none"
    self _authAt: -3 put: theGroupObj "readGroups remove: theGroupObj ifAbsent:[]" .
    self _authAt: -4 put: theGroupObj "writeGroups remove: theGroupObj ifAbsent:[]" .
    ].
  ].

]

{ #category : 'Deprecated' }
GsObjectSecurityPolicy >> groupNo: groupIndex group: aGroupString authorization: anAuthorizationSymbol [

self deprecated: 'GsObjectSecurityPolicy>>groupNo:group:authorization: obsolete, deprecated v3.0. Use #group:authorization:.'.
^ self group: aGroupString authorization: anAuthorizationSymbol

]

{ #category : 'Accessing' }
GsObjectSecurityPolicy >> groups [

"Returns an IdentitySet, the set of UserProfileGroups that are
 explicitly authorized to read or write in this GsObjectSecurityPolicy."

| result |
groupsRead ~~ nil ifTrue:[ result := groupsRead ]
                 ifFalse:[ result := IdentitySet new ].
groupsWrite ~~ nil ifTrue:[ result := result + groupsWrite ].
^ result

]

{ #category : 'Accessing Authorization' }
GsObjectSecurityPolicy >> groupsWithAuthorization: anAccessSymbol [

"Returns an IdentitySet of UserProfileGroups with the authorization of
 anAccessSymbol (one of #write, #read, or #none) for the receiver."

| result |

( #( #write #read #none) includesValue: anAccessSymbol)
  ifFalse: [  ^ self _error: #segErrBadAuthKind args: { anAccessSymbol } ].

result := IdentitySet new .
AllGroups valuesDo:[ :aGroup |
  (self authorizationForGroup: aGroup groupName) == anAccessSymbol ifTrue:[
    result add: aGroup
    ].
  ].
^ result

]

{ #category : 'Accessing' }
GsObjectSecurityPolicy >> name [
 | n |
 n := self dynamicInstVarAt: #name.
 n ifNil:[
    itsId == 4 ifTrue:[ ^ 'GsIndexingObjectSecurityPolicy' ].
    itsId == 5 ifTrue:[ ^ 'SecurityDataObjectSecurityPolicy' ].
    ^ 'ObjectSecurityPolicy' , itsId asString
 ].
 ^ n

]

{ #category : 'Updating' }
GsObjectSecurityPolicy >> name: aString [
  self dynamicInstVarAt: #name put: aString

]

{ #category : 'Accessing' }
GsObjectSecurityPolicy >> number [

"Returns the index of the receiver in its Repository"

^ itsId

]

{ #category : 'Accessing' }
GsObjectSecurityPolicy >> objectSecurityPolicyId [

"Returns the index of the receiver in its Repository"

^ itsId

]

{ #category : 'Accessing' }
GsObjectSecurityPolicy >> owner [

"Returns the UserProfile of the receiver's owner."

^ itsOwner

]

{ #category : 'Updating' }
GsObjectSecurityPolicy >> owner: aUserProfile [

"Redefines the receiver's owner to be the user represented by aUserProfile.
 To execute this method, you must be authorized to write in the receiver's
 GsObjectSecurityPolicy (customarily, the DataCurator GsObjectSecurityPolicy)."

itsOwner := aUserProfile

]

{ #category : 'Accessing Authorization' }
GsObjectSecurityPolicy >> ownerAuthorization [

"Returns a Symbol that defines whether the GsObjectSecurityPolicy's owner is authorized to
 write (and read) in this GsObjectSecurityPolicy (#write), to read only (#read), or neither
 (#none)."

^ #( #none #read #write) at: ownerAuthorization + 1

]

{ #category : 'Updating Authorization' }
GsObjectSecurityPolicy >> ownerAuthorization: anAuthorizationSymbol [

"Redefines the authorization for the GsObjectSecurityPolicy's owner to one of the following
 authorization Symbols:

 * #write (the GsObjectSecurityPolicy's owner can both read and write in this GsObjectSecurityPolicy).

 * #read (read only).

 * #none (neither read nor write permission).

 Generates an error if anAuthorizationSymbol is not one of (#read, #write,
 #none).

 Requires the ObjectSecurityPolicyProtection privilege, if the objectSecurityPolicy is not owned
 by the GemStone UserProfile under which this session is running."

 | authInt |
 authInt := #( #none #read #write ) indexOf: anAuthorizationSymbol .
 authInt == 0 ifTrue:[
   ^ self _error: #segErrBadAuthKind args: { anAuthorizationSymbol }
   ].
 self _authAt: 5 put: (authInt - 1)

]

{ #category : 'Accessing' }
GsObjectSecurityPolicy >> repository [

"Returns the Repository containing the receiver."

^ itsRepository

]

{ #category : 'Deprecated' }
GsObjectSecurityPolicy >> segmentId [

self deprecated: 'GsObjectSecurityPolicy>>segmentId is deprecated v3.0 and provided for backwards compatibility.
Use #objectSecurityPolicyId instead.'.

^self objectSecurityPolicyId

]

{ #category : 'Execution' }
GsObjectSecurityPolicy >> setCurrentWhile: aBlock [

"Sets the receiver to be the current security policy while the given block
 executes.  Catches all errors and reinstalls the current security policy
 to avoid having the receiver be left as the current security policy.  Returns
 the result of evaluating the zero-argument block."

^ self class setCurrent: self while: aBlock

]

{ #category : 'Accessing' }
GsObjectSecurityPolicy >> trackReads [
 "Returns true if read tracking is enabled for objects in this security policy.
 Read tracking occurs when an object is faulted into temporary object memory of a session.
 See also DisableObjectReadLogging privilege in UserProfile,
 DynamicDisableObjectReadLogging privilege in UserProfile,
 System class >> setObjectReadTracking: , System class >> objectReadTrackingEnabled,
 Object >> trackRead, and stone config items STN_OBJECT_READ_LOG*  ."

 ^ trackReads == true
]

{ #category : 'Updating' }
GsObjectSecurityPolicy >> trackReads: aBoolean [
"If aBoolean == true, enables read tracking for objects in this security policy.
 If aBoolean == false, disables read tracking for objects in this security policy.

 If aBoolean == true, an error is signalled if any element of AllUsers 
 has a userId not compatible with writing that userId to a .csv file. 
 Read tracking is also subject to the STN_OBJECT_READ_LOG* items in the stone config file,
 the DisableObjectReadLogging privilege in UserProfile,
 the DynamicDisableObjectReadLogging privilege in UserProfile,
 and the method System class >> setObjectReadTracking: . "

 aBoolean _validateClass: Boolean .
 aBoolean ifTrue:[
   AllUsers do:[:upro | UserProfile _validateUserId: upro userId ].
 ].
 trackReads := aBoolean
]

{ #category : 'Accessing Authorization' }
GsObjectSecurityPolicy >> usersWithAuthorization: anAccessSymbol [

"Returns a IdentitySet containing all users with the authorization of
 anAccessSymbol (one of #write, #read, or #none) for the receiver."

| result |

( #( #write #read #none) includesValue: anAccessSymbol)
  ifFalse: [  ^ self _error: #segErrBadAuthKind args: { anAccessSymbol } ].

result := IdentitySet new .
AllUsers do:[:aUserProfile |
  (self authorizationForUser: aUserProfile) == anAccessSymbol ifTrue:[
     result add: aUserProfile
     ].
  ].
^ result

]

{ #category : 'Accessing Authorization' }
GsObjectSecurityPolicy >> worldAuthorization [

"Returns a Symbol that defines whether all system users are authorized to write
 (and read) in this GsObjectSecurityPolicy (#write), to read only (#read), or neither
 (#none)."

^ #( #none #read #write) at: worldAuthorization + 1

]

{ #category : 'Updating Authorization' }
GsObjectSecurityPolicy >> worldAuthorization: anAuthorizationSymbol [

"Redefines the authorization for all users to one of the following
 authorization Symbols:

 * #write (all users can both read and write in this GsObjectSecurityPolicy).

 * #read (read only).

 * #none (neither read nor write permission).

 Generates an error if anAuthorizationSymbol is not one of (#read, #write,
 #none).

 Requires the ObjectSecurityPolicyProtection privilege, if the objectSecurityPolicy is not owned
 by the GemStone UserProfile under which this session is running."

 | authInt |
 authInt := #( #none #read #write ) indexOf: anAuthorizationSymbol .
 authInt == 0 ifTrue:[
   ^ self _error: #segErrBadAuthKind args: { anAuthorizationSymbol }
   ].
 self _authAt: 6 put: (authInt - 1)

]

{ #category : 'Storing and Loading' }
GsObjectSecurityPolicy >> writeTo: aPassiveObject [

"Instances of GsObjectSecurityPolicy cannot be converted to passive form.  This method writes
 nil to aPassiveObject and stops GemStone Smalltalk execution with a notifier."

  aPassiveObject writeObject: nil.
  self _error: #rtErrAttemptToPassivateInvalidObject.
]
