Extension { #name : 'Date' }

{ #category : 'Instance Creation' }
Date class >> _checkFormat: anArray [

"Private.  Verifies that anArray is a valid string-formatting specification for
 the receiver.  Generates an error if it is not."

| v |

anArray _validateClass: Array.
(anArray size < 6)
  ifTrue:[ Date _error: #rtErrBadFormatSpec args: { anArray } ].

"Check for a combination of the digits 1, 2, and 3"
((anArray at: 1) + (anArray at: 2) + (anArray at: 3) == 6 and:
        [(anArray at: 1) * (anArray at: 2) * (anArray at: 3) == 6])
  ifFalse:[ Date _error: #rtErrBadFormatSpec args: { anArray } ].

(anArray at: 4) _validateClass: Character.

((v := anArray at: 5) == 1 or: [v == 2 or: [v == 3]])
  ifFalse:[ Date _error: #rtErrBadFormatSpec args: { anArray } ].

((anArray at: 6) == 1 or: [(anArray at: 6) == 2])
  ifFalse:[ Date _error: #rtErrBadFormatSpec args: { anArray } ].

]

{ #category : 'Instance Creation' }
Date class >> _getMonthFrom: aCharCollection [

"Private.  Returns the SmallInteger that corresponds to the month of the year
 that matches aCharCollection.

 The argument may include any number of characters, but must exactly match a
 legal month name (or the initial portion of that month name).  If the argument
 matches more than one month, the first month matched is used.  The search
 begins with January."

| whichMonth monthArray argSize matchMonth |

monthArray := MonthNames value.
matchMonth:= [:monthStr | | strSize match i |
   i:= 1.
   match:= false.
   strSize:= monthStr size.
   [ ((i <= argSize) and:[i <= strSize]) and:
     [match:= (aCharCollection at: i) isEquivalent: (monthStr at: i)]]
   whileTrue: [
      i:= i + 1.
   ].
   match
].

   argSize:= aCharCollection size.
   whichMonth:= 1.
   [ (whichMonth <= 12) and:
     [(matchMonth value: (monthArray at: whichMonth)) not]
   ]
   whileTrue:
      [whichMonth := whichMonth + 1].

   (whichMonth <= 12)
      ifTrue: [ ^whichMonth].
   ^ 0

]

{ #category : 'Instance Creation' }
Date class >> _newDay: day monthNumber: month year: year [

"Creates and returns an instance of SmallDate if possible, otherwise
 returns an instance of the receiver from the specified values.
 Generates an error if any of the values are out of range.
 If self == Date or self == SmallDate,  result is a SmallDate if possible."

<primitive: 316>
| blk |
blk := [:x :argName | 
  | min max |
  x _isSmallInteger ifFalse:[ 
    ArgumentTypeError new name: argName expectedClass: SmallInteger actualArg: x; signal
  ].
  (x < (min := SmallInteger minimum32bitInteger) 
      or:[ x > (max := SmallInteger maximum32bitInteger)]) ifTrue:[
    OutOfRange new name: argName min: min max: max actual: x ; signal
  ].
 ].
blk value: day value: 'day' .
blk value: month value: 'month' .
blk value: year value: 'year' .
^ self _primitiveFailed: #_newDay:monthNumber:year:
       args: { day . month . year }
]

{ #category : 'Instance Creation' }
Date class >> _today [

"Creates and returns an instance of the receiver from the system calendar
 on the machine that is running the Gem process, which is assumed to
 represent the current date. 
 If self == Date or self == SmallDate,  result is a SmallDate if possible."

<primitive: 315>
^ self _primitiveFailed: #today

]

{ #category : 'Inquiries' }
Date class >> dayOfWeek: dayName [

"Returns a SmallInteger that gives the numeric index of the day of the week
 described by dayName. The index is a number between 1 and 7 inclusive,
 where 1 signifies Sunday. dayName must be the full symbolic name of the
 day. Raises an error if dayName is not a valid week day name."

 "Example: Date dayOfWeek: 'Tuesday'."

| aDay |
aDay := dayName asString asLowercase.
aDay at: 1 put: ((aDay at: 1) asUppercase).
^ WeekDayNames value indexOf: aDay
    ifAbsent: [ dayName _error: #rtErrInvalidArgument
    args: { 'argument is not a week day name' }].

]

{ #category : 'Instance Creation' }
Date class >> fromStream: aStream [

"Creates and returns an instance of the receiver by reading a String from
 aStream.  The String expresses the date in the default format (DD/MM/YYYY).
 Generates an error if the String does not conform to the format."

^ self fromStream: aStream usingFormat: #(1 2 3 $/ 1 1)

]

{ #category : 'Instance Creation' }
Date class >> fromStream: aStream usingFormat: anArray [

"Creates and returns an instance of the receiver by reading a String from
 aStream.  The String expresses the date in the format specified by anArray.
 The expression is terminated either by a space Character or by the end of the
 Stream.  Generates an error if the String does not conform to the format,
 or if anArray contains an incorrect formatting specification.

 See the class documentation of Date for a complete description of the
 String-formatting specification Array.

 If the month format (5th element) indicates either an abbreviation (2) or an
 entire name (3), then this method tries to determine the month by decoding a
 Character substring.  That substring may include any number of characters, but
 must exactly match a legal month name (or the initial portion of that month
 name).  If the substring matches more than one month, the first month matched
 is used (the search begins with January)."

| dayInt monthInt yearInt dateDelim result parseField parseBlocks |

"This block returns a string up from the input stream up to the specified
 delimiter.  If also allows an end-of-file if that parameter is set true.
 It then skips over the delimiter if it is found.
"
parseField := [ :delim :allowEof | | str |
  str := aStream contents class new.
  [ ((aStream peek isEquivalent: delim) not) and:[aStream atEnd not] ]
    whileTrue: [ str add: aStream next ].

  (aStream atEnd) ifTrue:[
     allowEof ifFalse:[ Date _error: #rtErrBadFormat args: { aStream } ].
     ]
  ifFalse:[ aStream next "skip over delimiter" ].
  str
  ].

parseBlocks := {
  "parse day"
  [ :delim | | nextField |
    nextField := parseField value: delim value: (delim == $  ).
    dayInt := Integer fromCompleteString: nextField
  ] .
  "parse month"
  [ :delim | | nextField |
    nextField := parseField value: delim value: (delim == $  ).
    (nextField = '' )
    ifTrue:
      [ Date _error: #rtErrBadFormat args: { aStream }].
    (anArray at: 5) == 1
    ifTrue:
      [ monthInt := Integer fromCompleteString: nextField ]
    ifFalse:
      [ monthInt := self _getMonthFrom: nextField ].
    (monthInt < 1 or:[monthInt > 12])
    ifTrue:
      [ Date _error: #rtErrBadFormat args: { aStream } ]
  ] .
  "parse year"
  [ :delim | | nextField |
    nextField := parseField value: delim value: (delim == $  ).
    yearInt := Integer fromCompleteString: nextField.
    (anArray at: 6) == 2
    ifTrue:
      [ (yearInt > 99)
        ifFalse: [yearInt := yearInt + ((Date today year) // 100 * 100) ]
      ]
  ]
 }.

self _checkReadStream: aStream forClass: CharacterCollection.

Date _checkFormat: anArray.

dateDelim := anArray at: 4.

"parse the date, with day, month, year in the specified format order"
(parseBlocks at:( anArray at: 1)) value: dateDelim .
(parseBlocks at:( anArray at: 2)) value: dateDelim .
(parseBlocks at:( anArray at: 3)) value: $  .

result := self newDay: dayInt monthNumber: monthInt year: yearInt .
^ result

]

{ #category : 'Instance Creation' }
Date class >> fromString: aString [

"Creates and returns an instance of the receiver from the String aString.
 The String expresses the date in the default format (DD/MM/YYYY).
 Generates an error if the String does not conform to the format."

^ self fromString: aString usingFormat: #(1 2 3 $/ 1 1)

]

{ #category : 'Instance Creation' }
Date class >> fromString: aString usingFormat: anArray [

"Creates and returns an instance of the receiver from the String aString.
 The String expresses the date in the format specified by anArray.  The
 expression is terminated either by a space Character or by the end of the
 String.  Generates an error if the String does not conform to the format,
 or if anArray contains an incorrect formatting specification.

 See the class documentation of Date for a complete description of the
 String-formatting specification Array.

 If the month format (5th element) indicates either an abbreviation (2) or an
 entire name (3), then this method tries to determine the month by decoding a
 character substring.  That substring may include any number of Characters, but
 must exactly match a legal month name (or the initial portion of that month
 name).  If the substring matches more than one month, the first month matched
 is used (the search begins with January)."

| s result |

s := ReadStreamPortable on: aString.
result := self fromStream: s usingFormat: anArray.
[ s atEnd ]
whileFalse:
  [ (s next == $  )
    ifFalse:
      [ self _errIncorrectFormat: aString ]
  ].
^ result

]

{ #category : 'Inquiries' }
Date class >> indexOfMonthName: monthName [

"Return an Integer corresponding to the month with the name monthName.
 If monthName is not a recognized name, return 0, otherwise return an
 Integer between 1 and 12 inclusive, where 1 signifies January."

| monthArray |
monthArray := MonthNames value.
1 to: 12 do: [:index | (monthArray at: index) = monthName ifTrue: [^ index]] .
^ 0

]

{ #category : 'Inquiries' }
Date class >> isLeap: year [

"Returns true if year is a leap year; false otherwise."

"a year is a leap year if: (it is evenly divisible by 4 and it is not a
 century year) or (it is a century year and evenly divisible by 400)"

((year \\ 100) == 0)
   ifTrue: [^ ((year \\ 400) == 0)].
^ ((year \\ 4) == 0)

]

{ #category : 'Storing and Loading' }
Date class >> loadFrom: passiveObj [

"Creates and returns an active instance of the receiver from the passive form
 of the object."
| inst yr dy |
yr := passiveObj readObject .
yr  _isSmallInteger ifFalse:[ Error signal:'unexpected ', yr class name ].
dy := passiveObj readObject .
dy  _isSmallInteger ifFalse:[ Error signal:'unexpected ', dy class name ].
self == SmallDate ifTrue:[
  inst := self newDay: dy year: yr .
  inst isSpecial ifFalse:[ Error signal:'a SmallDate should be special'].
] ifFalse:[
  inst := self _basicNew _year: yr day: dy .
  passiveObj hasRead: inst .
].
^ inst

]

{ #category : 'Instance Creation' }
Date class >> migrateNew [

"Override default migrateNew behavior with #_basicNew."

^ self _basicNew

]

{ #category : 'Inquiries' }
Date class >> nameOfMonth: anIndex [

"Returns a String that gives the name, in the user's native language, of the
 month of the year whose numeric index is anIndex.  The index is a number
 between 1 and 12 inclusive, where 1 signifies January."

^ (MonthNames value) at: anIndex.

]

{ #category : 'Instance Creation' }
Date class >> new [

"Disallowed.  To create a new Date, use another instance creation method."

self shouldNotImplement: #new

]

{ #category : 'Instance Creation' }
Date class >> new: anInteger [

"Disallowed.  To create a new Date, use another instance creation method."

self shouldNotImplement: #new:

]

{ #category : 'Instance Creation' }
Date class >> newDay: dayInt month: monthString year: yearInt [

"Creates and returns an instance of the receiver from the specified values.
 Generates an error if any of the values are out of range."

 | monthInt |

 monthInt := self _getMonthFrom: monthString .
 (monthInt < 1 or: [ monthInt > 12 ]) ifTrue:[
   Date _error: #rtErrBadFormat args: { { dayInt . monthString . yearInt } }.
 ].
^ self _newDay: dayInt monthNumber: monthInt year: yearInt .

]

{ #category : 'Instance Creation' }
Date class >> newDay: day monthNumber: month year: year [

(month < 1 or:[ month > 12]) ifTrue: [
  ArgumentError signal: 'Incorrect specified month: ' , month asString
].
(day < 1 or:[ day > (self numberOfDaysIn: month year: year)]) ifTrue: [
  ArgumentError signal: 'Incorrect specified day: ', day asString.
].
^ self _newDay: day monthNumber: month year: year

]

{ #category : 'Instance Creation' }
Date class >> newDay: day year: year [

"Creates and returns an instance of the receiver from the specified values.
 Generates an error if any of the values are out of range."

^ self _newDay: day monthNumber: 1 year: year.

]

{ #category : 'Instance Creation' }
Date class >> numberOfDaysIn: month year: aYear [

| x |
x := #( 31 nil 31 30 31 30 31 31 30 31 30 31 ) atOrNil: month .
x ifNotNil:[ ^ x ] .
month == 2 ifFalse:[ ArgumentError signal: 'Incorrect specified month: ' , month asString ].
(((aYear \\ 100) == 0)
   ifTrue: [ ((aYear \\ 400) == 0)]
   ifFalse: [ ((aYear \\ 4) == 0) ])
  ifTrue: [^ 29].
^ 28

]

{ #category : 'Instance Creation' }
Date class >> today [

  "Returns an instance of the receiver representing the current Date, taking into
   consideration the current GemStone TimeZone.

   See Date class >>_today for 3.2.x and earlier implementation that gets date
   directly from the OS and does not reflect repository TimeZone."

   | parts |
   parts := DateAndTime now asFloatParts.
     "parts is { year. dayOfYear. monthIndex. dayOfMonth. hour. minute. second } "
   ^self newDay: (parts at: 4) monthNumber: (parts at: 3) year: (parts at: 1).

]

{ #category : 'New Indexing Comparison' }
Date >> _classSortOrdinal [

^ 50

]

{ #category : 'Accessing' }
Date >> _daysInMonth: month [

"Returns a SmallInteger that gives the number of days in the month
 specified by the Integer month."

((month == 1) or: [(month == 3) or: [(month == 5) or: [(month == 7) or:
   [(month == 8) or: [(month == 10) or: [(month == 12)]]]]]])
   ifTrue: [^ 31].
((month == 4) or: [(month == 6) or: [(month == 9) or: [(month == 11)]]])
   ifTrue: [^ 30].
(self leap)
   ifTrue: [^ 29].
^ 28

]

{ #category : 'Formatting' }
Date >> _monthAbbrev: anIndex [

"Private.  Returns a three-letter String that gives the abbreviation, in the
 user's native language, of the name of the month whose numeric index is
 anIndex.  The index is a number between 1 and 12 inclusive, where 1 signifies
 January."

|theMonth itsAbbrev|

theMonth := Date nameOfMonth: anIndex.  "get its full name"
itsAbbrev := String new.
1 to: 3 do: "take the first three letters"
   [:aChar | itsAbbrev := itsAbbrev , (theMonth at: aChar)].
^ itsAbbrev

]

{ #category : 'Private' }
Date >> _year: yr day: dy [
  "used by PassiveObject "
   year := yr .
   dayOfYear := dy
]

{ #category : 'Accessing' }
Date >> _yearMonthDay [

"Private.  Returns a three-element Array of SmallIntegers containing the year,
 the index of the month, and the day of the month described by the receiver."

<primitive: 239 >

^ self _primitiveFailed: #_yearMonthDay

]

{ #category : 'Comparing' }
Date >> < aDate [

"Returns true if the receiver represents a date before that of the argument,
 and false if it doesn't.  Generates an error if the argument is not
 a Date."

| argYear myYear |

argYear := aDate year.
(myYear := self year) == argYear
  ifTrue: [ ^ self dayOfYear < aDate dayOfYear ]
  ifFalse: [ ^ myYear < argYear ].

]

{ #category : 'Comparing' }
Date >> = aDate [

"Returns true if the receiver represents the same date as that of the
 argument, and false if it doesn't."

(self == aDate) ifTrue: [ ^ true ].
(aDate isKindOf: Date ) ifFalse: [ ^false ].
^ (self year == aDate year) and: [self dayOfYear == aDate dayOfYear]

]

{ #category : 'Comparing' }
Date >> > aDate [

"Returns true if the receiver represents a date after that of the argument,
 and false if it doesn't.  Generates an error if the argument is not
 a Date."

| argYear myYear |

argYear := aDate year.
(myYear := self year) == argYear
  ifTrue: [ ^ self dayOfYear > aDate dayOfYear ]
  ifFalse: [^  myYear > argYear ].

]

{ #category : 'Arithmetic' }
Date >> addDays: anInteger [

"Returns a Date that describes a date anInteger days later than that
 of the receiver."

^ (self class) newDay: (self dayOfYear + anInteger) year: self year.

]

{ #category : 'Arithmetic' }
Date >> addMonths: anInteger [

"Returns a Date that describes a date anInteger months later than that of the
 receiver.

 This method attempts to keep the day of the month the same.  If the
 new month has fewer days than the receiver's original month, then it
 truncates to the last day of the new month."

| yr month day newYear newMonth newDay newDate generatedDay |

yr := self year.
month := self month.
day := self day.

newMonth := month + anInteger.
newYear := yr + ((newMonth - 1) // 12).
newMonth := (newMonth - 1) \\ 12 + 1.
newDate := self class _newDay: day monthNumber: newMonth year: newYear.
generatedDay := newDate day.
(generatedDay ~= day)
  ifTrue: [
    newDay := newDate _daysInMonth: newMonth.
    newDate := self class _newDay: newDay monthNumber: newMonth year: newYear
    ].
^ newDate.

]

{ #category : 'Arithmetic' }
Date >> addYears: anInteger [

"Returns a Date that describes a date anInteger years later than that of the
 receiver."

| yr month day newYear newDay newDate generatedDay |

yr := self year.
month := self month.
day := self day.

newYear := yr + anInteger.
newDate := self class _newDay: day monthNumber: month year: newYear.
generatedDay := newDate day.
(generatedDay ~= day)
  ifTrue: [
    newDay := newDate _daysInMonth: month.
    newDate := self class _newDay: newDay monthNumber: month year: newYear
    ].
^ newDate.

]

{ #category : 'Converting' }
Date >> asCanonicalForm [
	"Answer self, or, if I am a Date with an equivalent
	SmallDate, answer that SmallDate."

	| cls res |
	(cls := self class) ~~ SmallDate
		ifTrue: [ 
			cls == Date
				ifTrue: [ 
					res := cls _newDay: self day monthNumber: self month year: self year.
					res class == SmallDate
						ifTrue: [ ^ res ] ] ].
	^ self
]

{ #category : 'Converting' }
Date >> asDays [

"Returns an Integer that represents the receiver in units of days since
 January 1, 1901."

| numYears numDays |

numYears := self year - 1901.
numDays := (numYears * 365) + (numYears // 4) +
           ((numYears + 300) // 400) - (numYears // 100) + self dayOfYear - 1.
^ numDays.

]

{ #category : 'Formatting' }
Date >> asString [

"Returns a String that expresses the receiver in the default format
 (DD/MM/YYYY)."

| t result |

t := self _yearMonthDay.
result := (t at: 3) _digitsAsString .
result addAll: '/';
  addAll: (t at: 2) _digitsAsString;
  addAll: '/';
  addAll: (t at: 1) _digitsAsString.
^ result

]

{ #category : 'Formatting' }
Date >> asStringUsingFormat: anArray [

"Returns a String that expresses the receiver in the format defined by anArray.
 Generates an error if anArray contains an incorrect formatting specification.

 See the class documentation of Date for a complete description of the
 String-formatting specification Array."

| t dateSeparator monthName aString day yearNumber |

t := self _yearMonthDay.

Date _checkFormat: anArray.

dateSeparator := (anArray at: 4) asString.

((anArray at: 5) == 2) "get the month name according to the format"
   ifTrue: [monthName := self _monthAbbrev: (t at: 2)]
   ifFalse: [((anArray at: 5) == 3) "month as number is default"
      ifTrue: [monthName := Date nameOfMonth: (t at: 2)]
      ifFalse: [monthName := (t at: 2) _digitsAsString]].

((anArray at: 6) == 2)
   ifTrue: [yearNumber := ((t at: 1) \\ 100) _digitsAsString]
   ifFalse: [yearNumber := (t at: 1) asString].  "YYYY is default"

day := (t at:3) _digitsAsString.
((anArray at: 1) == 2) "month first"
   ifTrue: [aString := monthName , dateSeparator]
   ifFalse: [((anArray at: 1) == 3) "yearNumber first"
      ifTrue: [aString := yearNumber , dateSeparator]
      ifFalse: [aString := day , dateSeparator]].  "day first is default"

((anArray at: 2) == 1) "day second"
   ifTrue: [aString addAll: day; addAll: dateSeparator] "yearNumber second"
   ifFalse: [((anArray at: 2) == 3) "month second is default"
      ifTrue: [aString addAll: yearNumber; addAll: dateSeparator]
      ifFalse: [aString addAll: monthName; addAll: dateSeparator]].

((anArray at: 3) == 1) "day third"
   ifTrue: [aString addAll: day]
   ifFalse: [((anArray at: 3) == 2) "month third"
      ifTrue: [aString addAll: monthName]
      ifFalse: [aString addAll: yearNumber]].  "yearNumber third is default"

^ aString

]

{ #category : 'Accessing' }
Date >> at: anIndex put: aValue [

"Disallowed.  You may not change the value of a Date."

self shouldNotImplement: #at:put:

]

{ #category : 'Accessing' }
Date >> day [

"Returns a SmallInteger that gives the day of the month described by the
 receiver."

^  (self _yearMonthDay) at: 3

]

{ #category : 'Accessing' }
Date >> dayOfMonth [

"Returns a SmallInteger that gives the day of the month described by the
 receiver."

^ self day.

]

{ #category : 'Accessing' }
Date >> dayOfWeek [

"Returns a SmallInteger that gives the numeric index of the day of the week
 described by the receiver.  The index is a number between 1 and 7 inclusive,
 where 1 signifies Sunday."

^ self julianDay - 2299295 - 1 \\ 7 + 1

]

{ #category : 'Accessing' }
Date >> dayOfYear [

"Returns a SmallInteger that gives the day of the year described by the
 receiver."

^ dayOfYear

]

{ #category : 'Accessing' }
Date >> daysInMonth [

"Returns a SmallInteger that gives the number of days in the month
 described by the receiver."

^ self _daysInMonth: self month


]

{ #category : 'Accessing' }
Date >> daysInYear [

"Returns a SmallInteger that gives the number of days in the year
 described by the receiver."

(self leap) ifTrue: [^ 366].
^ 365

]

{ #category : 'Comparing' }
Date >> hash [

"Returns an Integer hash code for the receiver."

^ ((self year hash) bitShift: -1) bitXor: (self dayOfYear hash)

]

{ #category : 'Accessing' }
Date >> julianDay [

"Returns the Julian Day of the receiver, a SmallInteger that gives the number of
 days since January 1, 4713 B.C., as defined in Communications of the ACM,
 algorithm #199."

<primitive: 46 >

^ self _primitiveFailed: #julianDay

]

{ #category : 'Accessing' }
Date >> leap [

"Returns true if the receiver describes a leap year and false if it does not."

| yr |
  "a year is a leap year if: (it is evenly divisible by 4 and it is not a
   century year) or (it is a century year and evenly divisible by 400)"

yr := self year .
((yr \\ 100) == 0)
   ifTrue: [^ ((yr \\ 400) == 0)].
^ ((yr \\ 4) == 0)

]

{ #category : 'Accessing' }
Date >> month [

"Returns a SmallInteger that gives the numeric index of the month of the year
 described by the receiver.  The index is a number between 1 and 12 inclusive,
 where 1 signifies January."

^ (self _yearMonthDay) at: 2

]

{ #category : 'Accessing' }
Date >> monthIndex [

"Returns a SmallInteger that gives the numeric index of the month of the year
 described by the receiver.  The index is a number between 1 and 12 inclusive,
 where 1 signifies January."

^ (self _yearMonthDay) at: 2

]

{ #category : 'Accessing' }
Date >> monthName [

"Returns a String that gives the name of the month of the year described by the
 receiver, in the user's native language."

^ MonthNames value at: self month

]

{ #category : 'Accessing' }
Date >> monthOfYear [

"Returns a SmallInteger that gives the numeric index of the month of the year
 described by the receiver.  The index is a number between 1 and 12 inclusive,
 where 1 signifies January."

^ self month.

]

{ #category : 'Inquiries' }
Date >> next: dayName [
"Returns the next date whose weekday name is dayName"

^ self addDays:
   ((self class dayOfWeek: dayName) - 1 - self weekdayIndex - 1) \\ 7 + 1

]

{ #category : 'Inquiries' }
Date >> nextMonth: monthName [
"Returns the next Date which describes a date later than the receiver and has
 month named monthName."

| tmp index |
index := self class indexOfMonthName: monthName asString .
^ self month == index
    ifTrue: [self addYears: 1]
   ifFalse: [tmp := self.
             [tmp month == index]
                whileFalse: [tmp := tmp addMonths: 1].
              tmp]

]

{ #category : 'Inquiries' }
Date >> previous: dayName [

"Returns the previous date whose weekday name is dayName."

^ self subtractDays: self dayOfWeek -
       (self class dayOfWeek: dayName) -
       1 \\ 7 + 1.

]

{ #category : 'Inquiries' }
Date >> previousMonth: monthName [
"Returns the next Date which describes a date earlier than the receiver and has
 month named monthName."

| tmp index |
index := self class indexOfMonthName: (monthName asString).
^ self month == index
    ifTrue: [self subtractYears: 1]
    ifFalse: [tmp := self.
              [tmp month == index]
                 whileFalse: [tmp := tmp subtractMonths: 1].
               tmp]

]

{ #category : 'Formatting' }
Date >> printJsonOn: aStream [
  (self asStringUsingFormat: #( 3 2 1 $- 1 1)) printJsonOn: aStream 

]

{ #category : 'Formatting' }
Date >> printOn: aStream [

"Puts a displayable representation of the receiver on aStream."

aStream nextPutAll: self asString .

]

{ #category : 'Accessing' }
Date >> size: anInteger [

"Disallowed.  You may not change the size of a Date."

self shouldNotImplement: #size:

]

{ #category : 'Arithmetic' }
Date >> subtractDate: aDate [

"Returns a positive Integer that counts the number of times midnight occurs
 between the times described by the receiver and aDate."

^ (self asDays - aDate asDays) abs

]

{ #category : 'Arithmetic' }
Date >> subtractDays: anInteger [

"Returns a Date that describes a date anInteger days earlier than that
 of the receiver."

^ self class newDay: self dayOfYear - anInteger year: self year

]

{ #category : 'Arithmetic' }
Date >> subtractMonths: anInteger [

"Returns a Date that describes a date anInteger months earlier than that of the
 receiver.

 This method attempts to keep the day of the month the same.  If the
 new month has fewer days than the receiver's original month, then it
 truncates to the last day of the new month."

^ self addMonths: (anInteger negated).

]

{ #category : 'Arithmetic' }
Date >> subtractYears: anInteger [

"Returns a Date that describes a date anInteger years earlier than that of the
 receiver."

^ self addYears: (anInteger negated).

]

{ #category : 'Formatting' }
Date >> USDateFormat [

"Returns a String that expresses the date of the receiver. The date is in
 United States format, month first (MM/DD/YY)."

^self asStringUsingFormat: #(2 1 3 $/ 1 2 $: false )

]

{ #category : 'Inquiries' }
Date >> weekdayIndex [
"Return a SmallInteger between 1 and 7 representing the day of the week of the
 receiver where Monday is 1 and Sunday is 7."

^ (self asDays + 1) \\ 7 + 1

]

{ #category : 'Accessing' }
Date >> weekDayName [

"Returns a String that gives the name of the day of the week described by the
 receiver, in the user's native language."

^ WeekDayNames value at: (self dayOfWeek)

]

{ #category : 'Storing and Loading' }
Date >> writeTo: passiveObj [

"Writes the passive form of the receiver into passiveObj."

passiveObj writeClass: self class.
self year writeTo: passiveObj.
self dayOfYear writeTo: passiveObj.
passiveObj space

]

{ #category : 'Accessing' }
Date >> year [

"Returns a SmallInteger that gives the year described by the receiver."

^ year

]
