Extension { #name : 'Collection' }

{ #category : 'Instance Creation' }
Collection class >> with: aValue [

"Returns an instance of the receiver containing the argument."

| inst |
inst := self new.
inst add: aValue.
^inst

]

{ #category : 'Instance Creation' }
Collection class >> with: aValue with: val2 [

"Returns an instance of the receiver containing the arguments."

| inst |
inst := self new.
inst add: aValue; add: val2.
^inst

]

{ #category : 'Instance Creation' }
Collection class >> with: aValue with: val2 with: val3 [

"Returns an instance of the receiver containing the arguments."

| inst |
inst := self new.
inst add: aValue; add: val2; add: val3.
^inst

]

{ #category : 'Instance Creation' }
Collection class >> with: aValue with: val2 with: val3 with: val4 [

"Returns an instance of the receiver containing the arguments."

| inst |
inst := self new.
inst add: aValue; add: val2; add: val3; add: val4.
^inst

]

{ #category : 'Instance Creation' }
Collection class >> withAll: aCollection [

"Returns an instance of the receiver containing the elements of the argument."

| result |

result:= self new.
result addAll: aCollection.
^result

]

{ #category : 'Private' }
Collection >> _asCollectionForSorting [

"Redefine this method for classes that understand _at: or have different
 semantics for the do: method."

| result |

result := { } .
self accompaniedBy: result do: [:res :element | res add: element ].
^ result.

]

{ #category : 'Private' }
Collection >> _asIdentityBag [

"Returns the receiver."

^ self asIdentityBag

]

{ #category : 'Private' }
Collection >> _deepCopyWith: copiedObjDict [

| copy myClass |

copy := copiedObjDict at: self otherwise: nil.
copy ifNotNil: [ ^ copy ].

myClass := self class.
copy := myClass new.
copiedObjDict at: self put: copy.

self _deepCopyNamedIvsWith: copiedObjDict to: copy .

1 to: self _basicSize do: [ :n | | anElement |
  anElement := self _at: n.
  copy add: (anElement _deepCopyWith: copiedObjDict)
].
^ copy.

]

{ #category : 'Error Handling' }
Collection >> _errorNotFound: anObject [

"Sends an error message indicating that the expected object was not found."

^ self _error: #objErrNotInColl args: { anObject }

]

{ #category : 'Private' }
Collection >> _findRangeIndexWithPath: pathArray [

"Redefine this method for classes that support indexes."

^ nil

]

{ #category : 'Private' }
Collection >> _indexedPaths [

"Returns the indexed paths for the receiver.  This is a private method that
 should only be used for sorting/indexing."

"This method should be reimplemented for classes where _indexedPaths is
 defined."

^ nil

]

{ #category : 'Sorting' }
Collection >> _sortPaths: thePaths directions: theBooleans [

""

^ self _sortPaths: thePaths directions: theBooleans  persistentRoot: nil

]

{ #category : 'Sorting' }
Collection >> _sortPaths: thePaths directions: theBooleans persistentRoot: persistentArrayOrNil [
  ""

  | thePath pathArrays indexObjs result |
  pathArrays := Array new: thePaths size.	" do some error checking "
  1 to: thePaths size do: [ :i |
    thePath := thePaths at: i.
    thePath size > 1024
      ifTrue: [
        " check the string size "
        ^ thePath _error: #'assocErrPathTooLong' ].
    (thePath includesValue: $*)
      ifTrue: [
        " check if sorting over set-valued instance variable (not allowed) "
        ^ self _error: #'rtErrBagInvalidSortSpecification' args: {thePath} ].
    pathArrays at: i put: thePath asArrayOfPathTerms ].
  indexObjs := self _indexObjectsFor: pathArrays.
  result := (indexObjs at: 1)
    _sortOn: indexObjs
    directions: theBooleans
    persistentRoot: persistentArrayOrNil.
  ^ result

]

{ #category : 'Private' }
Collection >> _validateNotEmpty [

^ self isEmpty
    ifTrue:[ self _error: #objErrCollectionEmpty]
    ifFalse:[ true ]

]

{ #category : 'Comparing' }
Collection >> = aCollection [

"Returns true if all of the following conditions are true:

 1.  The receiver and aCollection are of the same class.
 2.  The two collections are of the same size.
 3.  The corresponding elements of the receiver and aCollection are equal."

(self == aCollection)
  ifTrue: [ ^ true ].

(self class == aCollection class)
  ifFalse: [ ^ false ].

(self size == aCollection size)
  ifFalse: [ ^ false ].

self do: [ :anElement |
  (aCollection includes: anElement)
    ifFalse: [ ^ false ].
  ].

^ true.

]

{ #category : 'Adding' }
Collection >> add: newObject [

"Makes newObject one of the receiver's elements and returns newObject."

Collection subclassResponsibility: #add:

]

{ #category : 'Adding' }
Collection >> addAll: aCollection [

"Adds all of the elements of aCollection to the receiver and returns
 aCollection."

(self == aCollection) ifTrue: [ ^ self addAll: (aCollection copy) ].
aCollection accompaniedBy: self do: [ :me :each | me add: each ].
^ aCollection.

]

{ #category : 'Enumerating' }
Collection >> allSatisfy: aBlock [

	"Return true if <aBlock> evaluates to true for every element of the receiver.
	Return true if the receiver is empty. Otherwise return false."

	self do: [:each |
		(aBlock value: each) ifFalse: [^false].
	].
	^true.

]

{ #category : 'Enumerating' }
Collection >> any [
	"Return an arbitrary element of the receiver. Error if the receiver is empty."

	self do: [:each | ^each].
	self _error: #assocErrNoElementsDetected.

]

{ #category : 'Enumerating' }
Collection >> anySatisfy: aBlock [

        "Return true if <aBlock> evaluates to true for any element of the receiver.
        Otherwise return false. Return false if the receiver is empty."

	self do: [:each |
		(aBlock value: each) ifTrue: [^true].
	].
	^false.

]

{ #category : 'Converting' }
Collection >> asArray [

"Returns an Array with the contents of the receiver."

^Array withAll: self

]

{ #category : 'Converting' }
Collection >> asBag [

"Returns a Bag with the contents of the receiver."

^Bag withAll: self

]

{ #category : 'Converting' }
Collection >> asByteArray [

"Returns an Array with the contents of the receiver."

^ByteArray withAll: self

]

{ #category : 'Converting' }
Collection >> asIdentityBag [

"Returns an IdentityBag with the contents of the receiver."

^IdentityBag withAll: self

]

{ #category : 'Converting' }
Collection >> asIdentitySet [

"Returns an IdentitySet with the contents of the receiver."

^IdentitySet withAll: self

]

{ #category : 'Converting' }
Collection >> asOrderedCollection [

"Returns an OrderedCollection with the contents of the receiver."

^OrderedCollection withAll: self

]

{ #category : 'Converting' }
Collection >> asSet [

"Returns a Set with the contents of the receiver."

^Set withAll: self

]

{ #category : 'Converting' }
Collection >> asSortedCollection [

"Returns a SortedCollection with the contents of the receiver."

^SortedCollection withAll: self

]

{ #category : 'Converting' }
Collection >> asSortedCollection: sortBlock [

"Returns a SortedCollection with the contents of the receiver, using the
 given sort block."

| coll |
coll := SortedCollection sortBlock: sortBlock.
coll addAll: self.
^coll

]

{ #category : 'Converting' }
Collection >> asSortedOrderedCollection [

"Returns an OrderedCollection that has been sorted with a SortedCollection
 and having the contents of the receiver."

^OrderedCollection withAll: (SortedCollection withAll: self)

]

{ #category : 'Clustering' }
Collection >> clusterDepthFirst [

"This method clusters the receiver and its named and unnamed instance variables
 in depth-first order.  Returns true if the receiver has already been clustered
 during the current transaction; returns false otherwise."

self cluster
  ifTrue: [ ^ true ]
  ifFalse:[
    1 to: self namedSize do: [ :i | (self instVarAt: i) clusterDepthFirst ].
    self do: [ :each | each clusterDepthFirst ].
    ^ false
    ].

]

{ #category : 'Enumerating' }
Collection >> collect: aBlock [

"Evaluates aBlock with each of the receiver's elements as the argument.
 Collects the resulting values into a collection of class specified by
 sending #speciesForCollect message to the receiver. Returns the new
 collection. The argument aBlock must be a one-argument block.

 For SequenceableCollections, the result preserves the ordering of the
 receiver.  That is, if element a comes before element b in the receiver,
 then element a is guaranteed to come before b in the result."

|result|

result := self speciesForCollect new .
self accompaniedBy: result do: [ :res :each | res add: (aBlock value: each) ].
^ result

]

{ #category : 'Enumerating' }
Collection >> detect: aBlock [

"Evaluates aBlock repeatedly, with elements of the receiver as the argument.
 Returns the first element for which aBlock evaluates to true.  If none of the
 receiver's elements evaluates to true, generates an error.  The argument
 aBlock must be a one-argument block."

^ self detect: aBlock
       ifNone: [^ self _error: #assocErrNoElementsDetected args: { aBlock }]

]

{ #category : 'Enumerating' }
Collection >> detect: aBlock ifNone: exceptionBlock [

"Evaluates aBlock repeatedly, with elements of the receiver as the argument.
 Returns the first element for which aBlock evaluates to true.  If none of the
 receiver's elements evaluates to true, this evaluates the argument
 exceptionBlock and returns its value.  The argument aBlock must be a
 one-argument block, and exceptionBlock must be a zero-argument block."

self do: [:each| (aBlock value: each) ifTrue: [^each]].
^exceptionBlock value

]

{ #category : 'Enumerating' }
Collection >> do: aBlock [

"Evaluates the one-argument block aBlock using each element of the
 receiver in order.  Returns the receiver."

"Must be reimplemented in subclasses such as Dictionary which have different
 format or implementation."

1 to: self size do: [:i |
  aBlock value: (self at: i)
].
^ self

]

{ #category : 'Enumerating' }
Collection >> do: operationBlock separatedBy: aBlock [
	"Evaluates the one-argument block operationBlock using each element of the
	 receiver in order.  aBlock is evaluated between successive elements.
	 Returns the receiver. "

	| index |
	index := 0.
	self do:
			[:each |
			(index := index + 1) > 1 ifTrue: [aBlock value].
			operationBlock value: each].
	^self

]

{ #category : 'Error Handling' }
Collection >> errorDifferentSizeCollections [

"Reports an error indicating that the size of the receiver collection is
 different from the size of the argument collection."

^ self _error: #objErrDiffSizeColl

]

{ #category : 'Error Handling' }
Collection >> errorInvalidArgClass: argument classes: classArray [

"Reports an error indicating that the class of argument is not one of those
 specified in classArray."

^ self _error: #rtErrInvalidArgClass args: { argument . classArray }.

]

{ #category : 'Repository Conversion' }
Collection >> getIndexInfo [
  "Returns indexing information for the receiver."

  ^ {}

]

{ #category : 'Comparing' }
Collection >> hash [

"Returns a numeric hash key for the receiver."

| hashValue |

hashValue := (97133 bitXor: (self size)) bitXor: (self class asOop).
"For large collections, the hash value is just a function of its size
 and class"
(self size > 64) ifTrue: [ ^ hashValue abs ].

self do: [ :anElement |
  (anElement isKindOf: Collection)
    ifTrue: [ hashValue := hashValue bitXor: anElement size ]
    ifFalse: [ hashValue := hashValue bitXor: anElement hash ].
  ].
^ hashValue abs.

]

{ #category : 'Searching' }
Collection >> identicalOccurrencesOf: anObject [

"Returns the number of the receiver's elements that are identical (==) to
 anObject."

| result |

result := 0.
self do: [ :element |
  (anObject == element) ifTrue: [ result := result + 1 ]
  ].
^result

]

{ #category : 'Searching' }
Collection >> includes: anObject [

"Returns true if anObject is equal to one of the elements of the receiver.
 Returns false otherwise."

self do: [ :element | (anObject = element) ifTrue: [ ^true ]].
^false

]

{ #category : 'Searching' }
Collection >> includesIdentical: anObject [

"Returns true if anObject is identical to one of the elements of the receiver.
 Returns false otherwise."

self do: [ :element | (anObject == element) ifTrue: [ ^true ]].
^false

]

{ #category : 'Searching' }
Collection >> includesValue: anObject [

"Returns true if anObject is equal to one of the elements of the receiver.
 Returns false otherwise."

"Note: This method is the same as #includes:"

^ self includes: anObject.

]

{ #category : 'Enumerating' }
Collection >> inject: aValue into: aBlock [

"Accumulates a running value associated with evaluating the argument,
 aBlock, with the current value and the each element of the receiver
 as block arguments.  The initial value is the value of the argument, aValue.
 For example: total := #(1 2 3 4) inject: 0 into: [:sum:int | sum + int]"

| val |

val := aValue.
self do: [:element | val := aBlock value: val value: element ].
^val

]

{ #category : 'Testing' }
Collection >> isEmpty [

"Returns true if the receiver is empty.  Returns false otherwise."

^self size == 0

]

{ #category : 'Testing' }
Collection >> notEmpty [

"Returns true if the receiver is not empty.  Returns false otherwise."

^self size ~~ 0

]

{ #category : 'Searching' }
Collection >> occurrencesOf: anObject [

"Returns the number of the receiver's elements that are equal to anObject."

| count |

count := 0.
self do: [ :element| (anObject = element) ifTrue: [count := count + 1]].
^count.

]

{ #category : 'Json' }
Collection >> printJsonOn: aStream [

	| delimiter |
	delimiter := ''.
	aStream nextPut: $[.
	self do: [:each |
		aStream nextPutAll: delimiter.
		each printJsonOn: aStream.
		delimiter := ','.
	].
	aStream nextPut: $].

]

{ #category : 'Formatting' }
Collection >> printNonRecursiveRepresentationOn: aStream recursionSet: anIdentitySet [
	"Put a displayable representation of the receiver on the given stream
	 while avoiding recursion from object reference loops."

	| count sz |
	super printOn: aStream recursionSet: anIdentitySet.
	aStream nextPutAll: '( ' .
	count := 1 .
	sz := self size .
	self do:[:anElement |
	  anElement printOn: aStream recursionSet: anIdentitySet.
	  aStream isFull ifTrue:[
		  "prevent infinite recursion when printing cyclic structures, and
		   limit the size of result when printing large collections."
		  aStream _nextPut:( (aStream _collection endsWith: '...') ifTrue:[ $) ] ifFalse:[ ' ...)' ]).
		  ^ self
		] .
	  count < sz ifTrue:[ aStream nextPutAll: ', ' ].
	  count := count + 1 .
	].
	aStream nextPut: $) .

]

{ #category : 'Formatting' }
Collection >> printOn: aStream [

"Puts a displayable representation of the receiver on the given stream."

^ self printNonRecursiveOn: aStream

]

{ #category : 'Formatting' }
Collection >> printString [
  | ws str |
  str := String new.
  ws := PrintStream printingOn: str maxSize: 700 . "fix 51281"
  self printOn: ws.
   "contents might have been converted to another subclass of CharacterCollection."
  ^ ws _collection
]

{ #category : 'Hashing' }
Collection >> rehash [
	"Re-establish any hash invariants of the receiver."

]

{ #category : 'Enumerating' }
Collection >> reject: aBlock [

"Evaluates aBlock with each of the receiver's elements as the argument.  Stores
 the values for which aBlock is false into a collection of the same class as
 the receiver, and returns the new collection.  The argument aBlock must be a
 one-argument block.

 For SequenceableCollections, the result preserves the ordering of the
 receiver.  That is, if element a comes before element b in the receiver,
 then element a is guaranteed to come before b in the result."

|result|
result := self species new.
self do: [:each|
  (aBlock value: each) ifFalse: [result add: each]
  ].
^ result

]

{ #category : 'Removing' }
Collection >> remove: oldObject [

"Removes from the receiver an object that is equivalent to oldObject and
 returns oldObject.  If several elements of the receiver are equivalent to
 oldObject, only one instance is removed.  If oldObject has no equivalent
 elements in the receiver, raises an error."

^ self remove: oldObject ifAbsent: [self _errorNotFound: oldObject]

]

{ #category : 'Removing' }
Collection >> remove: oldObject ifAbsent: anExceptionBlock [

"Removes from the receiver an object that is equivalent to oldObject and
 returns oldObject.  If several elements of the receiver are equivalent to
 oldObject, only one instance is removed.  If oldObject has no equivalent
 elements in the receiver, anExceptionBlock is evaluated and the result of the
 evaluation is returned."

^ self subclassResponsibility: #remove:ifAbsent:

]

{ #category : 'Removing' }
Collection >> removeAll: aCollection [

"For each element in aCollection, removes from the receiver one element that is
 equivalent to the element in aCollection.  Returns aCollection if successful."

aCollection == self ifTrue:[
  self size: 0 .
  ^ aCollection
  ].
aCollection accompaniedBy: self do: [ :me :element | me remove: element ].
^ aCollection.

]

{ #category : 'Removing' }
Collection >> removeAllIdentical: aCollection [

"For each element in aCollection, removes from the receiver one element that is
 identical to the element in aCollection.  Returns aCollection if successful."

aCollection accompaniedBy: self do: [:me :element | me removeIdentical: element ].
^ aCollection.

]

{ #category : 'Removing' }
Collection >> removeAllPresent: aCollection [

"Removes from the receiver one occurrence of each element of aCollection that is
 also an element of the receiver.  Differs from removeAll: in that, if some
 elements of aCollection are not present in the receiver, no error is generated.
 Returns aCollection."

aCollection do: [ :anElement |
  self removeIfPresent: anElement
  ].

^ aCollection

]

{ #category : 'Removing' }
Collection >> removeAllSuchThat: aBlock [
   "Remove all elements of the receiver for which aBlock returns true.
    Answer the removed elements.
    This is a fallback implementation. It is not efficient."

  | toBeRemoved |
  toBeRemoved := self select: aBlock.
  self removeAllIdentical: toBeRemoved.
  ^ toBeRemoved
]

{ #category : 'Removing' }
Collection >> removeIdentical: oldObject [

"Removes from the receiver an object that is identical to oldObject, and
 returns oldObject.  If several elements of the receiver are identical to
 oldObject, only one instance is removed.  If oldObject is not present in
 the receiver, raises an error."

^ self removeIdentical: oldObject ifAbsent: [self _errorNotFound: oldObject]

]

{ #category : 'Removing' }
Collection >> removeIdentical: oldObject ifAbsent: anExceptionBlock [

"Removes from the receiver an object that is identical to oldObject and
 returns oldObject.  If several elements of the receiver are identical to
 oldObject, only one instance is removed.  If oldObject is not present in
 the receiver, anExceptionBlock is evaluated and the result of the
 evaluation is returned."

^ self subclassResponsibility: #removeIdentical:ifAbsent:

]

{ #category : 'Enumerating' }
Collection >> select: aBlock [

"Evaluates aBlock with each of the receiver's elements as the argument.
 Stores the values for which aBlock is true into a collection of the
 same class as the receiver, and returns the new collection.
 The argument aBlock must be a one-argument block.

 For SequenceableCollections, the result preserves the ordering of the
 receiver.  That is, if element a comes before element b in the receiver,
 then element a is guaranteed to come before b in the result.

 The new collection that this method returns does not retain any indexes of the
 receiver.  If you want to perform indexed selections on the new collection,
 you must build all of the necessary indexes.  For more information, see the
 GemStone Programming Guide."

| result |

result:= self species new.
self do: [ :each | (aBlock value: each) ifTrue: [result add: each]].
^result

]

{ #category : 'Sorting' }
Collection >> sortAscending [

"Returns an Array containing the elements of the receiver sorted in ascending
 order."

^ self sortAscending: '' persistentRoot: nil.

]

{ #category : 'Sorting' }
Collection >> sortAscending: aSortSpec [

"Returns an Array containing the elements of the receiver, sorted in ascending
 order, as determined by the values of the instance variables represented by
 aSortSpec.  The argument aSortSpec must be either a String representing a
 single path, or an Array holding up to 16 such Strings (each representing a
 path).  If aSortSpec is an Array, the first path in the Array is the primary
 sort key, and the remaining paths are taken in order as subordinate keys.

 Each path in aSortSpec must follow the rules for equality indexes.
 In addition, if any path in aSortSpec is an empty path (that is, a zero-length
 String), the sort is performed upon the elements of the receiver itself,
 rather than upon the instance variables of those elements."

^self sortAscending: aSortSpec persistentRoot: nil

]

{ #category : 'Sorting' }
Collection >> sortAscending: aSortSpec persistentRoot: persistentArrayOrNil [

"Returns an Array containing the elements of the receiver, sorted in ascending
 order, as determined by the values of the instance variables represented by
 aSortSpec.  The argument aSortSpec must be either a String representing a
 single path, or an Array holding up to 16 such Strings (each representing a
 path).  If aSortSpec is an Array, the first path in the Array is the primary
 sort key, and the remaining paths are taken in order as subordinate keys.

 Each path in aSortSpec must follow the rules for equality indexes.
 In addition, if any path in aSortSpec is an empty path (that is, a zero-length
 String), the sort is performed upon the elements of the receiver itself,
 rather than upon the instance variables of those elements."

| arg |

arg := { } .

(aSortSpec isKindOf: CharacterCollection)
  ifTrue: [ arg add: aSortSpec ; add: #ASCENDING ]
  ifFalse: [
     aSortSpec accompaniedBy: arg do: [ :aArg :each | aArg add: each ; add: #ASCENDING ]
  ].

^ self sortWith: arg persistentRoot: persistentArrayOrNil

]

{ #category : 'Sorting' }
Collection >> sortDescending [

"Returns an Array containing the elements of the receiver sorted in descending
 order."

^ self sortDescending: '' persistentRoot: nil.

]

{ #category : 'Sorting' }
Collection >> sortDescending: aSortSpec [

"Returns an Array containing the elements of the receiver, sorted in descending
 order, as determined by the values of the instance variables represented by
 aSortSpec.  The argument aSortSpec must be either a String representing a
 single path, or an Array holding up to 16 such Strings (each representing a
 path).  If aSortSpec is an Array, the first path in the Array is the primary
 sort key, and the remaining paths are taken in order as subordinate keys.

 Each path in aSortSpec must follow the rules for equality indexes.
 In addition, if any path in aSortSpec is an empty path (that is, a zero-length
 String), the sort is performed upon the elements of the receiver itself,
 rather than upon the instance variables of those elements."

^ self sortDescending: aSortSpec persistentRoot: nil

]

{ #category : 'Sorting' }
Collection >> sortDescending: aSortSpec persistentRoot: persistentArrayOrNil [

"Returns an Array containing the elements of the receiver, sorted in descending
 order, as determined by the values of the instance variables represented by
 aSortSpec.  The argument aSortSpec must be either a String representing a
 single path, or an Array holding up to 16 such Strings (each representing a
 path).  If aSortSpec is an Array, the first path in the Array is the primary
 sort key, and the remaining paths are taken in order as subordinate keys.

 Each path in aSortSpec must follow the rules for equality indexes.
 In addition, if any path in aSortSpec is an empty path (that is, a zero-length
 String), the sort is performed upon the elements of the receiver itself,
 rather than upon the instance variables of those elements."

| arg |

arg := { } .

(aSortSpec isKindOf: CharacterCollection)
  ifTrue: [ arg add: aSortSpec ; add: #DESCENDING ]
  ifFalse: [
    aSortSpec accompaniedBy: arg do: [:aArg :each | aArg add: each ; add: #DESCENDING ]
  ].

^ self sortWith: arg persistentRoot: persistentArrayOrNil

]

{ #category : 'Sorting' }
Collection >> sortWith: aSortPairArray [

"Returns an Array containing the elements of the receiver, sorted according to
 the contents of aSortPairArray.  The argument aSortPairArray is an Array of
 Strings that represent path/direction pairs, in the following form:

 aCollection sortWith: #('a.b' 'ASCENDING' 'a.c' 'DESCENDING' ...)

 That Array may contain up to 16 path/direction pairs.  The first path in the
 Array is the primary sort key, and the remaining paths are taken in order as
 subordinate keys.

 In aSortPairArray, each path String must follow the rules for equality
 indexes.  Each direction String must be either 'ASCENDING' or
 'DESCENDING' (case-insensitive); otherwise, an error is generated.

 In addition, if any path in aSortPairArray is an empty path (that is, a
 zero-length String), the sort is performed upon the elements of the receiver
 itself, rather than upon the instance variables of those elements."

^ self sortWith: aSortPairArray persistentRoot: nil

]

{ #category : 'Sorting' }
Collection >> sortWith: aSortPairArray persistentRoot: persistentArrayOrNil [

"Returns an Array containing the elements of the receiver, sorted according to
 the contents of aSortPairArray.  The argument aSortPairArray is an Array of
 Strings that represent path/direction pairs, in the following form:

 aCollection sortWith: #('a.b' 'ASCENDING' 'a.c' 'DESCENDING' ...)

 That Array may contain up to 16 path/direction pairs.  The first path in the
 Array is the primary sort key, and the remaining paths are taken in order as
 subordinate keys.

 In aSortPairArray, each path String must follow the rules for equality
 indexes.  Each direction String must be either 'ASCENDING' or
 'DESCENDING' (case-insensitive); otherwise, an error is generated.

 In addition, if any path in aSortPairArray is an empty path (that is, a
 zero-length String), the sort is performed upon the elements of the receiver
 itself, rather than upon the instance variables of those elements.

 If persistentArrayOrNil is notNil, then it is expected to be an empty persistent array and
 the array will be used to persist large temporary data structures created during
 the sorting operation. IndexManager>>autoCommit must be true in order for periodic
 commits to  be made during the sorting operation. When the sort operation is complete
 the persistent array will be emptied and a final commit performed. The persistentArrayOrNil
 and  IndexManager>>autoCommit should be used when a collection is so large that it
 isn't practical to allocate enough temporary memory."

| paths booleans thePath directionStr directionBool result spaSize |

paths := { } .
booleans := { } .

((spaSize := aSortPairArray size) ~~ 0 and:[ spaSize even])
    ifFalse: [ ^ aSortPairArray _error: #assocErrSortOddLengthArray ].

1 to: spaSize by: 2 do: [ :i |
    thePath := aSortPairArray at: i.
    thePath _isOneByteString ifFalse:[
      thePath _validateClasses: { String }.
      ].
    directionStr := (aSortPairArray at: i + 1) .
    directionBool := true .
    directionStr == #ASCENDING
      ifFalse:[
        (directionStr == #DESCENDING )
          ifTrue:[ directionBool := false ]
          ifFalse:[
            (Symbol _existingWithAll: directionStr asUppercase) ifNotNil:[ :directionSym|
               directionSym == #ASCENDING
               ifFalse:[
                 directionSym == #DESCENDING
                   ifTrue: [ directionBool := false ]
                   ifFalse: [ directionStr _error: #assocErrBadDirection ].
	       ].
	    ] ifNil:[  directionStr _error: #assocErrBadDirection ].
	  ].
       ].
    paths add: thePath.
    booleans add: directionBool .
].
result := self _sortPaths: paths directions: booleans persistentRoot: persistentArrayOrNil.
^ result

]
