Extension { #name : 'Boolean' }

{ #category : 'Instance Creation' }
Boolean class >> fromStream: aStream [

"If the next characters in aStream are true or false (case insensitive, leading
 spaces permitted), this method returns the appropriate Boolean.  Otherwise,
 generates an error."

| char |

self _checkReadStream: aStream forClass: CharacterCollection.

char := aStream next.
[ char == $  ] whileTrue: [ char := aStream next ].
aStream skip: -1.

((char := aStream next) isEquivalent: $T)
ifTrue:
  [ ((aStream next isEquivalent: $R)
    and: [(aStream next isEquivalent: $U)
    and: [aStream next isEquivalent: $E ]])
    ifTrue:
      [ ^ true ]
  ]
ifFalse:
  [ (char isEquivalent: $F)
    ifTrue:
      [ ((aStream next isEquivalent: $A)
        and: [(aStream next isEquivalent: $L)
        and: [(aStream next isEquivalent: $S)
        and: [aStream next isEquivalent: $E ]]])
        ifTrue:
          [ ^ false ]
      ]
  ].

self _errIncorrectFormat: aStream

]

{ #category : 'Instance Creation' }
Boolean class >> fromString: aString [

"If aString contains true or false, returns the appropriate Boolean.  Leading
 and trailing spaces are permitted in the String.  If aString contains any
 characters other than true or false, this method generates an error."

| s result |

s := ReadStreamPortable on: aString.
result := self fromStream: s.
[ s atEnd ]
whileFalse:
  [ (s next == $  )
    ifFalse:
      [ self _errIncorrectFormat: aString ]
  ].
^ result

]

{ #category : 'Storing and Loading' }
Boolean class >> loadFrom: passiveObj [

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form and returns an equivalent instance of Boolean."

"This method supports filein of Booleans in the format required by GeODE
 versions before 2.0.1."

|inst|
inst := passiveObj next == $t.
passiveObj hasRead: inst.
^inst

]

{ #category : 'Instance Creation' }
Boolean class >> new [

"Disallowed.  You cannot create new instances of Boolean."

self shouldNotImplement: #new

]

{ #category : 'Flow of Control' }
Boolean >> _and: aBlock [

"(Reserved selector.)  Nonevaluating conjunction.  Returns the value of the
 zero-argument block aBlock if the receiver is true.  Otherwise, returns false
 without evaluating the argument.

 The following is an optimized control structure, not a recursive send."

^ self and: [ 
    aBlock _isExecBlock ifFalse:[ ArgumentError signal:'expected a block'].
    aBlock value
  ]

]

{ #category : 'Decompiling without Sources' }
Boolean >> _asSource [

^ self asString

]

{ #category : 'New Indexing Comparison' }
Boolean >> _classSortOrdinal [

^ 20

]

{ #category : 'New Indexing Comparison - for Compare' }
Boolean >> _idxForCompareBooleanGreaterThanOrEqualToSelf: aBoolean [

"second half of a double dispatch call from Boolean>>_idxForCompareGreaterThanOrEqualTo:. Note that aBoolean should be the receiver in any >= comparison"

aBoolean = self
  ifTrue: [ ^true].
^aBoolean == true

]

{ #category : 'New Indexing Comparison - for Compare' }
Boolean >> _idxForCompareBooleanGreaterThanSelf: aBoolean [

"second half of a double dispatch call from Boolean>>_idxForCompareGreaterThan:. Note that aBoolean should be the receiver in any > comparison"

self = aBoolean
    ifTrue: [ ^ false ].
^aBoolean == true

]

{ #category : 'New Indexing Comparison - for Compare' }
Boolean >> _idxForCompareBooleanLessThanOrEqualToSelf: aBoolean [

"second half of a double dispatch call from Boolean>>_idxForCompareLessThanOrEqualTo:. Note that aBoolean should be the receiver in any <= comparison"

aBoolean = self
  ifTrue: [ ^true].
^aBoolean == false

]

{ #category : 'New Indexing Comparison - for Compare' }
Boolean >> _idxForCompareBooleanLessThanSelf: aBoolean [

"second half of a double dispatch call from Boolean>>_idxForCompareLessThan:. Note that aBoolean should be the receiver in any < comparison"

self = aBoolean
    ifTrue: [ ^ false ].
^aBoolean == false

]

{ #category : 'New Indexing Comparison' }
Boolean >> _idxForCompareGreaterThan: arg [

""

^arg _idxForCompareBooleanGreaterThanSelf: self

]

{ #category : 'New Indexing Comparison' }
Boolean >> _idxForCompareGreaterThanOrEqualTo: arg [

""

^arg _idxForCompareBooleanGreaterThanOrEqualToSelf: self

]

{ #category : 'New Indexing Comparison' }
Boolean >> _idxForCompareLessThan: arg [

""

^arg _idxForCompareBooleanLessThanSelf: self

]

{ #category : 'New Indexing Comparison' }
Boolean >> _idxForCompareLessThanOrEqualTo: arg [

""

^arg _idxForCompareBooleanLessThanOrEqualToSelf: self

]

{ #category : 'New Indexing Comparison - for Sort' }
Boolean >> _idxForSortBooleanGreaterThanOrEqualToSelf: anObject [

"second half of a double dispatch call from Boolean>>_idxForSortGreaterThanOrEqualTo:. Note that aBoolean should be the receiver in any >= comparison"

anObject = self
  ifTrue: [ ^ true ].
^anObject

]

{ #category : 'New Indexing Comparison - for Sort' }
Boolean >> _idxForSortBooleanGreaterThanSelf: anObject [

"second half of a double dispatch call from Boolean>>_idxForSortGreaterThan:. Note that aBoolean should be the receiver in any > comparison"

anObject = self
  ifTrue: [ ^ false ].
^anObject

]

{ #category : 'New Indexing Comparison - for Sort' }
Boolean >> _idxForSortBooleanLessThanOrEqualToSelf: anObject [

"second half of a double dispatch call from Boolean>>_idxForSortLessThanOrEqualTo:. Note that aBoolean should be the receiver in any <= comparison"

anObject = self
  ifTrue: [ ^ true ].
^anObject not

]

{ #category : 'New Indexing Comparison - for Sort' }
Boolean >> _idxForSortBooleanLessThanSelf: anObject [

"second half of a double dispatch call from Boolean>>_idxForSortLessThan:. Note that aBoolean
should be the receiver in any < comparison"

anObject = self
  ifTrue: [ ^ false ].
^anObject not

]

{ #category : 'New Indexing Comparison' }
Boolean >> _idxForSortGreaterThan: arg [

""

^arg _idxForSortBooleanGreaterThanSelf: self

]

{ #category : 'New Indexing Comparison' }
Boolean >> _idxForSortGreaterThanOrEqualTo: arg [

""

^arg _idxForSortBooleanGreaterThanOrEqualToSelf: self

]

{ #category : 'New Indexing Comparison' }
Boolean >> _idxForSortLessThan: arg [

""

^arg _idxForSortBooleanLessThanSelf: self

]

{ #category : 'New Indexing Comparison' }
Boolean >> _idxForSortLessThanOrEqualTo: arg [

""

^arg _idxForSortBooleanLessThanOrEqualToSelf: self

]

{ #category : 'Flow of Control' }
Boolean >> _or: aBlock [

"(Reserved selector.)  Nonevaluating disjunction.  Returns the value of the
 zero-argument block aBlock if the receiver is false.  Otherwise, returns true
 without evaluating the argument.

 The following is an optimized control structure, not a recursive send."

^ self or:[ 
    aBlock _isExecBlock ifFalse:[ ArgumentError signal:'expected a block'].
    aBlock value 
  ]

]

{ #category : 'Logical Operations' }
Boolean >> & aBoolean [

"Evaluating conjunction (AND).  Returns true if both the receiver and the
 argument aBoolean are true."

<primitive: 42>
aBoolean _validateClass: Boolean.
self _primitiveFailed: #& args: { aBoolean }.
self _uncontinuableError

]

{ #category : 'Logical Operations' }
Boolean >> | aBoolean [

"Evaluating disjunction (OR).  Returns true if either the receiver or the
 argument aBoolean is true."

<primitive: 43>
aBoolean _validateClass: Boolean.
self _primitiveFailed: #| args: { aBoolean }.
self _uncontinuableError

]

{ #category : 'Flow of Control' }
Boolean >> and: aBlock [

"(Reserved selector.) Nonevaluating conjunction.  Returns the value of the
 zero-argument block aBlock if the receiver is true.  Otherwise, returns false
 without evaluating the argument.

 The following is an optimized control structure, not a recursive send."

^ self and: [ 
   aBlock _isExecBlock ifFalse:[ ArgumentError signal:'expected a block'].
   aBlock value
 ]

]

{ #category : 'Converting' }
Boolean >> asBit [
  "Returns 1 if self==true, 0 otherwise"

  ^ self ifTrue:[ 1 ] ifFalse:[ 0 ]
]

{ #category : 'Formatting' }
Boolean >> asString [

"Returns a String containing true or false, depending on the receiver."

self
  ifTrue:
    [ ^ String withAll: 'true' ]
  ifFalse:
    [ ^ String withAll: 'false' ]

]

{ #category : 'Converting' }
Boolean >> asSymbol [
 "Returns #true if self==true, #false otherwise"

  ^ self ifTrue:[ #true ] ifFalse:[ #false ]
]

{ #category : 'Clustering' }
Boolean >> clusterDepthFirst [

"Returns true.  (Because Booleans are self-defining objects, this method has no
 effect.)"

^ true

]

{ #category : 'Storing and Loading' }
Boolean >> containsIdentity [

"Private."

^true

]

{ #category : 'Copying' }
Boolean >> copy [

"Overrides the inherited method to return the receiver.  The pseudo-variables
 true and false are the only instances of Boolean, and must preserve identity."

^ self

]

{ #category : 'Logical Operations' }
Boolean >> eqv: aBoolean [

"Returns true if the receiver is identical to aBoolean."

^ self == aBoolean

]

{ #category : 'Flow of Control' }
Boolean >> ifFalse: aBlock [

"(Reserved selector.)  Returns the value of the zero-argument block aBlock if
 the receiver is false.  Otherwise, returns nil without evaluating the
 argument."

"The following is an optimized control structure, not a recursive send."

^ self ifFalse: [
    aBlock _isExecBlock ifFalse:[ ArgumentError signal:'expected a block'].
   ^ aBlock value 
 ]

]

{ #category : 'Flow of Control' }
Boolean >> ifFalse: falseBlock ifTrue: trueBlock [

"(Reserved selector.)  Returns the value of the zero-argument block falseBlock
 if the receiver is false.  Otherwise, returns the value of the zero-argument
 block trueBlock without evaluating falseBlock."

"The following is an optimized control structure, not a recursive send."

self ifFalse:[
  falseBlock _isExecBlock ifFalse:[ ArgumentError signal:'expected a block'].
  ^ falseBlock value
] ifTrue: [
  trueBlock _isExecBlock ifFalse:[ ArgumentError signal:'expected a block'].
  ^ trueBlock value
]

]

{ #category : 'Flow of Control' }
Boolean >> ifTrue: aBlock [

"(Reserved selector.)  Returns the value of the zero-argument block aBlock if
 the receiver is true.  Otherwise, returns nil without evaluating the argument."

"The following is an optimized control structure, not a recursive send."

^ self ifTrue: [
   aBlock _isExecBlock ifFalse:[ ArgumentError signal:'expected a block'].
   ^ aBlock value
]

]

{ #category : 'Flow of Control' }
Boolean >> ifTrue: trueBlock ifFalse: falseBlock [

"(Reserved selector.)  Returns the value of the zero-argument block falseBlock
 if the receiver is false.  Otherwise, returns the value of the zero-argument
 block trueBlock without evaluating falseBlock."

"The following is an optimized control structure, not a recursive send."

^ self ifTrue: [
   trueBlock _isExecBlock ifFalse:[ ArgumentError signal:'expected a block'].
   ^ trueBlock value
  ] ifFalse: [ 
   falseBlock _isExecBlock ifFalse:[ ArgumentError signal:'expected a block'].
   ^ falseBlock value
  ]

]

{ #category : 'Testing' }
Boolean >> isSpecial [

"Returns true if the receiver is a special object."

^ true

]

{ #category : 'Logical Operations' }
Boolean >> not [

"This implementation is used only by #perform .  
 If the receiver is a Boolean, #not is a special selector that does not do 
 a message send.

 Negation.  Returns true if the receiver is false.  Returns false if the
 receiver is true."

^ self == false
]

{ #category : 'Flow of Control' }
Boolean >> or: aBlock [

"(Reserved selector.) Nonevaluating disjunction.  Returns the value of the
 zero-argument block aBlock if the receiver is false.  Otherwise, returns true
 without evaluating the argument.

 The following is an optimized control structure, not a recursive send."

^ self or:[ 
    aBlock _isExecBlock ifFalse:[ ArgumentError signal:'expected a block'].
    aBlock value 
 ]

]

{ #category : 'Json' }
Boolean >> printJsonOn: aStream [

	aStream nextPutAll: (self ifTrue: ['true'] ifFalse: ['false']).

]

{ #category : 'Formatting' }
Boolean >> printString [

"Returns a String whose contents are a displayable representation of the
 receiver."

"GemStone does not allow the creation of new kinds of Boolean, so there is no
 point in creating a stream and sending printOn:."

^self asString

]

{ #category : 'Storing and Loading' }
Boolean >> writeTo: passiveObj [

"Converts the receiver to its passive form and writes that information on
 passiveObj."

passiveObj nextPut: (self ifTrue: [ $* ] ifFalse: [ $~ ])


]

{ #category : 'Logical Operations' }
Boolean >> xor: aBoolean [

"Exclusive OR.  Returns true if the receiver is not identical to aBoolean."

<primitive: 45>
aBoolean _validateClass: Boolean.
self _primitiveFailed: #xor: args: { aBoolean }.
self _uncontinuableError

]
