Extension { #name : 'Association' }

{ #category : 'Instance Creation' }
Association class >> newWithKey: aKey value: aValue [

"Returns a new Association with the argument aKey as its key and with aValue as
 its value."

^ super new key: aKey value: aValue

]

{ #category : 'Indexing Support' }
Association >> _idxValue [

	^self value

]

{ #category : 'Accessing' }
Association >> _value [

" Returns the value portion of the receiver.
  Faster than #value  in Gs64 v3.0 since send of #value
  assumes ExecBlock is most likely receiver. "

^value

]

{ #category : 'Updating' }
Association >> _value: aValue [

" Updates the value portion of the receiver.
  Faster than #value:  in Gs64 v3.0 since send of #value:
  assumes ExecBlock is most likely receiver. "

value := aValue

]

{ #category : 'Comparing' }
Association >> < anObject [

"Returns true if the key of the receiver collates before anObject.
 Returns false otherwise."

^ key == self
  ifTrue: [ false ]
  ifFalse: [ anObject > self key ]

]

{ #category : 'Comparing' }
Association >> <= anObject [

"Returns true if the key of the receiver collates before anObject, or if the
 key of the receiver is equivalent to anObject.  Returns false otherwise."

^ key == self
  ifTrue: [ true ]
  ifFalse: [ anObject >= self key ]

]

{ #category : 'Comparing' }
Association >> = anObject [

"Returns true if (a) the receiver and anObject are both kinds of Association,
 (b) the receiver and anObject have equal keys and (c) the receiver and
 anObject have equal values.  Returns false otherwise."

(self == anObject) ifTrue: [^ true].
(anObject isKindOf: Association) ifFalse:[^ false].
^ (key = anObject key) and: [value = anObject value].

]

{ #category : 'Comparing' }
Association >> > anObject [

"Returns true if the key of the receiver collates after anObject.
 Returns false otherwise."

^ anObject < self key

]

{ #category : 'Comparing' }
Association >> >= anObject [

"Returns true if the key of the receiver collates after anObject, or if the
 key of the receiver is equivalent to anObject.  Returns false otherwise."

^ anObject <= self key

]

{ #category : 'Comparing' }
Association >> ~= anObject [

"Returns true if one or more of the conditions specified in #= method are
 not satisfied.  Returns false otherwise."

^ (self = anObject) == false.

]

{ #category : 'Clustering' }
Association >> clusterDepthFirst [

"This method clusters the receiver, its key, and its value in
 depth-first order.  It returns true if the receiver has already
 been clustered during the current transaction, false otherwise."

  self cluster
  ifTrue:
    [ ^ true ]
  ifFalse:
    [ key clusterDepthFirst.
      value clusterDepthFirst.
      ^ false
    ].

]

{ #category : 'Comparing' }
Association >> hash [

"Returns an Integer hash code for the receiver."

^ key hash

]

{ #category : 'Testing' }
Association >> isAssociation [
  ^ true

]

{ #category : 'Formatting' }
Association >> isInPrintingRecursionSet: anIdentitySet [
	"Answer whether we are recursing through the receiver.

	 For associations, there are pathological cases one can create,
	 that we have chosen to portray in their logical representation.
	 Since some dictionaries use associations and others do not,
	 portray the results as if they all do.

	 In addition to the fast identity check, include a slower equality check."

	^(super isInPrintingRecursionSet: anIdentitySet)
		or: [anIdentitySet includesValue: self]

]

{ #category : 'Accessing' }
Association >> key [

"Returns the value of the receiver's key."

^key

]

{ #category : 'Updating' }
Association >> key: aKey [

"Sets the object aKey as the key of the receiver."

key := aKey

]

{ #category : 'Updating' }
Association >> key: aKey value: aValue [

"Sets the object aKey as the key of the receiver, and the object aValue as the
 value of the receiver."

key := aKey.
value := aValue

]

{ #category : 'Formatting' }
Association >> printNonRecursiveRepresentationOn: aStream recursionSet: anIdentitySet [
	"Put a displayable representation of the receiver on the given stream
	 while avoiding recursion from object reference loops."

	key printOn: aStream recursionSet: anIdentitySet.
	aStream nextPutAll: '->'.
	value printOn: aStream recursionSet: anIdentitySet

]

{ #category : 'Formatting' }
Association >> printOn: aStream [
	"Put a displayable representation of the receiver on the given
	 stream. If the key or the value is identical to the receiver then
	 avoid an infinite recursion. (There is currently no general way to
	 limit these kinds of recursions. See Bug #16259)"

	self printNonRecursiveOn: aStream

]

{ #category : 'Accessing' }
Association >> value [

"Returns the value portion of the receiver."

^value

]

{ #category : 'Updating' }
Association >> value: aValue [

"Sets the object aValue as the value of the receiver."

value := aValue

]
