Extension { #name : 'AbstractDictionary' }

{ #category : 'Private' }
AbstractDictionary class >> _tableSizeFor: targetSize [
  | limit |
  limit := 2034 - self instSize - 1 .
  (targetSize >= (limit - 100) and:[ targetSize <= limit ]) ifTrue:[
    "create the object with basicSize close to max small object size."
    ^ Integer _selectedPrimeGreaterThan: (limit min: 2028 )
  ].
  ^ Integer _selectedPrimeGreaterThan: targetSize .
]

{ #category : 'Storing and Loading' }
AbstractDictionary class >> loadFrom: passiveObj [

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into a new instance of the receiver.
 Returns the new instance."

| size inst |
size := passiveObj readSize.
inst := self new.
inst loadFrom: passiveObj size: size.
^inst

]

{ #category : 'Instance Creation' }
AbstractDictionary class >> new [

"(Subclass responsibility.)  Returns a new instance of AbstractDictionary."

^ self subclassResponsibility: #new


]

{ #category : 'Instance Creation' }
AbstractDictionary class >> new: count [

"(Subclass responsibility.) Returns a new instance of AbstractDictionary. The
 argument count provides a hint of the number of elements the instance should
 be designed to hold."

^ self subclassResponsibility: #new:

]

{ #category : 'Private' }
AbstractDictionary >> _asCollectionForSorting [

"Creates a new collection that can be easily indexed for sorting."

| result |

result := { } .
self do: [ :aValue | result add: aValue].
^ result.

]

{ #category : 'Accessing' }
AbstractDictionary >> _at: anIndex [

"Signals an error if the receiver is not indexable,
 or if anIndex is not a SmallInteger or is out of bounds.

 This method is for internal use in implementation of the kernel classes.

 The methods _at:, _basicAt:, and _basicSize operate on the logically
 indexable contents of the receiver.

 This method must not be overridden."

<primitive: 699>
(anIndex _isInteger)
  ifTrue: [^ self _errorIndexOutOfRange: anIndex]
  ifFalse: [^ self _errorNonIntegerIndex: anIndex].
self _primitiveFailed: #at: args: { anIndex }.
self _uncontinuableError

]

{ #category : 'Updating' }
AbstractDictionary >> _at: anIndex put: aValue [

"Stores the argument aValue in the indexed variable of the
 receiver indicated by anIndex.  The argument anIndex must not be
 larger than 1 + the size of the receiver, and must not be less than 1.

 Returns aValue.

 Generates an error if anIndex is not a SmallInteger or is out of
 bounds."

<primitive: 745>

(anIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: anIndex].
((anIndex < 1) | (anIndex > (self size + 1))) "out of bounds"
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].

self _primitiveFailed: #at:put: args: { anIndex . aValue }.
self _uncontinuableError

]

{ #category : 'Private' }
AbstractDictionary >> _atZ: zOffset [
  "return varying instVar specified by zero-based offset zOffset"
<primitive: 858>
| sz |
zOffset _validateClass: SmallInteger .
sz := self _basicSize .
(zOffset < 0 or:[ zOffset >= sz ]) ifTrue:[
  OffsetError new offset: zOffset maximum: sz - 1 ; signal
].
self _primitiveFailed: #_atZ: args: { zOffset }

]

{ #category : 'Private' }
AbstractDictionary >> _atZ: zOffset put: aValue [
  "store aValue into varying instVar specified by zero-based offset zOffset.
   will auto-grow up to 6 instVars past current end"
<primitive: 860>
| max |
zOffset _validateClass: SmallInteger .
max := self _basicSize + 6 .
(zOffset < 0 or:[ zOffset >= max ]) ifTrue:[
  OffsetError new offset: zOffset maximum: max ; signal
].
self _primitiveFailed: #_atZ:put: args: { zOffset . aValue }

]

{ #category : 'Private' }
AbstractDictionary >> _atZ: zOffset putKey: aKey value: aValue [
  "store aKey into varying instVar specified by zero-based offset zOffset.
   store aValue into varying instVar specified by zero-based offset zOffset + 1.
   will auto-grow up to 7 instVars past current end"
<primitive: 892>
| max |
zOffset _validateClass: SmallInteger .
max := self _basicSize + 6 .
(zOffset < 0 or:[ zOffset >= max ]) ifTrue:[
  OffsetError new offset: zOffset maximum: max ; signal
].
self _primitiveFailed: #_atZ:putKey:value:
     args: { zOffset . aKey . aValue }

]

{ #category : 'Accessing' }
AbstractDictionary >> _basicAt: anIndex [

"Signals an error if the receiver is not indexable,
 or if anIndex is not a SmallInteger or is out of bounds.

 This method is for internal use in implementation of the kernel classes.

 The methods _at:, _basicAt:, and _basicSize operate on the logically
 indexable contents of the receiver.

 This method must not be overridden."

<primitive: 699>
(anIndex _isInteger)
  ifTrue: [^ self _errorIndexOutOfRange: anIndex]
  ifFalse: [^ self _errorNonIntegerIndex: anIndex].
self _primitiveFailed: #at: args: { anIndex }.
self _uncontinuableError

]

{ #category : 'Updating' }
AbstractDictionary >> _basicAt: anIndex put: aValue [

"Stores the argument aValue in the indexed variable of the
 receiver indicated by anIndex.  The argument anIndex must not be
 larger than 1 + the size of the receiver, and must not be less than 1.

 Returns aValue.

 Generates an error if anIndex is not a SmallInteger or is out of
 bounds."

<primitive: 745>

(anIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: anIndex].
((anIndex < 1) | (anIndex > (self size + 1))) "out of bounds"
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].

self _primitiveFailed: #at:put: args: { anIndex . aValue }

]

{ #category : 'Private' }
AbstractDictionary >> _deepCopyWith: copiedObjDict [

| copy myClass |

copy := copiedObjDict at: self otherwise: nil.
copy ~~ nil ifTrue: [ ^ copy ].

myClass := self class .
copy := myClass new: (self size).
copiedObjDict at: self put: copy.

self _deepCopyNamedIvsWith: copiedObjDict to: copy .

self keysAndValuesDo: [ :aKey :aValue |
  copy at: (aKey _deepCopyWith: copiedObjDict)
       put: (aValue _deepCopyWith: copiedObjDict).
  ].

^ copy.

]

{ #category : 'Private' }
AbstractDictionary >> _deferredGciUpdateWith: valueArray [

^ self subclassResponsibility: #_deferredGciUpdateWith:

]

{ #category : 'Private' }
AbstractDictionary >> _errorKeyNotFound: aKey [

"No Association or key-value pair with given key, aKey, was found."

^ self _error: #rtErrKeyNotFound args: { aKey }

]

{ #category : 'Private' }
AbstractDictionary >> _nodesObjectSecurityPolicy: anObjectSecurityPolicy [
  "Assigns receiver's components to the given security policy. "

  self subclassResponsibility: #_nodesObjectSecurityPolicy:

]

{ #category : 'Private' }
AbstractDictionary >> _reportKeyNotFound: aKey with: aBlock [

"Returns the value of aBlock if aBlock is not nil.  Otherwise, raises an error."

aBlock == nil ifTrue:[^ self _errorKeyNotFound: aKey ] .
^aBlock value

]

{ #category : 'Comparing' }
AbstractDictionary >> = anAbstractDictionary [
"Returns true if all of the following conditions are true:

 1.  The receiver and anAbstractDictionary are of the same class.
 2.  The two dictionaries are of the same size.
 3.  The corresponding keys and values of the receiver and anAbstractDictionary
     are equal."

| notFound |
(self == anAbstractDictionary) ifTrue:[ ^ true ].

(self class == anAbstractDictionary class) ifFalse:[ ^ false ].

(self size == anAbstractDictionary size) ifFalse:[ ^ false ].
notFound := Object new .
self keysAndValuesDo: [ :aKey :aValue | | otherVal|
  otherVal := anAbstractDictionary at: aKey otherwise: notFound .
  (aValue == self or:[ aValue == anAbstractDictionary]) ifTrue:[
    (otherVal == self or:[ otherVal == anAbstractDictionary]) ifFalse:[ ^ false].
  ] ifFalse:[
    aValue = otherVal ifFalse:[ ^ false ]
  ].
].
^ true.

]

{ #category : 'Enumerating' }
AbstractDictionary >> accompaniedBy: anObj keysAndValuesDo: aBlock [

"Iteratively evaluates the threee argument block, aBlock,
 using anObj, each key and each value
 of the receiver as the arguments to the block.  Returns the receiver."

^ self subclassResponsibility: #accompaniedBy:keysAndValuesDo: .

]

{ #category : 'Adding' }
AbstractDictionary >> add: anAssociation [

"Adds the Association or the key-value pair contained in anAssociation to the
 receiver.  If the receiver already includes an Association/key-value pair
 whose key is equal to that of anAssociation, then this method redefines the
 value portion of that Association/key-value pair. Returns anAssociation."

self at: anAssociation key put: anAssociation value.
^ anAssociation.

]

{ #category : 'Adding' }
AbstractDictionary >> addAll: aCollection [

"Adds to the receiver all the Associations or key-value pairs contained in
 aCollection.  aCollection must be a collection of Associations or a dictionary.
 Returns the argument, aCollection."

(aCollection isKindOf: AbstractDictionary) ifTrue: [
  aCollection accompaniedBy: self
    keysAndValuesDo:[ :rcvr :aKey :aVal| rcvr at: aKey put: aVal ].
] ifFalse: [
  aCollection accompaniedBy: self do: [ :me :anAssociation | me add: anAssociation ]
].
^ aCollection.

]

{ #category : 'Printing' }
AbstractDictionary >> asReportString [

"Returns a String that lists the key-value pairs, one on each line."

| result lf tab |
result := String new.
tab := Character tab .
lf := Character lf .
self keysAndValuesDo: [ :key :val |
  result add: key printString; add: tab ; add: val printString; add: lf
].
^ result

]

{ #category : 'Enumerating' }
AbstractDictionary >> associationsAsArray [

"Returns an Array containing all of the receiver's Associations"

^ self selectAssociationsAsArray: [ :assoc| true ]

]

{ #category : 'Enumerating' }
AbstractDictionary >> associationsDetect: aBlock [

"Evaluates aBlock repeatedly, with the Associations of the receiver as the
 argument.  Returns the first Association for which the block evaluates to true
 when the Association is used as the argument to the block.  If none of the
 receiver's Associations evaluates to true, generates an error.  The argument
 aBlock must be a one-argument block."

^ self associationsDetect: aBlock
       ifNone: [^ self _error: #assocErrNoElementsDetected args: { aBlock }]

]

{ #category : 'Enumerating' }
AbstractDictionary >> associationsDetect: aBlock ifNone: exceptionBlock [

"Evaluates aBlock repeatedly, with the Associations of the receiver as the
 argument.  Returns the first Association for which aBlock evaluates to true
 when the Association is used as the argument to the block.  If none of the
 receiver's Associations evaluates to true, this method evaluates the argument
 exceptionBlock and returns its value.  The argument aBlock must be a
 one-argument block, and exceptionBlock must be a zero-argument block."

self associationsDo: [ :anAssoc|
  (aBlock value: anAssoc) ifTrue: [ ^anAssoc ].
  ].

^ exceptionBlock value.

]

{ #category : 'Enumerating' }
AbstractDictionary >> associationsDo: aBlock [

"Iteratively evaluates the one argument block, aBlock, using each Association
 in the receiver as the argument to the block. If the receiver stores the
 key-value pairs directly, new Associations are created for the key-value pairs
 and used as the arguments to aBlock.  Returns the receiver."

^ self subclassResponsibility: #associationsDo:

]

{ #category : 'Accessing' }
AbstractDictionary >> at: aKey [

"Returns the value of the Association with key aKey.  Generates an error if no
 such key exists."

    | sentinel result |
    sentinel := AbsDictSentinel"created in bom.c"  .
    (result := self at: aKey otherwise: sentinel) == sentinel ifTrue:
        [^self _errorKeyNotFound: aKey].
    ^result

]

{ #category : 'Accessing' }
AbstractDictionary >> at: aKey ifAbsent: aBlock [

"Returns the value associated with key aKey.  If no such key/value pair or
 Association exists, returns the result of evaluating the zero-argument block
 aBlock."

^ self subclassResponsibility: #at:ifAbsent.

]

{ #category : 'Accessing' }
AbstractDictionary >> at: aKey ifAbsentPut: aBlock [

"Returns the value associated with key aKey. If no such key/value pair or
 Association exists, returns the result of evaluating the zero-argument block
 aBlock. In the latter case, the result of  evaluating the block aBlock is
 also stored in the receiver using the key aKey."

| known existing value |

known := 'a'.
existing := self at: aKey otherwise: known.
existing == known
  ifTrue: [
    value := aBlock value.
    self at: aKey put: value.
    ^value
    ].
^existing

]

{ #category : 'Accessing' }
AbstractDictionary >> at: key ifPresent: aBlock [
"Lookup the given key in the receiver. If it is present, answer the value of
 evaluating the given one argument block with the value associated with the key.
 Otherwise, answer nil."

| v |
v := self at: key ifAbsent:[ ^ nil].
^ aBlock value: v
]

{ #category : 'Accessing' }
AbstractDictionary >> at: aKey otherwise: value [

"Returns the value that corresponds to aKey.  If no such key/value pair or
 Association exists, returns the given alternate value."

    ^self at: aKey ifAbsent: [value]

]

{ #category : 'Updating' }
AbstractDictionary >> at: aKey put: aValue [

"Creates a new Association with the given key and value and adds it to the
 receiver or adds the key-value pair to the receiver depending on the class of
 the receiver.  If the receiver already contains an Association or key-value
 pair with the given key, this method makes aValue the value of that
 Association or key-value pair.  Returns aValue."

^ self subclassResponsibility: #at:put:

]

{ #category : 'Enumerating' }
AbstractDictionary >> collect: aBlock [

"Evaluates aBlock with each of the receiver's values as the argument and
 collects the resulting values into the appropriate Dictionary at the
 corresponding key values."

| result |

result := self speciesForCollect new: (self size).
self keysAndValuesDo: [ :aKey :aValue |
  result at: aKey put: (aBlock value: aValue).
  ].

^ result.

]

{ #category : 'Enumerating' }
AbstractDictionary >> collectAssociations: aBlock [

"Evaluates aBlock with each of the receiver's Associations (or Associations
 created using the key-value pairs) as the argument and collects the resulting
 values into an Array. Returns the newly created Array."

| anArray index |

anArray := Array new: (self size).
index := 0.
self associationsDo: [ :anAssoc |
  index := index + 1.
  anArray at: index put: (aBlock value: anAssoc).
  ].

^ anArray.

]

{ #category : 'Deprecated' }
AbstractDictionary >> collectValues: aBlock [

self deprecated: 'AbstractDictionary>>collectValues: deprecated long before v3.0. Use collect: instead.'.
^ self collect: aBlock.

]

{ #category : 'Enumerating' }
AbstractDictionary >> collectValuesAsArray: aBlock [

"Evaluates aBlock with each of the receiver's values as the argument and
 collects the resulting values into an Array. Returns the new Array."

| result |
result := { } .
self valuesDo: [ :aValue |
  result add: (aBlock value: aValue).
  ].
^ result.

]

{ #category : 'Deprecated' }
AbstractDictionary >> detectAssociations: aBlock [

self deprecated: 'AbstractDictionary>>detectAssociations: deprecated long before v3.0. Use associationsDetect: instead.'.
^ self associationsDetect: aBlock

]

{ #category : 'Deprecated' }
AbstractDictionary >> detectAssociations: aBlock ifNone: exceptionBlock [

self deprecated: 'AbstractDictionary>>detectAssociations:ifNone: deprecated long before v3.0. Use associationsDetect:ifNone: instead.'.
^ self associationsDetect: aBlock ifNone: exceptionBlock

]

{ #category : 'Deprecated' }
AbstractDictionary >> detectValues: aBlock [

self deprecated: 'AbstractDictionary>>detectValues: deprecated long before v3.0.  Use keysAndValuesDo: instead.'.
^ self detectValues: aBlock ifNone: [ ^ self errorNoElementDetected: aBlock ].

]

{ #category : 'Deprecated' }
AbstractDictionary >> detectValues: aBlock ifNone: exceptionBlock [

self deprecated: 'AbstractDictionary>>detectValues:ifNone: deprecated long before v3.0. Use keysAndValuesDo: instead.'.
self keysAndValuesDo: [ :aKey :aValue |
  (aBlock value: aValue) ifTrue: [ ^ aKey ].
  ].
^ exceptionBlock value.

]

{ #category : 'Enumerating' }
AbstractDictionary >> do: aBlock [

"Iteratively evaluates the one argument block, aBlock, using the value part of
 each Association or key-value pair as the argument of the block.  Returns the
 receiver."

^ self subclassResponsibility: #do.

]

{ #category : 'Deprecated' }
AbstractDictionary >> doAssociations: aBlock [

self deprecated: 'AbstractDictionary>>doAssociations: deprecated long before v3.0. Use associationsDo: instead.'.
^ self associationsDo: aBlock.

]

{ #category : 'Deprecated' }
AbstractDictionary >> doKeys: aBlock [

self deprecated: 'AbstractDictionary>>doKeys: deprecated long before v3.0. Use keysDo: instead.'.
^ self keysDo: aBlock.

]

{ #category : 'Deprecated' }
AbstractDictionary >> doKeysAndValues: aBlock [

self deprecated: 'AbstractDictionary>>doKeysAndValues: deprecated long before v3.0. Use keysAndValuesDo: instead.'.
^ self keysAndValuesDo: aBlock.

]

{ #category : 'Deprecated' }
AbstractDictionary >> doValues: aBlock [

self deprecated: 'AbstractDictionary>>doValues: deprecated long before v3.0. Use valuesDo: instead.'.
^ self valuesDo: aBlock

]

{ #category : 'Private' }
AbstractDictionary >> errorKeyNotFound: aKey [

"No Association or key/value pair with the given key, aKey was found."

^ self _error: #rtErrKeyNotFound args: { aKey }

]

{ #category : 'Private' }
AbstractDictionary >> errorNilKey [

"A nil key was provided as an argument."

^ self _error: #rtErrNilKey.

]

{ #category : 'Private' }
AbstractDictionary >> errorNoElementDetected: aBlock [

"Private."

^ self _error: #assocErrNoElementsDetected args: { aBlock }

]

{ #category : 'Comparing' }
AbstractDictionary >> hash [

"Returns a numeric hash key for the receiver."

| hashValue |

hashValue := 97633 bitXor: (self size).
"For large dictionaries, the hash value is just a function of its size"
(self size > 64) ifTrue: [ ^ hashValue abs ].
self keysDo: [ :aKey |
   "Skip if the key is a dictionary."
   (aKey isKindOf: AbstractDictionary)
     ifFalse: [
       hashValue := hashValue bitXor: aKey hash
       ]
     ].
^ hashValue abs

]

{ #category : 'Hashing' }
AbstractDictionary >> hashFunction: aKey [

"The hash function should perform some operation on the value of the
 key (aKey) which returns a value in the range 1..tableSize."

^(aKey hash \\  self tableSize) + 1

]

{ #category : 'Searching' }
AbstractDictionary >> includes: aValue [

"Returns true if the receiver contains a value that is equal to aValue.
 Returns false otherwise."

self valuesDo: [ :element | (aValue = element) ifTrue: [ ^true ]].
^ false.

]

{ #category : 'Searching' }
AbstractDictionary >> includesAssociation: anAssociation [

"Returns true if anAssociation is equal to one of the Associations of the
 receiver.  Returns false otherwise."

self associationsDo: [ :assoc | (anAssociation = assoc) ifTrue: [ ^true ]].
^false

]

{ #category : 'Searching' }
AbstractDictionary >> includesIdentical: aValue [

"Returns true if the receiver contains a value that is identical to aValue.
 Returns false otherwise."

self valuesDo: [ :element | (aValue == element) ifTrue: [ ^true ]].
^ false.

]

{ #category : 'Searching' }
AbstractDictionary >> includesIdenticalAssociation: anAssociation [

"Returns true if anAssociation is identical to one of the Associations of the
 receiver.  Returns false otherwise."

self associationsDo: [ :assoc | (anAssociation == assoc) ifTrue: [ ^true ]].
^false

]

{ #category : 'Searching' }
AbstractDictionary >> includesKey: aKey [

"Returns true if the receiver contains an Association or a key-value pair whose
 key is equal to aKey.  Returns false otherwise."

| res |
res := true .
self at: aKey ifAbsent:[ res := false ] .
^ res .

]

{ #category : 'Deprecated' }
AbstractDictionary >> includesValue: aValue [

self deprecated: 'AbstractDictionary>>includesValue: deprecated long before v3.0. Use includes: instead.'.
^ self includes: aValue.

]

{ #category : 'Accessing' }
AbstractDictionary >> keyAtValue: anObject [

"Returns the key of the first value equal to anObject. If no match is found,
 runtime error objErrNotInColl is signaled."

"Note: In some implementations of Smalltalk, nil is returned if a match is
 not found."

^self keyAtValue: anObject
      ifAbsent: [^ self _error: #objErrNotInColl args: { anObject }]

]

{ #category : 'Accessing' }
AbstractDictionary >> keyAtValue: anObject ifAbsent: aBlock [

"Returns the key of the first value equal to the given object, anObject.
 If no match is found, evaluates and returns the result of the block aBlock."

^ self subclassResponsibility: #keyAtValue:ifAbsent:

]

{ #category : 'Enumerating' }
AbstractDictionary >> keysAndValuesDo: aBlock [

"Iteratively evaluates the two argument block, aBlock, using each key and value
 of the receiver as the argument to the block.  Returns the receiver."

^ self subclassResponsibility: #keysAndValuesDo:.

]

{ #category : 'Enumerating' }
AbstractDictionary >> keysDo: aBlock [

"Iteratively evaluates the one argument block, aBlock, using each key of
 the receiver as the argument to the block. Returns the receiver."

^ self subclassResponsibility: #keysDo:.

]

{ #category : 'Storing and Loading' }
AbstractDictionary >> loadFrom: passiveObj size: varyingSize [

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into the receiver."

"This method is similar to loadFrom:, but is used for objects whose size
 is not known when they are first instantiated (such as an IdentitySet)."

^self basicLoadFrom: passiveObj size: varyingSize

]

{ #category : 'Instance Migration' }
AbstractDictionary >> migrateIndexable: anotherObject myClass: cls otherClass: othercls [

"For dictionaries, we need to reconstruct the indexable component rather
 than just copy it over (in case the new class uses a different structure)."

anotherObject keysAndValuesDo: [ :key :value |
    self at: key put: value ].


]

{ #category : 'Updating' }
AbstractDictionary >> objectSecurityPolicy: anObjectSecurityPolicy [

  "Assigns the receiver and all its components to the given security policy.
   Returns the receiver."

  super objectSecurityPolicy: anObjectSecurityPolicy.
  self _nodesObjectSecurityPolicy: anObjectSecurityPolicy.

]

{ #category : 'Searching' }
AbstractDictionary >> occurrencesOf: aValue [

"Returns the number of Associations or key-value pairs in the receiver with
 value equal to 'aValue'."

| numOccurrences |

numOccurrences := 0.
self valuesDo: [ :aVal |
  aValue = aVal ifTrue: [ numOccurrences := numOccurrences + 1 ]
  ].

^ numOccurrences

]

{ #category : 'Searching' }
AbstractDictionary >> occurrencesOfIdentical: aValue [

"Returns the number of Associations or key-value pairs in the receiver with
 a value that is identical to aValue."

| numOccurrences |

numOccurrences := 0.
self valuesDo: [ :aVal |
  aValue == aVal ifTrue: [ numOccurrences := numOccurrences + 1 ]
  ].

^ numOccurrences

]

{ #category : 'Deprecated' }
AbstractDictionary >> occurrencesOfValue: aValue [

self deprecated: 'AbstractDictionary>>occurrencesOfValue: deprecated long before v3.0. Use occurrencesOf: instead.'.
^ self occurrencesOf: aValue.

]

{ #category : 'Json' }
AbstractDictionary >> printJsonOn: aStream [

	| delimiter |
	delimiter := ''.
	aStream nextPut: ${.
	self keysAndValuesDo: [:key :value |
		aStream nextPutAll: delimiter.
		key asString printJsonOn: aStream.
		aStream nextPut: $:.
		value printJsonOn: aStream.
		delimiter := ','.
	].
	aStream nextPut: $}.

]

{ #category : 'Formatting' }
AbstractDictionary >> printNonRecursiveRepresentationOn: aStream recursionSet: anIdentitySet [
	"Put a displayable representation of the receiver on the given stream
	 while avoiding recursion from object reference loops."

	| count sz myCls |

	myCls := self class .
	aStream nextPutAll: myCls name describeClassName .
	(myCls whichClassIncludesSelector: #associationsDo:) == AbstractDictionary
		ifTrue:[ ^ self "can't safely execute associationsDo: " ].

	aStream nextPutAll: '( ' .
	count := 1 .
	sz := self size .
	self associationsDo: [:anAssoc |
		anAssoc printOn: aStream recursionSet: anIdentitySet.
		aStream isFull ifTrue:[
			"prevent infinite recursion when printing cyclic structures, and
			limit the size of result when printing large collections."
			aStream _nextPut:( (aStream _collection endsWith: '...') ifTrue:[ $) ] ifFalse:[ ' ...)' ]).
			^ self
			] .
		count < sz ifTrue:[ aStream nextPutAll: ', ' ].
		count := count + 1.
		].
	aStream nextPut: $).

]

{ #category : 'Formatting' }
AbstractDictionary >> printOn: aStream [
	"Put a displayable representation of the receiver on the given stream."

	self printNonRecursiveOn: aStream

]

{ #category : 'Enumerating' }
AbstractDictionary >> reject: aBlock [

"Evaluates aBlock with each of the receiver's values as the argument. Stores
 the key-value pairs for which aBlock is false into a dictionary of the same
 class as the receiver, and returns the new dictionary. The argument aBlock
 must be a one-argument block."

| result |

result := self species new: (self size).
self keysAndValuesDo: [ :aKey :aValue |
  (aBlock value: aValue) ifFalse: [result at: aKey put: aValue]
  ].

^ result.

]

{ #category : 'Enumerating' }
AbstractDictionary >> rejectAssociations: aBlock [

"Evaluates aBlock with each of the receiver's elements as the argument.  Stores
 the values for which aBlock is false into a collection of the same class as
 the receiver, and returns the new collection.  The argument aBlock must be a
 one-argument block.  Uses associative access when the argument is a
 SelectionBlock."

| newCollection |

newCollection := self species new.
self associationsDo: [ :element |
  (aBlock value: element) ifFalse: [ newCollection add: element ].
  ].

^ newCollection.

]

{ #category : 'Deprecated' }
AbstractDictionary >> rejectValues: aBlock [

self deprecated: 'AbstractDictionary>>rejectValues: deprecated long before v3.0. Use reject: instead.'.
^ self reject: aBlock.

]

{ #category : 'Enumerating' }
AbstractDictionary >> rejectValuesAsArray: aBlock [

"Evaluates aBlock with each of the receiver's values as the argument and
 returns an Array containing the values for which aBlock evaluates false."

| result |
result := { } .
self valuesDo: [ :aValue |
  (aBlock value: aValue) ifFalse: [result add: aValue].
  ].

^ result.

]

{ #category : 'Removing' }
AbstractDictionary >> remove: anObject [

"Disallowed.  Use #removeKey: instead."

self shouldNotImplement: #remove:

]

{ #category : 'Removing' }
AbstractDictionary >> remove: anObject ifAbsent: anExceptionBlock [

"Disallowed.  Use #removeKey:ifAbsent: instead."

self shouldNotImplement: #remove:ifAbsent.

]

{ #category : 'Removing' }
AbstractDictionary >> removeAll: aCollection [

"Disallowed.  Use #removeAllKeys: instead."

self shouldNotImplement: #removeAll:

]

{ #category : 'Removing' }
AbstractDictionary >> removeAllIdentical: aCollection [

"Disallowed."

self shouldNotImplement: #removeAllIdentical:

]

{ #category : 'Removing' }
AbstractDictionary >> removeAllKeys: keys [

"Removes all the keys equal to the given keys from the receiver. An error is not
 generated if keys equal to any of the specified keys are not present. Returns
 the collection keys."

^ keys accompaniedBy: self do: [ :me :aKey | me removeKey: aKey otherwise: nil ].

]

{ #category : 'Removing' }
AbstractDictionary >> removeAllKeys: keys ifAbsent: aBlock [

"Removes all the keys equal to the given keys from the receiver and returns the
 collection keys. For any key which is not a valid key of the receiver, aBlock
 is evaluated with the key as the argument."

^ keys accompaniedBy: self do: [:me :aKey | me removeKey: aKey ifAbsent: [aBlock value: aKey] ].

]

{ #category : 'Removing' }
AbstractDictionary >> removeIdentical: anObject [

"Disallowed."

self shouldNotImplement: #removeIdentical:

]

{ #category : 'Removing' }
AbstractDictionary >> removeIdentical: anObject ifAbsent: anExceptionBlock [

"Disallowed."

self shouldNotImplement: #removeIdentical:ifAbsent.

]

{ #category : 'Removing' }
AbstractDictionary >> removeKey: aKey [

"Removes the Association or key-value pair with key equal to aKey from the
 receiver and returns the value portion of that Association or key-value pair
 respectively. If no Association is present with key equal to aKey, reports an
 error."

^ self removeKey: aKey ifAbsent: [self _errorKeyNotFound: aKey ].

]

{ #category : 'Removing' }
AbstractDictionary >> removeKey: aKey ifAbsent: aBlock [

"Removes the Association or key-value pair with key equal to aKey from the
 receiver and returns the value associated with the Association or the key-value
 pair.  If no Association or key-value pair is present with key equal to
 aKey, evaluates the zero-argument block aBlock and returns the result of that
 evaluation."

^ self subclassResponsibility: #removeKey:ifAbsent.

]

{ #category : 'Removing' }
AbstractDictionary >> removeKey: aKey otherwise: notFoundValue [

"Removes the Association or key-value pair with key equal to aKey from the
 receiver and returns the value associated with the Association or the key-value
 pair.  If no Association or key-value pair is present with key equal to
 aKey, returns notFoundValue .

 KeyValueDictionary has an optimized implementation.
"

^ self removeKey: aKey ifAbsent:[ notFoundValue ]

]

{ #category : 'Deprecated' }
AbstractDictionary >> removeKeys: keys [

self deprecated: 'AbstractDictionary>>removeKeys: deprecated long before v3.0. Use removeAllKeys: instead.'.
^ self removeAllKeys: keys

]

{ #category : 'Enumerating' }
AbstractDictionary >> select: aBlock [

"Evaluates aBlock with each of the receiver's values as the argument.  Stores
 the values for which aBlock is true into the dictionary of the same class as
 the receiver, and returns the new dictionary.  The argument aBlock must be a
 one-argument block."

|result|

result := self species new.
self keysAndValuesDo: [ :aKey :aValue |
  (aBlock value: aValue) ifTrue: [result at: aKey put: aValue]
  ].

^ result.

]

{ #category : 'Enumerating' }
AbstractDictionary >> selectAssociations: aBlock [

"Evaluates aBlock with each of the receiver's elements as the argument.  Stores
 the values for which aBlock is true into a collection of the same class as the
 receiver, and returns the new collection.  The argument aBlock must be a
 one-argument block.  Uses associative access when the argument is a
 SelectionBlock."

| newCollection |

newCollection := self species new.
self associationsDo: [ :element |
  (aBlock value: element) ifTrue: [ newCollection add: element ].
  ].

^ newCollection.

]

{ #category : 'Enumerating' }
AbstractDictionary >> selectAssociationsAsArray: aBlock [

"Evaluates aBlock with each of the receiver's associations as the argument.  Stores
 the values for which aBlock is true into a new Array and returns the Array.
 The argument aBlock must be a one-argument block.  "

| newCollection |

newCollection := { }  .
self associationsDo: [ :element |
  (aBlock value: element) ifTrue: [ newCollection add: element ].
  ].

^ newCollection.

]

{ #category : 'Deprecated' }
AbstractDictionary >> selectValues: aBlock [

self deprecated: 'AbstractDictionary>>selectValues: deprecated long before v3.0. Use select: instead.'.
^ self select: aBlock.

]

{ #category : 'Deprecated' }
AbstractDictionary >> selectValuesAsArray: aBlock [


"Evaluates aBlock with each of the receiver's values as the argument and
 returns an Array containing the values for which aBlock evaluates true."

| result index |
self deprecated: 'AbstractDictionary>>selectValuesAsArray: deprecated long before v3.0.'.
result := Array new: (self size).
index := 1.
self valuesDo: [ :aValue |
  (aBlock value: aValue)
    ifTrue: [result at: index put: aValue].
  index := index + 1.
  ].

^ result.

]

{ #category : 'Accessing' }
AbstractDictionary >> size [

"Returns the number of elements (Associations/key-value pairs) contained in the
 receiver."

^ self subclassResponsibility: #size.

]

{ #category : 'Updating' }
AbstractDictionary >> size: newSize [

"Disallowed.  You should not change the size of a dictionary explicitly."

self shouldNotImplement: #size: .

]

{ #category : 'Accessing' }
AbstractDictionary >> squeakBasicAt: anIndex [
  ^ self _basicAt: anIndex

]

{ #category : 'Updating' }
AbstractDictionary >> squeakBasicAt: anIndex put: aValue [

  ^ self _basicAt: anIndex put: aValue

]

{ #category : 'Private' }
AbstractDictionary >> tableSize [

"Returns the size of hash table used for storing the entries."

^ self subclassResponsibility: #tableSize.

]

{ #category : 'Accessing' }
AbstractDictionary >> values [

"Returns an OrderedCollection containing the receiver's values."

|result|
result:= OrderedCollection new.
self valuesDo: [ :value | result add: value ].
^ result.

]

{ #category : 'Enumerating' }
AbstractDictionary >> valuesDo: aBlock [

"Iteratively evaluates the one argument block, aBlock, using each value of
 the receiver as the argument to the block. Returns the receiver."

"Note: This method has the same behavior as #do:."

^ self subclassResponsibility: #valuesDo:.

]
