"
I'm a parser for tonel files. 
I parse a class with the following format: 

Tonel spec
====

    [comment]
    type { typeDefinition }
    (
        [{ methodMetadata }]
        method [
            methodBody ] 
    )*


comment
---
""
comment string
""
is optional (but it should be there, in good design ;)

type
---
Class|Trait|Extension

typeDefinition
---
a STON file with class/trait/extension metadata

methodMetadata
---
a STON file with method metadata
is optional (but also, recommended)

method
---
method declaration as this: 

Class[ class] >> selector

methodBody 
---
the method body (we do not parse contents, that's class builder task)
"
Class {
	#name : 'RwTonelParser',
	#superclass : 'Object',
	#instVars : [
		'packageReader',
		'stream',
		'filePath',
		'lastSelectorParsed',
		'classSelectors',
		'selectors'
	],
	#classVars : [
		'Character_lf'
	],
	#category : 'Rowan-Tonel-Core'
}

{ #category : 'initialization' }
RwTonelParser class >> initialize [
	self _addInvariantClassVar: #Character_lf value: Character lf
]

{ #category : 'accessing' }
RwTonelParser class >> lineEnding [
  "Answer the os-specific line endings.  See also #lineEndingSize if changing "

  ^ String with: Character_lf
]

{ #category : 'instance creation' }
RwTonelParser class >> on: aStream filePath: filePathStringOrNil forReader: aTonelReader [
	^ self new
		filePath: filePathStringOrNil;
		stream: aStream;
		packageReader: aTonelReader;
		yourself
]

{ #category : 'instance creation' }
RwTonelParser class >> on: aStream forReader: aTonelReader [
	^ self on: aStream filePath: aStream wrappedStreamName forReader: aTonelReader
]

{ #category : 'instance creation' }
RwTonelParser class >> onString: aString forReader: aTonelReader [
  ^ self on: (self readStreamClass on: aString) forReader: aTonelReader

]

{ #category : 'parsing' }
RwTonelParser class >> parseStream: aStream forReader: aTonelReader [
	^ (self on: aStream forReader: aTonelReader)
		 start

]

{ #category : 'parsing' }
RwTonelParser class >> parseString: aString forReader: aTonelReader [
	^ self parseStream: (self readStreamClass on: aString) forReader: aTonelReader

]

{ #category : 'accessing' }
RwTonelParser class >> readStreamClass [

	^ ReadStreamPortable

]

{ #category : 'accessing' }
RwTonelParser class >> writeStreamClass [

	^ WriteStreamPortable

]

{ #category : 'private' }
RwTonelParser >> _checkMethodOrderFor: selector isMeta: isMeta [
	| col sawClassMethods sawInstanceMethods |
	self checkTonelClassMethodOrder
		ifFalse: [ ^ self ].
	sawClassMethods := (self dynamicInstVarAt: #'sawClassMethods') ifNil: [ false ].
	sawInstanceMethods := (self dynamicInstVarAt: #'sawInstanceMethods')
		ifNil: [ false ].
	col := isMeta
		ifTrue: [ 
			sawInstanceMethods
				ifTrue: [ 
					self
						error:
							'The class method ' , selector
								,
									' is out of order - class methods must come before instance methods in a tonel class file.' ].
			self dynamicInstVarAt: #'sawClassMethods' put: true.
			self dynamicInstVarAt: #'checkTonelClassSelectors' ]
		ifFalse: [ 
			self dynamicInstVarAt: #'sawInstanceMethods' put: true.
			self dynamicInstVarAt: #'checkTonelInstanceSelectors' ].
	col add: selector.
	col last = selector
		ifFalse: [ 
			self
				error:
					'The '
						,
							(isMeta
								ifTrue: [ 'class ' ]
								ifFalse: [ 'instance ' ]) , 'method ' , selector
						,
							' is out of order - run the $ARCHBASE/build/canonicalizeTonelFiles.sh to canonicalize the file.' ]
]

{ #category : 'accessing' }
RwTonelParser >> checkTonelClassMethodOrder [
	"true means that it is an error if method selectors in a tonel class file are not in canonical order"

	^ (self dynamicInstVarAt: #'checkTonelClassMethodOrder') ifNil: [ false ]
]

{ #category : 'accessing' }
RwTonelParser >> checkTonelClassMethodOrder: aBoolean [
	"true means that it is an error if method selectors in a tonel class file are not in canonical order"

	self dynamicInstVarAt: #'checkTonelClassMethodOrder' put: aBoolean.
	aBoolean
		ifTrue: [ 
			self
				dynamicInstVarAt: #'checkTonelInstanceSelectors'
				put: (SortedCollection sortBlock: [ :a :b | a _unicodeLessThan: b ]).
			self
				dynamicInstVarAt: #'checkTonelClassSelectors'
				put: (SortedCollection sortBlock: [ :a :b | a _unicodeLessThan: b ]) ]
]

{ #category : 'private' }
RwTonelParser >> cleanSelector: aString [
	"BEWARE: I'm doing some heave assumptions here: I'm removing just ONE space (in case there 
	 is one) because I expect this to be a file generated by tonel, and tonel adds one space 
	 before start with the method body to make the format more readable. 
	 But of course this is not very good :("
  aString size == 0 ifTrue:[ RwTonelParseError signal:'empty selector string' ].
	^ (aString last = Character space
		ifTrue: [ aString allButLast ]
		ifFalse: [ aString ]) 
		trimLeft

]

{ #category : 'parsing' }
RwTonelParser >> comment [
	| result ch eatNext |
	result := String new writeStreamPortable.
	eatNext := false.
	stream next == $" ifFalse: [ RwTonelParseError signal: 'Can''t parse comment' ].	
	[ stream atEnd not
		and: [ 
				(ch := stream next) ~~ $" 
				or: [ eatNext := (stream peek == $") ] ] ]
	whileTrue: [ 
		result nextPut: ch.
		eatNext ifTrue: [ 
			stream skip: 1.
			eatNext := false ] ].
	^ self 
		removeFrom: '"',result contents,'"' 
		enclosingStart: $"
		end: $"
]

{ #category : 'private factory' }
RwTonelParser >> definitionForType: aString [
  ^ self packageReader definitionForType: aString

]

{ #category : 'parsing' }
RwTonelParser >> document [
  | type mlist |
	type := self typeDef.
	mlist := self methodDefList.
	"Optimized  { type . mlist } select: [:each | each ~~ nil ] "
  type ifNotNil:[  mlist ifNotNil:[ ^ { type . mlist } ]
                         ifNil:[ ^ { type } ]].
	mlist ifNotNil:[ ^ { mlist } ].
  ^ { }
]

{ #category : 'error handling' }
RwTonelParser >> error: messageText [
	^ RwTonelParseError signal: messageText
]

{ #category : 'private' }
RwTonelParser >> extractSelector: aString [
	| separators keywords ofs sz word ch trimmedWord res nKw |
	"separators := { 
		Character space. 
		Character tab. 
		Character lf. 
		Character newPage. 
		Character cr. 
		$: } collect:[:x | x codePoint] "
	[ 
	separators := #(32 9 10 12 13 58).
	keywords := {}.
	sz := aString size.
	ofs := 0.
	[ ofs < sz ]
		whileTrue: [ 
			"no temps in this block to avoid complex block"
			word := String new.
			[ 
			ofs < sz
				and: [ 
					(separators includesIdentical: (ch := aString at: (ofs := ofs + 1)) codePoint)
						== false ] ] whileTrue: [ word add: ch ].
			ch == $:
				ifTrue: [ word add: ch ].
			trimmedWord := word trimBoth.
			trimmedWord size ~~ 0
				ifTrue: [ keywords add: trimmedWord ] ].

	(nKw := keywords size) <= 2
		ifTrue: [ res := keywords at: 1 ]
		ifFalse: [ 
			res := String new.
			1 to: nKw by: 2 do: [ :j | res addAll: (keywords at: j) ] ].
	^ res asSymbol ]
		on: Error
		do: [ :ex | 
			ex addText: ' while extracting selector '.
			ex pass ]
]

{ #category : 'accessing' }
RwTonelParser >> filePath [
	^ filePath ifNil: [ ^ '' ]
]

{ #category : 'accessing' }
RwTonelParser >> filePath: filePathString [
	"set the name of the tonel file that is being read .. may be nil ... used in error messages"

	filePath := filePathString
]

{ #category : 'testing' }
RwTonelParser >> isEnter: aCharacter [
  | cp |
  cp := aCharacter codePoint .
  ^ cp == 10 or:[ cp == 13 ]
]

{ #category : 'testing' }
RwTonelParser >> isSeparator: aCharacter [ 
	"INLINE ^ aCharacter isSeparator"
  ^ aCharacter _unicodeStatus: 29 
]

{ #category : 'accessing' }
RwTonelParser >> lineEndingSize [
  "Must be consistent with   self class lineEnding size"
  ^ 1
]

{ #category : 'parsing' }
RwTonelParser >> metadata [
	| result ch count |
	(ch := stream peek) == ${
		ifFalse: [ 
			"use Error instead of RwTonelParseError, to avoid having try: reset the stream position"
			Error
				signal:
					'Missing method category or class properties. Expected ${, got '
						, ch printString ].
	result := String new.
	count := 0.
	[ stream atEnd ]
		whileFalse: [ 
			ch := stream next.
			result add: ch.
			ch == ${
				ifTrue: [ count := count + 1 ].
			ch == $}
				ifTrue: [ count := count - 1 ].
			count == 0
				ifTrue: [ 
					[ ^ STON fromString: result ]
						on: STONReaderError
						do: [ :ex | 
							ex addText: ', while reading metadata for class or method category'.
							ex pass ] ] ].

	RwTonelParseError signal: 'Can''t parse metadata'
]

{ #category : 'parsing' }
RwTonelParser >> method [
	| type selector |
	
"if there is a single $> then we read until we find the NEXT ONE"
	type := self readClassNameForMethod.
"if there is no leading $[ we read until we find the NEXT ONE"
	selector := self cleanSelector: (self untilExcludingChar: $[ ).
	type := type trimBoth substringsSpace . "substrings: ' ' "
	type size = 1 ifTrue: [ type := type copyWith: nil ].
  lastSelectorParsed := selector .
	^ { 
		type.
		selector.
	}
]

{ #category : 'parsing' }
RwTonelParser >> methodBody [
	"I read a methodbody (what is inside [ ... ])
	 Since a method body can contain enclosing brackets we need to be sure we will skip them and
	 correctly read the method. For that, I have to take into account: 
		- I can mention [] in comments
		- I can mention [] in strings
		- I can use $[, $] 
		- I can have inner blocks
		- I can mention a comment of the form ""$"" or a comment of the form '$'
	 all that needs to be skipped "
	| result char prevChar comment string count startPos |
	(char := stream peek) == $[ 
		ifFalse: [ 
			"use Error instead of RwTonelParseError, to avoid having try: reset the stream position"
			Error signal: 'Missing leading $[ for method body' ].
	startPos := stream position .
	result := String new .
	comment := false.
	string := false.
	prevChar := nil.
	count := 0.

	[ stream atEnd ]
	whileFalse: [ 
		char := stream next.
		result add: char.
		(char == $" and: [ string == false and: [ prevChar ~~ $$ or: [ comment ] ] ]) 
			ifTrue: [ comment := comment ifTrue:[false] ifFalse:[true] ]. 
		(char == $' and: [ comment == "not"false and: [ prevChar ~~ $$ or: [ string ] ] ]) 
			ifTrue: [ string := string "not"ifTrue:[false] ifFalse:[true] ]. 
		(comment or: [ string ]) ifFalse: [ 
			(char == $[ and: [  prevChar ~~ $$ ]) ifTrue: [ count := count +1 ].
			(char == $] and: [ prevChar ~~ $$ ]) ifTrue: [ count := count -1 ] ].
		count == 0 ifTrue: [ 
			^ self 
				removeFrom: result 
				enclosingStart: $[ 
				end: $]
				clean: #right ].
		prevChar := char ].
	stream position: startPos.  "start position is more informative than end position"
	RwTonelParseError signal: 'Can''t parse method body -- missing selector and/or missing trailing $]'
]

{ #category : 'parsing' }
RwTonelParser >> methodDef [

	| methodDef |
	self methodDef: [:isMeta :mDef |
		methodDef :=  mDef.
		"skip possible spaces at the end"
		self separator ].
	^methodDef
]

{ #category : 'parsing' }
RwTonelParser >> methodDef: aBlock [
  | ar def offset |
  ar := {
    self separator.
    self try: [ self metadata ].
    self separator.
    nil .  
    nil .
  }.
  offset := stream position . ar at: 4 put: self method .
  ar at: 5 put: self methodBody .

  (def := self newMethodDefinitionFrom: ar )
    offset: offset
    inFile: self filePath .

  aBlock
    value: (((ar at: 4) at: 1) at: 2) ~~ nil
    value: def
]

{ #category : 'parsing' }
RwTonelParser >> methodDefList [
	| result classMeths meths |
	self separator.	"to arrive to the end of the file in case there are no methods"
	classMeths := {}.
	meths := {}.
	result := {classMeths.
	meths}.
	[ 
	[ stream atEnd ]
		whileFalse: [ 
			self
				methodDef: [ :isMeta :mDef | 
					isMeta
						ifTrue: [ classMeths add: mDef ]
						ifFalse: [ meths add: mDef ].	"skip possible spaces at the end"
					self separator ] ] ]
		on: Error
		do: [ :ex | 
			lastSelectorParsed
				ifNotNil: [ 
					| str |
					str := ex details ifNil: [ '' ].
					ex
						details:
							str , ', last method parsed: ' , lastSelectorParsed printString  ].
			ex pass ].
	^ result
]

{ #category : 'private factory' }
RwTonelParser >> newMethodDefinitionFrom: anArray [
	| metadata className meta selector source categ fourth fourthTwo selsSet |
	metadata := (anArray at: 2) ifNil: [ Dictionary new ].
	className := ((fourth := anArray at: 4) at: 1) at: 1.
	[ Metaclass3 _validateNewClassName: className asSymbol ]
		on: Error
		do: [ :ex | self error: 'Invalid class name ' , className printString ].
	meta := ((fourth at: 1) at: 2) ~~ nil.
	selector := self extractSelector: (fourthTwo := fourth at: 2) trimBoth.
	selsSet := meta
		ifTrue: [ 
			selectors ifNil: [ selectors := SymbolSet new ].
			selectors ]
		ifFalse: [ 
			classSelectors ifNil: [ classSelectors := SymbolSet new ].
			classSelectors ].
	(selsSet includes: selector)
		ifTrue: [ self error: 'Duplicate selector ' , selector ].
	selsSet add: selector.
	self _checkMethodOrderFor: selector isMeta: meta.
	source := String new.
	source addAll: fourthTwo.
	(anArray at: 5) ifNotEmpty: [ :src | source addAll: src ].
	categ := metadata
		at: #'category'
		ifAbsent: [ 
			"to avoid error, resume with default category string"
			(RwTonelParseRequireMethodCategoryNotification
				className: className
				isMeta: meta
				selector: selector) signal ].

	^ self packageReader
		newMethodDefinitionForClassNamed: className
		classIsMeta: meta
		selector: selector
		category: categ
		source: source
]

{ #category : 'private factory' }
RwTonelParser >> newTypeDefinitionFrom: anArray [
	"add stream name and position information to the newly created definition, to provide better error information down stream"

	| offset def |
	offset := stream position.
	def := self packageReader newTypeDefinitionFrom: anArray.
	def offset: offset inFile: self filePath.
	^ def
]

{ #category : 'accessing' }
RwTonelParser >> packageReader [
	^ packageReader

]

{ #category : 'accessing' }
RwTonelParser >> packageReader: aPackageReader [ 
	packageReader := aPackageReader

]

{ #category : 'private stream' }
RwTonelParser >> readClassNameForMethod [
	| expectedChars className rightArrow leftBracket leftBracketChar |
	rightArrow := leftBracket := false.
	leftBracketChar := $[.
	expectedChars := IdentitySet new
		add: $>;
		add: leftBracketChar;
		yourself.
	className := self
		upToAnyOf: expectedChars
		do: [ :ch | 
			rightArrow := ch = $>.
			leftBracket := ch = leftBracketChar ].
	className := className trimBoth.
	leftBracket
		ifTrue: [ 
			rightArrow
				ifFalse: [ self error: 'missing >> in tonel method source' ] ].
	rightArrow
		ifTrue: [ 
			stream next = $>
				ifTrue: [ self skipSeparators ]
				ifFalse: [ self error: 'missing >> in tonel method source' ] ].
	^ className
]

{ #category : 'private' }
RwTonelParser >> removeFrom: aString enclosingStart: startChar end: endChar [
	^ self 
		removeFrom: aString 
		enclosingStart: startChar 
		end: endChar
		clean: #both

]

{ #category : 'private' }
RwTonelParser >> removeFrom: aString enclosingStart: startChar end: endChar clean: cleanSymbol [
  "cleanSymbol can be #left, #rigth and #both"

  | result stop ch start end |
  result := self class readStreamClass on: aString trimBoth.
  result peek == startChar
    ifFalse: [ RwTonelParseError signal: 'I cannot remove enclosing start' ].
  result skip: 1.
  (#(#'both' #'left') includesIdentical: cleanSymbol)
    ifTrue: [ 
      stop := self lineEndingSize.
      [ stop > 0 and: [ self isSeparator: (ch := result peek) ] ]
        whileTrue: [ 
          (self isEnter: ch)
            ifTrue: [ stop := stop - 1 ].
          result skip: 1 ] ].
  start := result position.
  result setToEnd.
  result skip: -1.
  result peek == endChar
    ifFalse: [ RwTonelParseError signal: 'I cannot remove enclosing end' ].
  result skip: -1.
  (#(#'both' #'right') includesIdentical: cleanSymbol)
    ifTrue: [ 
      stop := self lineEndingSize.
      [ stop > 0 and: [ self isSeparator: (ch := result peek) ] ]
        whileTrue: [ 
          (self isEnter: ch)
            ifTrue: [ stop := stop - 1 ].
          result skip: -1 ] ].
  end := result position.
  ^ result contents copyFrom: start + 1 to: end + 1
]

{ #category : 'parsing' }
RwTonelParser >> separator [
	[ stream atEnd not and: [ self isSeparator: stream peek ] ]
	whileTrue: [ stream next ].
	^ nil

]

{ #category : 'private stream' }
RwTonelParser >> skipSeparators [
[stream atEnd]
	whileFalse:
	[stream peek isSeparator ifFalse: [^ self].
	stream next]
]

{ #category : 'accessing' }
RwTonelParser >> start [
	^ self document

]

{ #category : 'accessing' }
RwTonelParser >> stream: aStream [ 
	stream := aStream

]

{ #category : 'private parsing' }
RwTonelParser >> try: aBlock [
	"^ self 
		try: aBlock 
		onSuccess: [ :parsedValue | parsedValue ] 
		onFailure: [ nil ]"
  | pos |
  pos := stream position. 
  ^ [ aBlock value ] 
    on: RwTonelParseError 
    do:[:ex | stream position: pos . nil ]

]

{ #category : 'private parsing' }
RwTonelParser >> try: aBlock onSuccess: successBlock [
	"^ self 
		try: aBlock 
		onSuccess: successBlock 
		onFailure: [ nil ] "
  | pos |
  pos := stream position.
  ^ [ successBlock value: aBlock value ]
    on: RwTonelParseError
    do:[:ex | stream position: pos . nil ]
]

{ #category : 'private parsing' }
RwTonelParser >> try: aBlock onSuccess: successBlock onFailure: failureBlock [
	| pos |
	pos := stream position.
	[ ^ successBlock value: aBlock value ]
	on: RwTonelParseError 
	do: [ :e | 
		stream position: pos.
		^ failureBlock value ]. 
]

{ #category : 'parsing' }
RwTonelParser >> type [
  stream peek ifNotNil:[ :ch |
    ch == $E ifTrue:[ self try: [ self word: 'Extension' ] onSuccess: [ :word | ^ word  ]].
	  ch == $C ifTrue:[ self try: [ self word: 'Class' ] onSuccess: [ :word | ^ word  ]].
   	self try: [ self word: 'Trait' ] onSuccess: [ :word | ^ word  ].
  ].	
	"at end"
	RwTonelParseError signal: 'Can''t parse type.'	

]

{ #category : 'parsing' }
RwTonelParser >> typeDef [
	^ self
		newTypeDefinitionFrom:
			{(self separator).
			(stream peek == $"
				ifTrue: [ self try: [ self comment ] ]).
			(self separator).
			(self type).
			(self separator).
			(stream peek == ${
				ifTrue: [ 
					self
						try: [ 
							| typeMetadata normalizedMetadata |
							typeMetadata := self metadata.
							normalizedMetadata := SymbolDictionary new.
							typeMetadata
								keysAndValuesDo: [ :key :value | normalizedMetadata at: key asLowercase asSymbol put: value ].
							normalizedMetadata ] ])}
]

{ #category : 'private stream' }
RwTonelParser >> untilExcludingChar: aCharacter [
	| result |
	result := stream upTo: aCharacter.
	stream position: stream position - 1 .
	^ result
]

{ #category : 'private stream' }
RwTonelParser >> upToAnyOf: subcollection do: aBlock [
"Answer a subcollection from the current access position to the occurrence (if any, but not inclusive) of any object in the collection.
Evaluate aBlock with this occurence as argument.
If no matching object is found, don't evaluate aBlock and answer the entire rest of the receiver."

| resultStream ch |
resultStream := AppendStream on: stream collectionSpecies new.
[ stream atEnd or: [ (subcollection includes: (ch := stream next)) and: [aBlock value: ch. true] ] ]
	whileFalse: [ resultStream nextPut: ch ].
^ resultStream contents
]

{ #category : 'private stream' }
RwTonelParser >> word: aString [
	| result |
	result := stream next: aString size.
	result = aString
		ifFalse: [ RwTonelParseError signal: 'Can''t parse ', aString ].
	^ result

]
