"
RowanBrowserService handles services that fall outside
the scope of other services with an emphasis on browser-
type tasks.
"
Class {
	#name : 'RowanBrowserService',
	#superclass : 'RowanService',
	#instVars : [
		'projects',
		'removedMethods',
		'allClasses',
		'hierarchyServices',
		'testPackages',
		'testCount',
		'dictionaries',
		'selectedClass',
		'newCachedSelectors',
		'newCachedClasses'
	],
	#category : 'Rowan-Services-Core'
}

{ #category : 'rsr' }
RowanBrowserService class >> templateClassName [

	^#RowanBrowserService
]

{ #category : 'client commands' }
RowanBrowserService >> abortTransaction [
  | autoCommitService autoCommitState |
  autoCommitState := RowanService autoCommit == #'failed'
    ifTrue: [ true ]
    ifFalse: [ RowanService autoCommit ].
  System abortTransaction.
  autoCommitService := RowanAutoCommitService new.
  autoCommitService autoCommit: autoCommitState.
  self updateProjects.
  self updateDictionaries
]

{ #category : 'client commands' }
RowanBrowserService >> abortTransactionAndUpdateServices: services [
	self abortTransaction.
	services
		do: [ :service | 
			service
				organizer: self organizer;
				updateLatest ]
]

{ #category : 'client commands' }
RowanBrowserService >> allClasses [
	allClasses := self basicAllClasses.
	updateType := #dontUpdate. 
	RowanCommandResult addResult: self.
	^allClasses
]

{ #category : 'client commands' }
RowanBrowserService >> allPackages [
	| theProjects |
	theProjects := Rowan image loadedProjects asArray.
	projects := theProjects
		collect: [ :project | 
			| projectService |
			projectService := RowanProjectService newNamed: project name.
			projectService packages: projectService packageServices.
			projectService packages
				do: [ :packageService | packageService projectName: project name ].
			projectService ].
	updateType := #'dontUpdate'.
	RowanCommandResult addResult: self
]

{ #category : 'client commands support' }
RowanBrowserService >> basicAllClasses [
	| theClasses packageNames |
	theClasses := SortedCollection sortBlock: [ :x :y | x name < y name ].
	packageNames := Rowan image packageNames.
	theClasses
		addAll:
			(self organizer classes
				collect: [ :theClass | 
					| service |
					service := RowanClassService new name: theClass name.
					service
						packageName: (self computePackageNameFor: theClass in: packageNames).
					service projectName: theClass rowanProjectName.
					service ]).
	^ theClasses asArray
]

{ #category : 'client commands' }
RowanBrowserService >> checkForAddedProjects: projectServices [
	| loadedProjects addedProjects loadedProjectServices |
	loadedProjects := Rowan image loadedProjects.
	addedProjects := Array new.
	loadedProjectServices := loadedProjects
		collect: [ :project | RowanProjectService newNamed: project name ].
	loadedProjectServices
		do: [ :loadedProjectService | 
			(projectServices includes: loadedProjectService)
				ifFalse: [ addedProjects add: loadedProjectService ] ].
	addedProjects
		do: [ :projectService | RowanCommandResult addResult: projectService update ]
]

{ #category : 'client commands' }
RowanBrowserService >> classHierarchy [
	| theClasses |
	theClasses :=  self allClasses collect:[:classService | classService theClass].
	hierarchyServices := self classHierarchy: theClasses. 
	updateType := #classHierarchyUpdate:browser:. 
	RowanCommandResult addResult: self.
]

{ #category : 'private' }
RowanBrowserService >> classHierarchy: theClasses [
  hierarchyServices := super classHierarchy: theClasses.
  ^ hierarchyServices
]

{ #category : 'client commands' }
RowanBrowserService >> classHierarchyForDictionariesNamed: dictionaryNames [
  | theClasses |
  theClasses := OrderedCollection new.
  dictionaryNames
    do: [ :dictionaryName | 
      (Rowan globalNamed: dictionaryName)
        keysAndValuesDo: [ :key :value | 
          value isClass
            ifTrue: [ theClasses add: value ] ] ].
  hierarchyServices := self classHierarchy: theClasses.
  updateType := #'classHierarchyUpdate:browser:'.
  RowanCommandResult addResult: self
]

{ #category : 'client commands' }
RowanBrowserService >> classHierarchyForPackagesNamed: packageNames [
  | theClasses |
  theClasses := OrderedCollection new.
  packageNames
    do: [ :packageName | 
      theClasses
        addAll:
          ((Rowan image loadedPackageNamed: packageName) loadedClasses 
            collect: [ :cls | cls handle]).
      theClasses
        addAll:
          ((Rowan image loadedPackageNamed: packageName) loadedClassExtensions
            collect: [ :cls | cls handle]) ].
  hierarchyServices := self classHierarchy: theClasses.
  updateType := #'classHierarchyUpdate:browser:'.
  RowanCommandResult addResult: self
]

{ #category : 'client commands' }
RowanBrowserService >> clearWindowRegistry [
	SessionTemps current at: #'rowanServicesWindowRegistry' put: Dictionary new
]

{ #category : 'client commands' }
RowanBrowserService >> commitTransaction [
	"makes sure our permanent method history does not have any rsr inst var values as Semaphores can prevent commits"

	| answeringService |
	self rowanMethodHistory
		keysAndValuesDo: [ :key :value | 
			key nilRsrVariables.
			value do: [ :svc | svc nilRsrVariables ] ].
	answeringService := RowanAnsweringService new answer: System commitTransaction.
	RowanCommandResult addResult: answeringService.
	^answeringService answer
]

{ #category : 'client commands' }
RowanBrowserService >> compileClass: definitionString [
  | anonymousMethod |
  self confirmDuplicateName: definitionString.
  anonymousMethod := definitionString
    _compileInContext: nil
    symbolList: Rowan image symbolList.
  SessionTemps current at: #'jadeiteCompileClassMethod' put: anonymousMethod
]

{ #category : 'client commands support' }
RowanBrowserService >> confirmDuplicateName: definitionString [
  | className |
  className := (definitionString subStrings at: 3) copyWithout: $'.
  (Rowan image symbolList resolveSymbol: className asSymbol)
    ifNotNil: [ 
      className = selectedClass
        ifFalse: [ 
          (self confirm: 'Class name is already an object. Continue?')
            ifFalse: [ ^ Error signal: 'Class not compiled. Name already exists.' ] ] ]
]

{ #category : 'client commands' }
RowanBrowserService >> defaultClassHierarchy [
	hierarchyServices := Dictionary new.   
	self organizer hierarchy keysAndValuesDo: [:key :value |
		| classService |
		classService := key == #nil ifTrue:[#nil] ifFalse: [RowanClassService basicForClassNamed: key name].
		hierarchyServices at: classService put: (value collect:[:cls | RowanClassService basicForClassNamed: cls name]) asArray.
	].
	updateType := #classHierarchyUpdate:browser:. 
	RowanCommandResult addResult: self.
]

{ #category : 'client commands' }
RowanBrowserService >> dictionariesWithTests [
	"for now, just return the dictionaries" 
	self organizer: ClassOrganizer new.	"when we call this method, our world has changed from a reload, etc."
	dictionaries := Rowan image symbolList names collect:[:name | RowanDictionaryService new name: name].
	updateType := #'testDictionaries:'.
	RowanCommandResult addResult: self
]

{ #category : 'client commands' }
RowanBrowserService >> findRemovedServices: services [

	services do:[:service | 
		service wasDeleted ifTrue:[
				service updateType: #removed:.
				RowanCommandResult addResult: service.
		]
	].
]

{ #category : 'initialize' }
RowanBrowserService >> initialize [
  super initialize.
  newCachedSelectors := Array new.
  newCachedClasses := Array new
]

{ #category : 'client commands' }
RowanBrowserService >> jfpwor_compileClass: definitionString [

	| newClass newClassService newMetaClassService |
	newClass := definitionString evaluate.
	newClassService := RowanClassService new name: newClass.
	newClassService update.
	newClassService updateType: #newClass:browser:.
	newMetaClassService := RowanClassService new name: newClass.
	newMetaClassService meta: true.
	newMetaClassService update.
	newMetaClassService updateType: #newClass:browser:.
	newClassService version > 1 ifTrue: [
		| oldClass oldClassService |
		oldClass := newClassService theClass classHistory at:
			            newClassService version - 1.
		oldClassService := RowanClassService new classServiceFromOop:
			                   oldClass asOop.
		oldClassService update.
		self
			jfpwor_compileMethodsFrom: oldClassService
			to: newClassService
			and: newMetaClassService.
		self
			jfpwor_recompileSubclassesFrom: oldClassService
			to: newClassService ].
	RowanCommandResult
		addResult: self;
		addResult: newClassService;
		addResult: newMetaClassService. "bring back class & instance side"
	selectedClass := newClassService

]

{ #category : 'client commands support' }
RowanBrowserService >> jfpwor_compileMethodsFrom: oldClassService to: newClassService and: newMetaClassService [

	| methodCompileErrors |
	methodCompileErrors := Array new.
	oldClassService methods do: [ :methodService |
		[
		newClassService
			saveMethodSource: methodService source
			category: methodService category ]
			on: CompileError
			do: [ :ex |
				| errorMethodService |
				errorMethodService := self
					                      jfpwor_methodServiceForCompileError: ex
					                      source: methodService source
					                      className: newClassService name.
				errorMethodService
					selector: methodService selector;
					category: methodService category.
				methodCompileErrors add: errorMethodService ] ].
	newClassService methodsFailingRecompilation: methodCompileErrors.
	oldClassService
		meta: true;
		update.
	oldClassService methods do: [ :methodService |
		[
		newMetaClassService
			saveMethodSource: methodService source
			category: methodService category ]
			on: CompileError
			do: [ :ex |
				| errorMethodService |
				errorMethodService := self
					                      jfpwor_methodServiceForCompileError: ex
					                      source: methodService source
					                      className: newMetaClassService name.
				errorMethodService
					selector: methodService selector;
					category: methodService category.
				methodCompileErrors add: errorMethodService ] ].
	newMetaClassService methodsFailingRecompilation: methodCompileErrors

]

{ #category : 'client commands support' }
RowanBrowserService >> jfpwor_methodServiceForCompileError: ex source: source className: className [

	| errorMethodService errorOffset errorText |
	errorOffset := ex gsArguments first first at: 2.
	errorText := ex gsArguments first first at: 3.
	errorMethodService := RowanMethodService new.
	errorMethodService source: source copy.
	errorMethodService source insertAll: errorText , '->' at: errorOffset.
	errorMethodService
		className: className;
		meta: false;
		firstReference: errorOffset;
		searchString: errorText.
	^errorMethodService

]

{ #category : 'client commands support' }
RowanBrowserService >> jfpwor_recompileSubclassesFrom: oldClassService to: newClassService [

        oldClassService theClass subclasses do: [ :subclass |
                | subclassService |
                subclassService := RowanClassService new name: subclass name.
                self jfpwor_compileClass: subclassService classCreationTemplate ]
]

{ #category : 'accessing' }
RowanBrowserService >> newCachedClasses [
	newCachedClasses ifNil: [newCachedClasses := Array new]. 
	^newCachedClasses
]

{ #category : 'accessing' }
RowanBrowserService >> newCachedClasses: object [
	newCachedClasses := object
]

{ #category : 'accessing' }
RowanBrowserService >> newCachedSelectors [
	^newCachedSelectors
]

{ #category : 'accessing' }
RowanBrowserService >> newCachedSelectors: object [
	newCachedSelectors := object
]

{ #category : 'client commands' }
RowanBrowserService >> newProjectNamed: projectName [
	| definedProject service |
	definedProject := (Rowan newProjectNamed: projectName)
		addLoadComponentNamed: 'Core';
		packageConvention: 'Rowan';
		gemstoneSetDefaultSymbolDictNameTo: 'UserGlobals';
		repoType: 'disk';
		packageFormat: 'tonel';
		projectsHome: '$ROWAN_PROJECTS_HOME';
		yourself.
	(definedProject projectsHome asFileReference / projectName / 'rowan')
		ensureDeleteAll.
	definedProject resolveProject write.
	service := RowanProjectService new name: projectName.
	service
		installProjectFromURL:
			'file:$ROWAN_PROJECTS_HOME/' , projectName , '/rowan/specs/' , projectName
				, '.ston'.
	service loadProjectNamed: projectName.
	self updateProjects
]

{ #category : 'client commands' }
RowanBrowserService >> newProjectNamed: projectName windowHandle: handle [
	| definedProjectService project |
	definedProjectService := RowanDefinedProjectService new name: projectName.
	definedProjectService projectOop: (project := Rowan newProjectNamed: projectName) asOop.
	self saveRootObject: definedProjectService projectOop windowHandle: handle. 
	definedProjectService := definedProjectService updateType: #newProject:.
	definedProjectService specService: (RowanLoadSpecService new initialize: project loadSpecification asOop).
	RowanCommandResult addResult: definedProjectService
]

{ #category : 'window registry' }
RowanBrowserService >> openWindows [

	"for testing"

	^SessionTemps current at: #rowanServicesWindowRegistry ifAbsent:[]
]

{ #category : 'client commands' }
RowanBrowserService >> packagesWithTests [
	| packageNames |
	self organizer: ClassOrganizer new.	"when we call this method, our world has changed from a reload, etc."
	testPackages := Set new.
	testCount := 0.
	packageNames := Rowan image packageNames.  
	(self organizer allSubclassesOf: TestCase)
		do: [ :sub | 
			| packageName testMethodCount |
			testMethodCount := (sub sunitSelectors
				select: [ :each | each beginsWith: 'test' ]) size.	"sending #testSelectors was slower"
			testCount := testCount + testMethodCount.
			testMethodCount > 0
				ifTrue: [ 
					packageName := self computePackageNameFor: sub in: packageNames.
					packageName = Rowan unpackagedName
						ifFalse: [ 
							testPackages
								add:
									((RowanPackageService forPackageNamed: packageName)
										updateProjectName;
										yourself) ].
					(Rowan image loadedClassExtensionsForClass: sub)
						do: [ :loadedThing | 
							testPackages
								add:
									((RowanPackageService forPackageNamed: loadedThing loadedPackage name)
										updateProjectName;
										yourself)	"don't update the entire package for performance improvement" ] ] ].
	updateType := #'testPackages:'.
	testPackages := testPackages asArray.
	RowanCommandResult addResult: self
]

{ #category : 'client commands' }
RowanBrowserService >> recompileMethodsAfterClassCompilation [
	"compileClass: must be run first"

	| theClass classService packageService projectService methodCompileErrors |
	methodCompileErrors := Array new.

	theClass := [ 
	[ (SessionTemps current at: #'jadeiteCompileClassMethod') _executeInContext: nil ]
		on: RwCompileErrorCompilingMethodsForNewClassVersionNotification
		do: [ :ex | 
			| methodService errorOffset errorText |
			errorOffset := ex compileError gsArguments first first at: 2.
			errorText := ex compileError gsArguments first first at: 3.
			methodService := RowanMethodService new.
			methodService source: ex compileError sourceString.
			methodService source insertAll: errorText , '->' at: errorOffset.
			methodService
				className: (ex compileError gsArguments at: 3);
				meta: false;
				firstReference: errorOffset;
				searchString: errorText.
			methodCompileErrors add: methodService.
			ex resume: true	"method compile error will be handled later" ] ]
		ensure: [ SessionTemps current at: #'jadeiteCompileClassMethod' put: nil ].
	[ 
	classService := RowanClassService new name: theClass name.
	classService update.
	classService updateSubclasses.
	classService isNewClass: true.	"if nothing else, the dirty state of the package/project services
	should be updated. Would like a less heavy weight solution than this, though."
	packageService := RowanPackageService forPackageNamed: classService packageName.
	packageService update.
	packageService
		testClasses;
		updateType: nil.
	projectService := RowanProjectService newNamed: packageService projectName.
	projectService update.
	packageService selectedClass: classService.
	RowanCommandResult addResult: classService.
	selectedClass := classService.
	classService updateType: #'newClass:browser:'.
	self updateSymbols: (Array with: theClass name asString).
	classService classHierarchy.
	classService methodsFailingRecompilation: methodCompileErrors.
	RowanCommandResult addResult: self.
	^ classService ]
		on: Error
		do: [ :ex | 
			Transcript
				cr;
				show: 'ERROR post class processing' ]
]

{ #category : 'client commands' }
RowanBrowserService >> releaseWindowHandle: integer [
	| registry object |
	registry := SessionTemps current
		at: #'rowanServicesWindowRegistry'
		ifAbsent: [ ^ self ].
	object := registry at: integer ifAbsent: [ ^ self ].
	Rowan loggingServiceClass current
		logComment:
			'Release object with oop: ' , object asOop printString , ' window handle: '
				, integer printString.
	registry removeKey: integer ifAbsent: [  ]
]

{ #category : 'client commands' }
RowanBrowserService >> reloadProjects: projectServices andUpdateServices: services [
	| projectNames answeringService |
	services do: [ :service | service organizer: self organizer ].
	projectServices do: [ :service | service organizer: self organizer ].
	projectServices do: [ :projectService | projectService reloadProject ].
	projectNames := projectServices
		collect: [ :projectService | projectService name ].
	services
		do: [ :service | 
			(projectNames includes: service rowanProjectName)
				ifTrue: [ service updateLatest ] ].
	answeringService := RowanAnsweringService new organizer: self organizer.
	answeringService updateAutocompleteSymbols.
]

{ #category : 'client commands' }
RowanBrowserService >> removeDictionariesNamed: dictionaryNames [
	"remove from both transient & persistent symbol lists" 

	dictionaryNames do:[:dictionaryName | 
		| dictionaryNameSymbol |
		dictionaryNameSymbol := dictionaryName asSymbol.
		(Rowan image symbolList names includes: dictionaryNameSymbol) ifTrue:[
			Rowan image symbolList removeDictionaryNamed: dictionaryNameSymbol].
		(System myUserProfile symbolList names includes: dictionaryNameSymbol) ifTrue:[
			System myUserProfile symbolList removeDictionaryNamed: dictionaryNameSymbol]].
	self updateDictionaries.
]

{ #category : 'client commands' }
RowanBrowserService >> removeMethods: methodServices [

	| notRemoved |
	notRemoved := Array new. 
	removedMethods := Array new. 
	methodServices do: [:methodService |
		| classService |
		classService := RowanClassService forClassNamed: methodService className. 
		classService meta: methodService meta. 
		classService removeSelector: methodService selector ifAbsent:[notRemoved add: methodService].
		classService updatePackageAndProject.
		(notRemoved includes: methodService) ifFalse:[
			methodService updateType: #removed:.
			RowanCommandResult addResult: methodService]. 
		removedMethods add: methodService].
	notRemoved isEmpty ifFalse:[
		self error: 'These selectors were not removed - ', (notRemoved collect:[:ea | ea selector]) printString].
	updateType := #methodsRemoved:. 
	RowanCommandResult addResult: self.
]

{ #category : 'client commands' }
RowanBrowserService >> saveRootObject: oop windowHandle: integer [
	" a window has been opened on the client. Save the 
	root object of the window so it won't be recycled"

	| registry object |
	registry := SessionTemps current
		at: #'rowanServicesWindowRegistry' 
		ifAbsentPut: [ Dictionary new ].
	registry at: integer ifAbsentPut: [ Array new ].
	object := Object _objectForOop: oop.
	(object isKindOf: GsProcess)
		ifTrue: [ RowanDebuggerService new saveProcessOop: object asOop	"make sure the process oop gets saved beyond the life of a debugger or notifier" ].
	((registry at: integer) includes: object)
		ifFalse: [ 
			object := registry at: integer ifAbsent: [ ^ self ].
			Rowan loggingServiceClass current
				logComment:
					'Saving object with oop: ' , object asOop printString , ' window handle: '
						, integer printString.
			(registry at: integer) add: (Object _objectForOop: oop) ]
]

{ #category : 'accessing' }
RowanBrowserService >> selectedClass [

	^selectedClass
]

{ #category : 'accessing' }
RowanBrowserService >> selectedClass: object [

	selectedClass := object
]

{ #category : 'client commands' }
RowanBrowserService >> turnOffNativeCode [
	"setting a breakpoint anywhere in the system will turn off native code"

	| methodService |
	methodService := RowanMethodService new
		selector: #'methodForTurningOffNativeCode';
		meta: false;
		className: 'RowanBrowserServiceServer'.
	methodService setBreakAt: 1.
]

{ #category : 'client commands' }
RowanBrowserService >> unloadProjectsNamed: projectNames [
	projectNames
		do: [ :projectName | 
			| project |
			project := RwProject newNamed: projectName.
			project isLoaded
				ifTrue: [ project unload ] ].
	self updateProjects
]

{ #category : 'client commands' }
RowanBrowserService >> unsetCurrentPackage [
	Rowan gemstoneTools topaz currentTopazPackageName: nil.
	self updateProjects.
	updateType := #resetCurrentProjectPackage.
]

{ #category : 'client commands' }
RowanBrowserService >> unsetCurrentProject [
	Rowan gemstoneTools topaz currentTopazProjectName: nil.
	Rowan gemstoneTools topaz currentTopazPackageName: nil.
	self updateProjects.
	updateType := #resetCurrentProjectPackage.
]

{ #category : 'client commands' }
RowanBrowserService >> unsetSecretBreakpoint [
	"used for turning off native code"

	| methodService |
	methodService := RowanMethodService new
		selector: #'methodForTurningOffNativeCode';
		meta: false;
		className: 'RowanBrowserServiceServer'.
	methodService clearBreakAt: 1.
]

{ #category : 'client commands' }
RowanBrowserService >> updateDictionaries [
	dictionaries := Rowan image symbolList names
		collect: [ :name | RowanDictionaryService new name: name asString ].
	dictionaries := dictionaries asOrderedCollection.
	updateType ifNil: [ updateType := OrderedCollection new ].
	updateType add: #'dictionaryListUpdate:'.
	RowanCommandResult addResult: self
]

{ #category : 'client commands' }
RowanBrowserService >> updateProjects [
  | sortedProjects |
  self packagesWithTests. "make sure tests are always updated" 
  sortedProjects := SortedCollection sortBlock: [ :a :b | a name < b name ].
  sortedProjects addAll: Rowan image loadedProjects.
  projects := sortedProjects
    collect: [ :project | RowanProjectService newNamed: project name ].
  updateType := Array with: updateType with: #'projectsUpdate:browser:'. "temporary hack" 
  RowanCommandResult addResult: self
]

{ #category : 'update' }
RowanBrowserService >> updateSymbols: classNames [
  self newCachedClasses addAll: classNames.
  updateType := #'addCachedSymbols:'
]
