!=========================================================================
! Copyright (C) GemTalk Systems 1986-2025.  All Rights Reserved.
!
! INDEXING/QUERY GOODIES
!
! Description -
!   This file contains various goodies for getting information about
!   indexes, query execution, etc.  
!
!   This is unsupported code available to aid customer support and consultants.
!
!   You must be logged in as SystemUser to file in this code.
!
!   ObsoleteIDX - This code may become obsolete in an upcoming release of GSS64.
!
!========================================================================


!------------------------------------------------------------------
! goodie #1.
! Method to explain the order of query evaluation and usage of indexes
!------------------------------------------------------------------
!
! This code outputs a description of the order of predicate evaluation and
! usage of indexes when querying NSCs.
! This code is provided as a "goodie", but is not part of the
! supported class library.
!
! The following example illustrates how to use this code:
!
! myBagOfEmployees explainQuery: { :emp |
!     (emp.address.zip == 97223) &
!     (emp.address.state = 'OR') &
!     (emp.age > 30) }
!
! example output of above:
!
!  The query predicates will be executed in the following order:
!  1.   address.state = OR (utilizing an index on 'address.state')
!  2.   address.zip == 97223
!  3.   age > 30
!
!------------------------------------------------------------------


category: 'ObsoleteIDX - Goodies'
method: QueryExecuter
_explainQuery: anArray

" anArray is a four element Array of Arrays describing a SelectBlock
in the manner needed to process a query on the receiver :
    1. Array of the bound variables from the predicate
    2. Array of the predicate's terms
    3. Array of the predicate's paths
    4. Array of Strings of the path names used in the predicate "

| ordering offset str predicateType index1 index2 searchOp ops vals cr |
ops := #( #< #> #= #== #<= #>= #~= #~~ #unary #dual).

" get a description of the operations to invoke on indexes "
self _buildIndexOperationsList: anArray.

self optimize.

" get the order to invoke the operations "
ordering := self _getOperationsOrder.

ordering isEmpty
  ifTrue: [
    ^ 'It has been determined that no elements can satisfy the query as posed.'
  ].

str := String new.
cr := Character lf.
str add: 'The query predicates will be executed in the following order:';
  add: cr.

1 to: ordering size do: [ :i |
  str add: i asString; add: '.   '.

  offset := ordering at: i.
  predicateType := self at: offset.
  searchOp := ops at: (self at: offset + 2) + 1.

  " if it is path-constant or constant-path "
  ( predicateType == 2 or: [ predicateType == 3 ] )
    ifTrue: [
      index1 := self at: offset + 1.
      searchOp = #dual
        ifTrue: [
          vals := self at: offset + 3.
          str add: (vals at: 2) asString; add: ' ';
            add: (ops at: (self _inverseOperatorFor: (vals at: 1)) + 1);
            add: ' '; add: index1 pathComponentsString; add: ' ';
            add: (ops at: (vals at: 3) + 1); add: ' ';
            add: (vals at: 4) asString.
          index1 isPathEvaluator
            ifFalse: [
              str add: ' (utilizing an index on '; add: $';
                add: index1 pathComponentsString; add: $'; add: ')'
            ].
          str add: cr
        ]
        ifFalse: [
          searchOp = #unary
            ifTrue: [
              str add: index1 pathComponentsString.
              index1 isPathEvaluator
                ifTrue: [ str add: ' ' ]
                ifFalse: [ str add: ' (utilizing an index)'; add: cr ].
            ]
            ifFalse: [
              str add: index1 pathComponentsString; add: ' ';
                add: searchOp;  add: ' ';
                add: (self at: offset + 3) asString.
              index1 isPathEvaluator 
                ifFalse: [
                  str add: ' (utilizing an index on '; add: $';
                    add: index1 pathComponentsString; add: $'; add: ')'
                ].
              str add: cr
            ]
        ]
    ].

  " if it is constant-constant "
  predicateType == 1
    ifTrue: [
      searchOp = #unary
        ifTrue: [
          str add: (self at: offset + 1) asString; add: cr
        ]
        ifFalse: [
          str add: (self at: offset + 1) asString; add: ' ';
            add: searchOp;  add: ' ';
            add: (self at: offset + 3) asString; add: cr
        ]
    ].

  " if it is path-path "
  predicateType == 4
    ifTrue: [
      index1 := self at: offset + 1.
      index2 := self at: offset + 3.
      str add: index1 pathComponentsString; add: ' ';
        add: searchOp; add: ' ';
        add: index2 pathComponentsString; add: cr
    ].
].

^ str
%
category: 'ObsoleteIDX - Goodies'
method: UnorderedCollection
explainQuery: aBlock

" Answer a string that shows the order of predicate evaluation for the query. "

| result |
  (aBlock class == SelectBlock)
      ifTrue:[
	  result := (QueryExecuter on: self)
	      _explainQuery: (aBlock queryBlock value: nil).
      ]
      ifFalse:[ result := 
  'This query will be executed by iterating through all elements,
  executing the code in the block for each element.'
      ].
^ result
%

!--------------------------------------------------------------------------
!
!------------------------------------------------------------------
! goodie #2.
! Methods to get internal indexing objects for locking purposes.
!------------------------------------------------------------------
!
! This code is provided as a "goodie", but is not part of the
! supported class library.
!
! Public methods
!
! anObject _getIndexObjectsToLock
!   returns a set of B-tree nodes and other application objects along
!   index paths
!
! anObject _getIndexObjectsToLockOnPath: pathString
!   returns a set of B-tree nodes and other application objects along
!   the given index path
!
! anNsc _getIndexObjectsToLock
!   returns a set of all B-tree nodes for each equality index
!
! anNsc _getIndexObjectsToLockOnPath: pathString
!   returns a set of all B-tree nodes for the given equality index
!
!------------------------------------------------------------------
!

category: 'ObsoleteIDX - Indexing Support (Locking)'
method: Object
_getIndexObjectsToLock

"Return an set of internal indexing objects to lock to ensure that
modifications to the receiver will be able to be committed."

 | result depList |
  depList := DependencyList for: self.
  depList == nil
    ifTrue: [
      ^ IdentitySet new
    ].

  result := IdentitySet new.
  " for each path term ... "
  1 to: depList size by: 2 do: [ :i |
    (depList at: i) _putIndexObjectsToLockFor: self into: result.
  ].
  ^ result
%

category: 'ObsoleteIDX - Indexing Support (Locking)'
method: Object
_getIndexObjectsToLockOnPath: pathString

"Returns a set of internal indexing objects to lock to ensure that
 modifications to the receiver will be able to be committed.  Only
 indexes with the given path are considered."

| result depList pathArray |
  depList := DependencyList for: self.
  depList == nil
    ifTrue: [
      ^ IdentitySet new
    ].

  pathArray := pathString asArrayOfPathTerms.
  result := IdentitySet new.
  " for each path term ... "
  1 to: depList size by: 2 do: [ :i |
    (depList at: i) _putIndexObjectsToLockFor: self
      into: result
      path: pathArray
  ].
^ result
%

category: 'ObsoleteIDX - Indexing Support (Locking)'
method: UnorderedCollection
_getIndexObjectsToLock

"Returns a set of internal indexing objects to lock to ensure that
 modifications to the receiver will be able to be committed.  This
 returns a set of all B-tree nodes for each equality index."

| result iList indexObj |
  result := super _getIndexObjectsToLock.

  iList := self _indexedPaths.
  iList == nil
    ifTrue: [
      ^ IdentitySet new
    ].

  1 to: iList size by: 2 do: [ :j |
    indexObj := iList at: j.
    " lock all B-tree nodes of equality index "
    indexObj isRangeEqualityIndex
      ifTrue: [
	indexObj btreeRoot _preOrderDo: [ :node | result add: node ]
      ]
  ].
^ result
%

category: 'ObsoleteIDX - Indexing Support (Locking)'
method: UnorderedCollection
_getIndexObjectsToLockOnPath: pathString

"Returns a set of internal indexing objects to lock to ensure that
 modifications to the receiver will be able to be committed.  This returns
 a set of all B-tree nodes for the equality index with the given path."

| result indexObj |

  self _indexedPaths == nil
    ifTrue: [
      ^ IdentitySet new
    ].

  indexObj := self _findRangeIndexWithPath: pathString asArrayOfPathTerms.
  result := IdentitySet new.

  " lock all B-tree nodes of equality index "
  indexObj ~~ nil
    ifTrue: [
      indexObj btreeRoot _preOrderDo: [ :node | result add: node ]
    ].
^ result
%

category: 'ObsoleteIDX - Indexing Support (Locking)'
method: PathTerm
_putIndexObjectsToLockFor: anObject into: set

"Place into a set any internal indexing objects to lock to ensure that
modifications to the anObject can be committed."

| ivOffset nextObj |

( nil == anObject or: [ self size == 0 ] )
  ifTrue: [ ^ set ].

(ivOffset := self _ivOffsetFor: anObject) == nil
  ifTrue: [
    anObject _errorInvalidOffset: name .
    self _uncontinuableError
  ].

" get the next object along the path "
nextObj := self _nextObjectFor: anObject atInstVar: ivOffset.

updateBtree ~~ nil
  ifTrue: [ " need to lock B-tree nodes "
    updateBtree btreeRoot _putNodesToLockForKey: nextObj value: anObject into: set
  ].

" no need to add index dictionary because it is RC
updateDict ~~ nil
  ifTrue: [ set add: updateDict ].
"

set add: anObject.

nil == nextObj
  ifTrue: [ ^ set ].

1 to: children size do: [ :i |
  " make recursive call to add mappings "
  (children at: i) _putIndexObjectsToLockFor: nextObj into: set
].
^ set
%

category: 'ObsoleteIDX - Indexing Support (Locking)'
method: PathTerm
_putIndexObjectsToLockFor: anObject into: set path: pathArray

"Place into a set any internal indexing objects to lock to ensure that
modifications to the anObject can be committed."

| ivOffset nextObj |

(pathArray at: offset) = name
  ifFalse: [ ^ set ].

( nil == anObject or: [ self size == 0 ] )
  ifTrue: [ ^ set ].

(ivOffset := self _ivOffsetFor: anObject) == nil
  ifTrue: [
    anObject _errorInvalidOffset: name .
    self _uncontinuableError
  ].

" get the next object along the path "
nextObj := self _nextObjectFor: anObject atInstVar: ivOffset.

updateBtree ~~ nil
  ifTrue: [ " need to lock B-tree nodes "
    updateBtree btreeRoot _putNodesToLockForKey: nextObj value: anObject into: set
  ].

" no need to add index dictionary because it is RC
updateDict ~~ nil
  ifTrue: [ set add: updateDict ].
"

set add: anObject.

nil == nextObj
  ifTrue: [ ^ set ].

1 to: children size do: [ :i |
  " make recursive call to add mappings "
  (children at: i) _putIndexObjectsToLockFor: nextObj into: set path: pathArray
].
^ set
%

category: 'ObsoleteIDX - Indexing Support (Locking)'
method: SetValuedPathTerm
_putIndexObjectsToLockFor: anNsc into: set

"Place into a set any internal indexing objects to lock to ensure that
modifications to the anObject can be committed."

| setElement sz |

( nil == anNsc or: [ self size == 0 ] )
    ifTrue: [ ^ set ].

anNsc class isNsc
    ifFalse: [  ^ self _errorPathObjectNotAnNsc: anNsc ].

set add: anNsc.

" no need to add index dictionary because it is RC
updateDict ~~ nil
  ifTrue: [ set add: updateDict ].
"

sz := children size.
1 to: anNsc size do: [ :i |
  setElement := anNsc _at: i.

  updateBtree ~~ nil
    ifTrue: [
      updateBtree btreeRoot _putNodesToLockForKey: setElement value: setElement into: set
    ].

  1 to: sz do: [ :j |
    (children at: j) _putIndexObjectsToLockFor: setElement into: set
  ]
].
^ set
%

category: 'ObsoleteIDX - Indexing Support (Locking)'
method: SetValuedPathTerm
_putIndexObjectsToLockFor: anNsc into: set path: pathArray

"Place into a set any internal indexing objects to lock to ensure that
modifications to the anObject can be committed."

| setElement sz |

(pathArray at: offset) = name
  ifFalse: [ ^ set ].

( nil == anNsc or: [ self size == 0 ] )
    ifTrue: [ ^ set ].

anNsc class isNsc
    ifFalse: [  ^ self _errorPathObjectNotAnNsc: anNsc ].

set add: anNsc.

" no need to add index dictionary because it is RC
updateDict ~~ nil
  ifTrue: [ set add: updateDict ].
"

sz := children size.
1 to: anNsc size do: [ :i |
  setElement := anNsc _at: i.

  updateBtree ~~ nil
    ifTrue: [
      updateBtree btreeRoot _putNodesToLockForKey: setElement value: setElement into: set
    ].

  1 to: sz do: [ :j |
    (children at: j) _putIndexObjectsToLockFor: setElement into: set path: pathArray
  ]
].
^ set
%

category: 'ObsoleteIDX - Indexing Support (Locking)'
method: BtreeInteriorNode
_putNodesToLockForKey: aKey value: aValue into: array

"An entry for the given key is located in the receiver.  Place the receiver 
 in the Array and recurse through the appropriate children."

| index eSize maxIndex childNode |
array add: self.

" find first child node that contains aKey "
index := self _binarySearchCoveringKey: aKey value: aValue.
(super at: index) _putNodesToLockForKey: aKey value: aValue into: array.

eSize := self entrySize.
maxIndex := self _lastIndex.
index := index + eSize.

" now scan child nodes until none contain the key/value "
[ index < maxIndex ] whileTrue: [
  childNode := super at: index.
  " see if first entry of child is for the key/value "
  (childNode _compareKey: aKey value: aValue equalToEntryAt: 2)
    ifTrue: [ childNode _putNodesToLockForKey: aKey value: aValue into: array ]
    ifFalse: [ ^ self ].
 index := index + eSize
]
%

category: 'ObsoleteIDX - Indexing Support (Locking)'
method: BtreeLeafNode
_putNodesToLockForKey: aKey value: aValue into: array

"An entry for the given key/value is located in the receiver.
 Place the receiver in the Array."

array add: self
%

